/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation, Intel Corporation.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TraceProfileFiltersUI.java,v 1.30 2010/08/29 23:35:16 jwest Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.core;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.Random;

import org.eclipse.hyades.trace.internal.ui.PDPluginImages;
import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.util.AbstractChangeable;
import org.eclipse.hyades.trace.ui.internal.util.FilterElementModifier;
import org.eclipse.hyades.trace.ui.internal.util.FilterList;
import org.eclipse.hyades.trace.ui.internal.util.FilterListContentProvider;
import org.eclipse.hyades.trace.ui.internal.util.FilterListLabelProvider;
import org.eclipse.hyades.trace.ui.internal.util.FilterSetElement;
import org.eclipse.hyades.trace.ui.internal.util.FilterTableElement;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.hyades.ui.internal.util.XMLUtil;
import org.eclipse.hyades.ui.util.GridUtil;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ColumnLayoutData;
import org.eclipse.jface.viewers.ColumnPixelData;
import org.eclipse.jface.viewers.ComboBoxCellEditor;
import org.eclipse.jface.viewers.IBasicPropertyConstants;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.PlatformUI;
import org.w3c.dom.Element;

import com.ibm.icu.text.Collator;

/**
 * 
 *
 */
public class TraceProfileFiltersUI extends AbstractChangeable implements SelectionListener, ICheckStateListener {
	protected Composite _control;

	protected FilterList _filterList;
	protected FilterSetList _filterSetList;

	private Button _upBtn;
	private Button _downBtn;

	private Button _addBtn;
	private Button _editBtn;
	private Button _removeBtn;

	private Table _table;
	protected CheckboxTableViewer _tableViewer;
	private static String[] tableColumnProperties =
		{
			FilterTableElement.P_TEXT,
			FilterTableElement.P_METHOD,
			FilterTableElement.P_VISIBILITY };

	private Label _filterSetLabel;
	private Table _filterSetTable;
	private CheckboxTableViewer _filterSetTableViewer;
	private Button _addSetBtn;
	private Button _editSetBtn;
	private Button _removeSetBtn;

	private Label _contentLabel;
	private Label _noteLabel;
	private FilterSetElement _oldSelection;
	
	private FilterSetElement activeFS;

	private boolean agentTerminated = false;

	public class FilterSetLabelProvider
		extends LabelProvider
		implements ITableLabelProvider {

		public FilterSetLabelProvider() {
			super();
		}
		public Image getColumnImage(Object element, int columnIndex) {
			return PDPluginImages.getImage(PDPluginImages.IMG_FILTER);
		}
		public Image getColumnImage(
			Viewer viewer,
			Object element,
			int columnIndex) {
			return PDPluginImages.getImage(PDPluginImages.IMG_FILTER);
		}
		public String getColumnText(Object element, int columnIndex) {
			return ((FilterSetElement)element).getDisplayName();
		}
		public String getColumnText(Viewer viewer, Object element, int columnIndex) {
			return ((FilterSetElement)element).getDisplayName();
		}
	}

	class FilterSetContentProvider implements IStructuredContentProvider {
		private CheckboxTableViewer fViewer;
		protected FilterSetElement fChangedElement;

		/**
		 * The constructor.
		 */
		public FilterSetContentProvider(CheckboxTableViewer tableViewer) {
			fViewer = tableViewer;
		}
		/**
		 * The visual part that is using this content provider is about
		 * to be disposed. Deallocate all allocated SWT resources.
		 */
		public void dispose() {
		}
		/**
		 * 
		 */
		public void elementChanged(
			FilterSetElement changedElement,
			String changedProperty) {
			fChangedElement = changedElement;
			fViewer.getControl().getDisplay().asyncExec(new Runnable() {
				public void run() {

					fViewer.refresh(fChangedElement);
				}
			});

		}
		/**
		 * Returns all the markers that should be shown for
		 * the current settings.
		 */

		public Object[] getElements(Object parent) {
			return ((FilterSetList) parent).elements();
		}
		public void inputChanged(
			Viewer viewer,
			Object oldInput,
			Object newInput) {
			fViewer = (CheckboxTableViewer) viewer;

			// Disconnect from old input.
			if (oldInput != null) {
				if (oldInput instanceof FilterList) {
					((FilterSetList) oldInput).removeElementChangedListener(
						this);

				}
			}

			// Connect to new input.
			if (newInput != null) {
				if (newInput instanceof FilterList)
					((FilterSetList) newInput).addElementChangedListener(this);
			}

		}
		public boolean isDeleted(Object element) {
			return false;
		}
	}

	public class FilterSetList implements IBasicPropertyConstants {
		ArrayList filters = new ArrayList();
		TableViewer fViewer;

		private ArrayList listeners;

		public FilterSetList(TableViewer viewer) {
			fViewer = viewer;
		}
		public void add(FilterSetElement element) {
			add(element, -1);
		}

		public void add(FilterSetElement element, int index) {
			if (index < 0)
				filters.add(element);
			else
				filters.add(index, element);
		}
		/**
		 * 
		 */
		public void addElementChangedListener(FilterSetContentProvider newListener) {

			if (newListener == null)
				return;
			if (listeners == null)
				listeners = new ArrayList();
			if (!listeners.contains(newListener))
				listeners.add(newListener);

		}
		/**
		 * Insert the method's description here.
		 * Creation date: (08/23/2000 11:34:27 AM)
		 * @return java.util.Enumeration
		 */
		public Object[] elements() {
			Collections.sort(filters, new Comparator() {
				public int compare(Object a, Object b) {
					String str1 = ((FilterSetElement)a).getDisplayName();
					String str2 = ((FilterSetElement)b).getDisplayName();
					
					if (str1 == null)
						str1 = "";//$NON-NLS-1$
					if (str2 == null)
						str2 = "";//$NON-NLS-1$
					
					return Collator.getInstance().compare(str1, str2);				
				}
			});				
			
			return filters.toArray();
		}
		/**
		 * 
		 */
		public void fireElementChanged(
			FilterSetElement changedElement,
			String changedProperty) {
			// forward change to content providers.
			if (listeners != null) {
				Iterator e = listeners.iterator();
				while (e.hasNext()) {
					FilterSetContentProvider l =
						(FilterSetContentProvider) e.next();
					l.elementChanged(changedElement, changedProperty);
				}
			}
		}
		public Object getElementAt(int index) {
			return filters.get(index);
		}
		/**
		 * Insert the method's description here.
		 * Creation date: (11/13/2000 12:53:52 PM)
		 * @return java.util.Vector
		 */
		public ArrayList getFilters() {
			return (ArrayList) filters.clone();
		}
		/**
		 * Insert the method's description here.
		 * Creation date: (09/18/2000 4:02:44 PM)
		 * @return int
		 */
		public int getSize() {
			return filters.size();
		}
		/**
		 * Insert the method's description here.
		 * Creation date: (08/22/2000 4:53:57 PM)
		 */
		public void removeAll() {
			filters.clear();
		}
		/**
		 * Insert the method's description here.
		 * Creation date: (08/22/2000 3:48:34 PM)
		 * @param idx int
		 */
		public Object removeElementAt(int idx) {
			Object element = filters.get(idx);

			if (element == null)
				return null;

			filters.remove(element);

			return element;
		}
		/**
		 * Removes a listener for selection changes.
		 */
		public void removeElementChangedListener(FilterSetContentProvider listener) {

			if (listeners != null)
				listeners.remove(listener);
		}
		
		
		public void setFilters(ArrayList filters)
		{
			this.filters = filters;
		}
		
//		/**
//		 * Insert the method's description here.
//		 * Creation date: (08/23/2000 11:34:27 AM)
//		 * @return java.util.Enumeration
//		 */
//		public Object[] uiElements() {
//			return filters.toArray();
//		}
	}

	class EditDialog extends Dialog implements ModifyListener {
		private String _class = "mypackage.*";
		private String _method;
		private String _rule;
		private Text _classText;
		private Text _methodText;
		private Combo _ruleText;
		private String _title;

		public EditDialog(
			Shell shell,
			String title,
			String clas,
			String method,
			String rule) {
			super(shell);

			if (clas != null)
				_class = clas;
			_method = method;
			_rule = rule;
			_title = title;
			
			setShellStyle(getShellStyle() | SWT.RESIZE);
		}

		protected void configureShell(Shell shell) {
			super.configureShell(shell);
			shell.setText(_title);
		}

		public void clas(String clas) {
			_class = clas;
		}

		public String clas() {
			return _class;
		}

		public void method(String method) {
			_method = method;
		}

		public String method() {
			return _method;
		}

		public void rule(String rule) {
			_rule = rule;
		}

		public String rule() {
			return _rule;
		}

		protected Control createDialogArea(Composite parent) {
			Composite result = (Composite) super.createDialogArea(parent);

			GridLayout layout;
			GridData data;
			Label label;
			Label patternLabel;
			
			layout = new GridLayout();
			layout.numColumns = 2;
			result.setLayout(layout);
			data = GridUtil.createFill();
			data.widthHint = 400;
			result.setLayoutData(data);

			patternLabel = new Label(result, SWT.WRAP);
			patternLabel.setText(UIPlugin.getResourceString(TraceMessages.FLT_SUPPPTRN));
			data = GridDataUtil.createHorizontalFill();
			data.horizontalSpan = 2;
			data.widthHint = 60;
			data.heightHint = 30;
			patternLabel.setLayoutData(data);
			
			label = new Label(result, SWT.NONE);
			label.setText(UIPlugin.getResourceString(TraceMessages.HEADER_FILTER));
			_classText = new Text(result, SWT.BORDER);
			_classText.setLayoutData(GridUtil.createHorizontalFill());

			label = new Label(result, SWT.NONE);
			label.setText(UIPlugin.getResourceString(TraceMessages.HEADER_MET));
			_methodText = new Text(result, SWT.BORDER);
			_methodText.setLayoutData(GridUtil.createHorizontalFill());

			label = new Label(result, SWT.NONE);
			label.setText(UIPlugin.getResourceString(TraceMessages.HEADER_VIS));
			_ruleText = new Combo(result, SWT.BORDER | SWT.READ_ONLY);
			_ruleText.setLayoutData(GridUtil.createHorizontalFill());

			_ruleText.add(TraceMessages.EXCLUDE);
			_ruleText.add(TraceMessages.INCLUDE);

			_classText.addModifyListener(this);
			_methodText.addModifyListener(this);
			_ruleText.select(1);

			if (_class != null)
				_classText.setText(_class);
			if (_method != null)
				_methodText.setText(_method);
			if (_rule != null)
				_ruleText.setText(_rule);

			_classText.setFocus();
			_classText.selectAll();
			return result;
		}

		public void modifyText(ModifyEvent e) {
			if (e.widget == _classText || e.widget == _methodText) {
				if (getButton(IDialogConstants.OK_ID) != null)
					getButton(IDialogConstants.OK_ID).setEnabled(
						_classText.getText().trim() != ""
							|| _methodText.getText().trim() != "");
			}
		}

		protected void okPressed() {

			_class = _classText.getText().trim();
			_method = _methodText.getText().trim();
			_rule = _ruleText.getText();
			super.okPressed();
		}

	}

	class EditWSetDialog extends Dialog implements ModifyListener {
		private String _name;
		private Text _nameText;
		private String _title;

		public EditWSetDialog(Shell shell, String title, String name) {
			super(shell);

			_name = name;
			_title = title;
		}

		protected void configureShell(Shell shell) {
			super.configureShell(shell);
			shell.setText(_title);
		}

		protected Control createDialogArea(Composite parent) {
			Composite result = (Composite) super.createDialogArea(parent);

			GridLayout layout;
			GridData data;
			Label label;

			layout = new GridLayout();
			layout.numColumns = 2;
			result.setLayout(layout);
			data = GridUtil.createFill();
			data.widthHint = 400;
			result.setLayoutData(data);

			label = new Label(result, SWT.NONE);
			label.setText(UIPlugin.getResourceString(TraceMessages.FLTST_NM));
			_nameText = new Text(result, SWT.BORDER);
			_nameText.setLayoutData(GridUtil.createHorizontalFill());
			_nameText.addModifyListener(this);

			if (_name != null)
				_nameText.setText(_name);
/*
			org.eclipse.ui.help.WorkbenchHelp.setHelp(
				_nameText,
				UIPlugin.getPluginId() + ".pfpp0007");
*/
			return result;
		}

		public void modifyText(ModifyEvent e) {
			if (e.widget == _nameText) {
				if (getButton(IDialogConstants.OK_ID) != null)
					getButton(IDialogConstants.OK_ID).setEnabled(
						_nameText.getText().trim().length() > 0);
			}
		}

		protected void okPressed() {
			_name = _nameText.getText().trim();
			super.okPressed();
		}
		
		/*
		 * Set the initial Ok button enablement.
		 */
		protected Control createContents(Composite parent) {
			Control result = super.createContents(parent);
			getButton(IDialogConstants.OK_ID).setEnabled(_nameText.getText().trim().length() > 0);
			return result;
		}
	}

	public TraceProfileFiltersUI() {
		super();
	}

	private void createColumns() {
		TableLayout layout = new TableLayout();
		_table.setLayout(layout);
		_table.setHeaderVisible(true);

		String headers[] =
			{
				UIPlugin.getResourceString(TraceMessages.HEADER_FILTER),
				UIPlugin.getResourceString(TraceMessages.HEADER_MET),
				UIPlugin.getResourceString(TraceMessages.HEADER_VIS)};
		ColumnLayoutData layouts[] =
			{
				new ColumnPixelData(120, true),
				new ColumnPixelData(110, true),
				new ColumnPixelData(70, true)};

		final TableColumn columns[] = new TableColumn[headers.length];

		for (int i = 0; i < headers.length; i++) {
			layout.addColumnData(layouts[i]);
			TableColumn tc = new TableColumn(_table, SWT.NONE, i);
			tc.setResizable(layouts[i].resizable);
			tc.setText(headers[i]);
			columns[i] = tc;
		}
	}
	public Composite createControl(Composite parent) {
		_control = createParentControl(parent);

		createFilterSet(_control);
		createFilterContents(_control);

		createNoteLabel(_control);

		return _control;
	}

	protected Composite createParentControl(Composite parent) {
		Composite result = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		layout.verticalSpacing = 0;
		result.setLayout(layout);
		GridData data = GridUtil.createFill();
		result.setLayoutData(data);
		return result;
	}

	protected void createNoteLabel(Composite parent) {
		GridData data;
		_noteLabel = new Label(parent, SWT.NONE);
		_noteLabel.setText(UIPlugin.getResourceString(TraceMessages.FLT_NOTE));

		data = new GridData();
		data.horizontalSpan = 2;
		_noteLabel.setLayoutData(data);
	}

	protected void createFilterContents(Composite parent) {
		GridData data;
		GridLayout layout;

		createVerticalIndent(parent);

		_contentLabel = new Label(parent, SWT.NONE);
		_contentLabel.setText(
			UIPlugin.getResourceString(TraceMessages.FLT_CNT));
		data = new GridData();
		data.horizontalSpan = 2;
		_contentLabel.setLayoutData(data);

		Composite tableGroup = new Composite(parent, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 1;
		tableGroup.setLayout(layout);
		data = GridUtil.createFill();
		data.heightHint = 180;
		tableGroup.setLayoutData(data);
		initializeTable(tableGroup);

		Composite btnGroup = new Composite(parent, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 1;
		btnGroup.setLayout(layout);
		btnGroup.setLayoutData(GridUtil.createVerticalFill());

		Composite addGroup = new Composite(btnGroup, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 1;
		layout.makeColumnsEqualWidth = true;
		addGroup.setLayout(layout);
		addGroup.setLayoutData(GridUtil.createVerticalFill());

		_addBtn = new Button(addGroup, SWT.PUSH);
		_addBtn.setText(UIPlugin.getResourceString(TraceMessages.ADD_FT1));
		_addBtn.setToolTipText(
			UIPlugin.getResourceString(TraceMessages.ADD_FTT));
		_addBtn.setLayoutData(GridUtil.createHorizontalFill());

		_editBtn = new Button(addGroup, SWT.PUSH);
		_editBtn.setText(UIPlugin.getResourceString(TraceMessages.EDTFLT_BTN1));
		_editBtn.setToolTipText(
			UIPlugin.getResourceString(TraceMessages.EDTFLT_TT));
		_editBtn.setLayoutData(GridUtil.createHorizontalFill());

		_removeBtn = new Button(addGroup, SWT.PUSH);
		_removeBtn.setText(UIPlugin.getResourceString(TraceMessages.RMV_TXT1));
		_removeBtn.setToolTipText(
			UIPlugin.getResourceString(TraceMessages.RMV_TIP));
		_removeBtn.setLayoutData(GridUtil.createHorizontalFill());

		Composite moveGroup = new Composite(btnGroup, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 1;
		layout.makeColumnsEqualWidth = true;
		moveGroup.setLayout(layout);
		moveGroup.setLayoutData(GridUtil.createHorizontalFill());

		_upBtn = new Button(moveGroup, SWT.PUSH);
		_upBtn.setText(UIPlugin.getResourceString(TraceMessages.MV_UP));
		_upBtn.setToolTipText(
			UIPlugin.getResourceString(TraceMessages.MV_UPT));
		_upBtn.setLayoutData(GridUtil.createHorizontalFill());

		_downBtn = new Button(moveGroup, SWT.PUSH);
		_downBtn.setText(UIPlugin.getResourceString(TraceMessages.MV_DN));
		_downBtn.setToolTipText(
			UIPlugin.getResourceString(TraceMessages.MV_DNT));
		_downBtn.setLayoutData(GridUtil.createHorizontalFill());

		_editBtn.setEnabled(false);
		_removeBtn.setEnabled(false);
		_upBtn.setEnabled(false);
		_downBtn.setEnabled(false);

		//add listeners
		_removeBtn.addSelectionListener(this);
		_upBtn.addSelectionListener(this);
		_downBtn.addSelectionListener(this);
		_addBtn.addSelectionListener(this);
		_editBtn.addSelectionListener(this);
		_table.addSelectionListener(this);

		PlatformUI.getWorkbench().getHelpSystem().setHelp(
			_table,
			UIPlugin.getPluginId() + ".pfpp0002");
	}

	protected void createFilterSet(Composite parent) {
		GridData data;
		GridLayout layout;

		_filterSetLabel = new Label(parent, SWT.NONE);
		_filterSetLabel.setText(UIPlugin.getResourceString(TraceMessages.SEL_FLTST));
		data = new GridData();
		data.horizontalSpan = 2;
		_filterSetLabel.setLayoutData(data);

		Composite tableGroup = new Composite(parent, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 1;
		tableGroup.setLayout(layout);
		data = GridUtil.createFill();
		data.heightHint = 100;

		tableGroup.setLayoutData(data);
		_filterSetTable = new Table(tableGroup, SWT.V_SCROLL | SWT.BORDER | SWT.SINGLE | SWT.FULL_SELECTION | SWT.CHECK);
		_filterSetTable.setLinesVisible(false);
		_filterSetTable.setLayoutData(GridUtil.createFill());

		_filterSetTableViewer = new CheckboxTableViewer(_filterSetTable);
		_filterSetTableViewer.setContentProvider(
			new FilterSetContentProvider(_filterSetTableViewer));
		_filterSetTableViewer.setLabelProvider(new FilterSetLabelProvider());

		_filterSetList = new FilterSetList(_filterSetTableViewer);
		_filterSetTableViewer.setInput(_filterSetList);
		_filterSetTableViewer.addCheckStateListener(this);

		TableLayout layoutt = new TableLayout();
		_filterSetTable.setLayout(layoutt);

		ColumnLayoutData l = new ColumnPixelData(420, true);
		layoutt.addColumnData(l);

		TableColumn tc = new TableColumn(_filterSetTable, SWT.NONE, 0);
		tc.setResizable(l.resizable);

		Composite btnGroup = new Composite(parent, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 1;
		btnGroup.setLayout(layout);
		btnGroup.setLayoutData(GridUtil.createVerticalFill());

		Composite addGroup = new Composite(btnGroup, SWT.NONE);
		layout = new GridLayout();
		layout.numColumns = 1;
		layout.makeColumnsEqualWidth = true;
		addGroup.setLayout(layout);
		addGroup.setLayoutData(GridUtil.createVerticalFill());

		_addSetBtn = new Button(addGroup, SWT.PUSH);
		_addSetBtn.setText(UIPlugin.getResourceString(TraceMessages.ADD_FT));
		_addSetBtn.setToolTipText(
			UIPlugin.getResourceString(TraceMessages.ADD_FSTT));
		_addSetBtn.setLayoutData(GridUtil.createHorizontalFill());

		_editSetBtn = new Button(addGroup, SWT.PUSH);
		_editSetBtn.setText(UIPlugin.getResourceString(TraceMessages.EDTST_BTN));
		_editSetBtn.setToolTipText(
			UIPlugin.getResourceString(TraceMessages.ED_FSTT));
		_editSetBtn.setLayoutData(GridUtil.createHorizontalFill());

		_removeSetBtn = new Button(addGroup, SWT.PUSH);
		_removeSetBtn.setText(UIPlugin.getResourceString(TraceMessages.RMV_TXT2));
		_removeSetBtn.setToolTipText(
			UIPlugin.getResourceString(TraceMessages.RM_FSTT));
		_removeSetBtn.setLayoutData(GridUtil.createHorizontalFill());

		_editSetBtn.setEnabled(false);
		_removeSetBtn.setEnabled(false);

		//add listeners
		_removeSetBtn.addSelectionListener(this);
		_addSetBtn.addSelectionListener(this);
		_editSetBtn.addSelectionListener(this);
		_filterSetTable.addSelectionListener(this);

		PlatformUI.getWorkbench().getHelpSystem().setHelp(
			_filterSetTable,
			UIPlugin.getPluginId() + ".pfps0001");
	}

	Composite _tableContainer;
	
	/** Label containing text saying that label will be generated on first profile */
//	Label _dynamicFilterContentsLabel;
	
	private void createTable(Composite parent) {
		
		// The table container is a new composite that contains the table, but
		// can also contain additional items, such as the label below (bug 323330)
		_tableContainer = new Composite(parent, SWT.NONE);
		_tableContainer.setLayout(new GridLayout(1, false));
		_tableContainer.setLayoutData(GridUtil.createFill());
		
		// This label only appears when the filter is listed as a filter that is auto-populated 
//		_dynamicFilterContentsLabel = new Label(_tableContainer, SWT.NONE);
		
		// Message is something along the lines of, 'contents of the filter will be populated on next profile'.
//		_dynamicFilterContentsLabel.setText(TraceMessages.FILTER_AUTO_POP_WARNING);
		
		
	    final GridData gd1 = new GridData();
	    gd1.exclude = true;
//	    _dynamicFilterContentsLabel.setLayoutData(gd1);
//	    _dynamicFilterContentsLabel.setVisible(false);
	    	    
		_table =
			new Table(
				_tableContainer,
				SWT.H_SCROLL
					| SWT.V_SCROLL
					| SWT.BORDER
					| SWT.MULTI
					| SWT.FULL_SELECTION);
		_table.setLinesVisible(true);
		
		final GridData gd2 = GridUtil.createFill();
		gd2.exclude = false;
		_table.setLayoutData(gd2);

	}

	public void enable(boolean enable) {

		// If the filter set is autopopulated, then display the appropriate message (bug 323330)
		if(activeFS != null && activeFS.getId() != null) {
			if(activeFS.isFilterAutoPopulated()) {
				showAutoGenFilterMessage(true);
				return;
			}
		}
		
		_addSetBtn.setEnabled(enable);
		_removeSetBtn.setEnabled(enable);
		_editSetBtn.setEnabled(enable);

		_filterSetLabel.setEnabled(enable);		
		_filterSetTable.setEnabled(enable);
		_filterSetTableViewer.getTable().setEnabled(enable);

		_contentLabel.setEnabled(enable);
		_addBtn.setEnabled(enable);
		_removeBtn.setEnabled(enable);
		_editBtn.setEnabled(enable);
		_upBtn.setEnabled(enable);
		_downBtn.setEnabled(enable);
		_noteLabel.setEnabled(enable);
		
		_table.setEnabled(enable);

		if (!enable) {
			_tableViewer.setCellEditors(null);
			_tableViewer.setCellModifier(null);
		} else {
			setTableCellEditors();
			setTableCellModifiler();
		}

	}

	protected void enableButtons() {
		if (isAgentTerminated()) {
			enable(false);
			return;
		}

		int selection = _table.getSelectionIndex();
		int selectionCount = _table.getSelectionCount();

		_removeBtn.setEnabled(_filterList.getSize() > 0 && selection != -1);
		_editBtn.setEnabled(_filterList.getSize() > 0 && selection != -1);

		_upBtn.setEnabled(selection > 0 && selectionCount == 1);
		_downBtn.setEnabled(
			selection >= 0
				&& selection < _table.getItemCount() - 1
				&& selectionCount == 1);

		selection = _filterSetTable.getSelectionIndex();

		_removeSetBtn.setEnabled(
			_filterSetList.getSize() > 1 && selection != -1);
		_editSetBtn.setEnabled(_filterSetList.getSize() > 0 && selection != -1);

	}
	public ArrayList getFilterSet() {
		if (_oldSelection != null) {
			_oldSelection.setChildren(
				(ArrayList) _filterList.getFilters().clone());
		}

		validateFilterSet(_filterSetList);
		return _filterSetList.getFilters();
	}
	
	/**
	 * Example:
	 * <filter name = enabled = >
	 * 	<contents>
	 * 		<content text = method = visibility = />
	 * 		....
	 * </contents>
	 * ....
	 * </filter>
	 * @deprecated Use TraceFilterManager.getInstance().serializeFiltersToXML()
	 */
	private static String getFilterSet(ArrayList list) 
	{
		return TraceFilterManager.serializeFiltersToXML(list);
	}

	/**
	 * Validates the filter set; add an EXCLUDE * filter if the 
	 * filter set contains only INCLUDE filter 
	 * @param filterSet
	 */
	private void validateFilterSet(FilterSetList filterSet) {
		ArrayList filters = filterSet.getFilters();

		for (int idx = 0; idx < filters.size(); idx++) {
			FilterSetElement set = (FilterSetElement) filters.get(idx);
			ArrayList elements = set.getChildren();
			boolean addExcludeFilter = true;
			for (int i = 0; i < elements.size(); i++) {
				FilterTableElement elem = (FilterTableElement) elements.get(i);
				if (elem
					.getVisibility()
					.equals(TraceMessages.EXCLUDE)) {
					addExcludeFilter = false;
					break; //the filter set is valid
				} else if (elem.getText().equals("*")) {
					addExcludeFilter = false;
					break; //the filter set is valid
				}
			}

			if (elements.size() > 0 && addExcludeFilter) {
				//the set has only include filters, add exclude * to exclude any other data
				elements.add(
					new FilterTableElement(
						"*",
						"*",
						TraceMessages.EXCLUDE));
				set.setChildren(elements);
			}
		}

	}

	protected void initializeTable(Composite parent) {
		createTable(parent);
		_tableViewer = new CheckboxTableViewer(_table);
		createColumns();

		_tableViewer.setContentProvider(
			new FilterListContentProvider(_tableViewer));
		_tableViewer.setLabelProvider(new FilterListLabelProvider());

		_filterList = new FilterList(_tableViewer);
		_tableViewer.setInput(_filterList);

		setTableCellEditors();
		setTableCellModifiler();
		_tableViewer.setColumnProperties(tableColumnProperties);
		
	}
	
	private static ArrayList getFiltersSet(String filters){
		Element prefElement = XMLUtil.loadDom(filters, "filters");
		return getFiltersSet(prefElement);
	}
	
	
	/**
	 * 
	 * @deprecated Use TraceFilterManager.getInstance().getFilterSets() to get the filter sets
	 * stored in the in the preference.
	 */
	private static ArrayList getFiltersSet(Element filters)
	{
		return TraceFilterManager.parseFiltersSet(filters);
	}

	private void setXMLFilterSet(Element filters)
	{
		//expects the filters element
		//which contains the nodes
			//filter
				//contents
		ArrayList list = getFiltersSet(filters);
		for (int idx = 0; idx < list.size(); idx++)
			addFilterSet((FilterSetElement) list.get(idx));
		
		selectTableItem(_filterSetTable, 0);
	}
	
	public void setFilterSet(String filters)
	{
		Element prefElement = XMLUtil.loadDom(filters, "filters");
		setXMLFilterSet(prefElement);
	}
	
	public void setFilterSet(Element filters)
	{
		setXMLFilterSet(filters);
	}
	
	public void setFilterSet(ArrayList filters)
	{
		_filterSetList.setFilters(filters);
		_filterSetTableViewer.refresh();
	}

	public void addFilterSet(FilterSetElement elem) {
		_filterSetList.add(elem);
		_filterSetTableViewer.refresh();
		_filterSetTableViewer.setChecked(elem, false);
	}

	public void reset() {
		activeFS = null;
		_oldSelection = null;

		_table.removeAll();
		_filterList.removeAll();
		_filterSetList.removeAll();
		_filterSetTableViewer.refresh();
	}
	
	public void resetFields()
	{
		activeFS = null;
		_oldSelection = null;
	}

	protected void selectTableItem(Table table, int selection) {
		table.setFocus();
		table.setSelection(selection);
		table.showSelection();

		if (table == _filterSetTable) {
			showFiltersForSelection(
				(FilterSetElement) _filterSetList.getElementAt(selection));
		}
	}

	private void showFiltersForSelection(FilterSetElement elem) {
		if (_oldSelection != null) {
			_oldSelection.setChildren(
				(ArrayList) _filterList.getFilters().clone());
		}

		_table.setRedraw(false);

		_filterList.removeAll();
		ArrayList list = elem.getChildren();
		for (int idx = 0; idx < list.size(); idx++) {
			FilterTableElement filter = (FilterTableElement) list.get(idx);
			filter.setParent(_filterList);
			_filterList.add(filter, _filterList.getSize());
			_tableViewer.refresh();
		}

		_tableViewer.refresh();
		_oldSelection = elem;

		_table.setRedraw(true);
	}

	public void widgetDefaultSelected(SelectionEvent event) {
	}

	public void widgetSelected(SelectionEvent event) {
		if (event.widget == _addSetBtn) {
			showWSetDialog(UIPlugin.getResourceString(TraceMessages.ADD_SETDG), null);
			enableButtons();
			return;

		} else if (event.widget == _editSetBtn) {
			IStructuredSelection sel = (IStructuredSelection)_filterSetTableViewer.getSelection();
			if (!sel.isEmpty()) {
				FilterSetElement element = (FilterSetElement)sel.getFirstElement();
				showWSetDialog(UIPlugin.getResourceString(TraceMessages.EDT_SETDG), element);
				return;
			}
		} else if (event.widget == _removeSetBtn) {
			int selection = _filterSetTable.getSelectionIndex();
			if (selection != -1) {
				boolean select = false;
				if (_filterSetTable.getItem(selection).getChecked())
					select = true;

				_filterSetList.removeElementAt(selection);
				if (select) {
					FilterSetElement elem =
						(FilterSetElement) _filterSetList.getElementAt(0);
					activeFS = elem;
					_filterSetTableViewer.setChecked(elem, true);
				}
				_filterSetTableViewer.refresh();

				selectTableItem(_filterSetTable, 0);
			}

			enableButtons();
			return;
		} else if (event.widget == _addBtn) {
			showEditDialog(
				UIPlugin.getResourceString(TraceMessages.ADD_FLTDLG),
				null,
				"*",
				null);
			enableButtons();
		} else if (event.widget == _editBtn) {
			int idx = _table.getSelectionIndex();
			if (idx != -1) {
				FilterTableElement sel =
					(FilterTableElement) _filterList.getElementAt(idx);

				String filter = sel.getText();
				String method = sel.getMethod();
				String vis = sel.getVisibility();

				showEditDialog(
					UIPlugin.getResourceString(TraceMessages.EDT_FLTDLG),
					filter,
					method,
					vis);
			}
		} else if (event.widget == _table) {
			enableButtons();
			return;
		} else if (event.widget == _filterSetTable) {
			int idx = _filterSetTable.getSelectionIndex();
			
			if (idx != -1) {

				// Set corresponding checkbox.
				FilterSetElement elem1 = (FilterSetElement) _filterSetList.getElementAt(idx);
				showFiltersForSelection(elem1);
				_filterSetTableViewer.setChecked(elem1, true);
				activeFS = elem1;

				// Clear all other checkboxes.
				for (int kdx = 0; kdx < _filterSetList.getSize(); kdx++) {
					if (kdx != idx) {
						FilterSetElement elem2 =
							(FilterSetElement) _filterSetList.getElementAt(kdx);
						_filterSetTableViewer.setChecked(elem2, false);
					}
				}
				_filterSetTableViewer.refresh();

				// If the filter set is autopopulated, then display the appropriate message (bug 323330)
				if(elem1.getId() != null && elem1.isFilterAutoPopulated()) {
					showAutoGenFilterMessage(true);
				} else {
					showAutoGenFilterMessage(false);					
				}

			} else {
				enableButtons();
			}
			
			return;
		} else if (event.widget == _removeBtn) {
			int selection = _table.getSelectionIndex();

			_filterList.removeElements(_table.getSelectionIndices());

			_tableViewer.refresh();

			int size = _filterList.getSize();
			if (size > 0) {
				if (selection > 0 && selection < size) {
					selectTableItem(_table, selection - 1);
				} else {
					selectTableItem(_table, 0);
				}
			}
			enableButtons();
		} else if (event.widget == _upBtn) {
			_tableViewer.refresh();

			int selection = _table.getSelectionIndex();
			if (selection > 0) {
				FilterTableElement sel =
					(FilterTableElement) _filterList.getElementAt(selection);
				FilterTableElement previous =
					(FilterTableElement) _filterList.getElementAt(
						selection - 1);

				String filter = sel.getText();
				String method = sel.getMethod();
				String vis = sel.getVisibility();

				sel.setText(previous.getText());
				sel.setMethod(previous.getMethod());
				previous.setText(filter);
				previous.setMethod(method);

				sel.setVisibility(previous.getVisibility());
				previous.setVisibility(vis);

				selectTableItem(_table, selection - 1);

			}

			_tableViewer.refresh();
			enableButtons();
		} else if (event.widget == _downBtn) {
			_tableViewer.refresh();

			int selection = _table.getSelectionIndex();
			if (selection < _table.getItemCount() - 1) {
				FilterTableElement sel =
					(FilterTableElement) _filterList.getElementAt(selection);
				FilterTableElement next =
					(FilterTableElement) _filterList.getElementAt(
						selection + 1);

				String filter = sel.getText();
				String method = sel.getMethod();
				String vis = sel.getVisibility();

				sel.setText(next.getText());
				sel.setMethod(next.getMethod());
				next.setText(filter);
				next.setMethod(method);
				sel.setVisibility(next.getVisibility());
				next.setVisibility(vis);

				selectTableItem(_table, selection + 1);

			}

			_tableViewer.refresh();
			enableButtons();
		}
	}

	private void showEditDialog(
		String title,
		String clas,
		String method,
		String rule) {

		EditDialog dialog =
			new EditDialog(_control.getShell(), title, clas, method, rule);
		dialog.open();

		if (dialog.getReturnCode() == Window.OK) {

			if (clas == null) {

				//add filter
				FilterTableElement elem =
					new FilterTableElement(
						dialog.clas(),
						dialog.method(),
						dialog.rule());

				if (dialog
					.rule()
					.equals(TraceMessages.EXCLUDE))
					_filterList.add(elem, _filterList.getSize());
				else
					_filterList.add(elem, 0);

				elem.setParent(_filterList);

				_tableViewer.refresh();

				if (dialog
					.rule()
					.equals(UIPlugin.getResourceString(TraceMessages.EXCLUDE)))
					selectTableItem(_table, _filterList.getSize() - 1);
				else
					selectTableItem(_table, 0);
			} else {
				int idx = _table.getSelectionIndex();
				if (idx != -1) {

					FilterTableElement sel =
						(FilterTableElement) _filterList.getElementAt(idx);

					sel.setText(dialog.clas());
					sel.setMethod(dialog.method());
					sel.setVisibility(dialog.rule());

					_tableViewer.refresh();

				}

			}
		}

	}

	private void showWSetDialog(String title, FilterSetElement element) {

		EditWSetDialog dialog =
			new EditWSetDialog(_control.getShell(), title, (element == null) ? null : element.getDisplayName());
		dialog.open();

		if (dialog.getReturnCode() == Window.OK) {

			if (element == null) {
				
				/*
				 * If there's already a filter set with the given name, highlight
				 * it (but don't modify it).
				 */
				for (int idx = 0; idx < _filterSetList.getSize(); idx++) {
					FilterSetElement el = (FilterSetElement) _filterSetList.getElementAt(idx);
					if (el.getDisplayName().equals(dialog._name)) {
						selectTableItem(_filterSetTable, idx);
						return;
					}
				}
				
				FilterSetElement elem =	new FilterSetElement(UIPlugin.PLUGIN_ID + "." + dialog._name + System.currentTimeMillis());
				elem.setKey(null);
				elem.setName(dialog._name);
				String filters = TraceConstants.DEFAULT_FILTER_SET;
				Element prefElement = XMLUtil.loadDom(filters, "filters");
				ArrayList list = getFiltersSet(prefElement);
				if (list.size() > 0) {
					FilterSetElement set = (FilterSetElement) list.get(0);
					elem.setChildren((ArrayList) set.getChildren().clone());
				}
				addFilterSet(elem);
				setActiveFilterSet(elem);
			} else {
				int idx = _filterSetTable.getSelectionIndex();
				if (idx != -1) {
					((FilterSetElement)_filterSetList.getElementAt(idx)).setName(dialog._name);
					_filterSetTableViewer.refresh(((FilterSetElement)_filterSetList.getElementAt(idx)));
					setActiveFilterSet((FilterSetElement)_filterSetList.getElementAt(idx));
				}
			}
		}

	}

	/** If true, disable the filter contents table, disable any buttons that
	 * can be used to alter the filter contents table, and bring up the message
	 * indicating the the contents are auto-populated. 
	 * (If false, the reverse occurs, and the filter contents are returned to normal) (Bug 323330)
	 */
	protected void showAutoGenFilterMessage(boolean enabled) {
		if(enabled) {
//			((GridData)_dynamicFilterContentsLabel.getLayoutData()).exclude = false;
			((GridData)_table.getLayoutData()).exclude = false;
			
//			_dynamicFilterContentsLabel.setVisible(true);
			_table.setVisible(true);
			
			_editBtn.setEnabled(false);
			_addBtn.setEnabled(false);
			_removeBtn.setEnabled(false);
			
			_table.setEnabled(false);
			
			_tableContainer.layout(false);
			_tableContainer.redraw();
			
		} else {

//			((GridData)_dynamicFilterContentsLabel.getLayoutData()).exclude = true;
			((GridData)_table.getLayoutData()).exclude = false;
			
//			_dynamicFilterContentsLabel.setVisible(false);
			_table.setVisible(true);
			
			_table.setEnabled(true);
			_tableContainer.layout(false);
			_tableContainer.redraw();
			
			enableButtons();
			_addBtn.setEnabled(true); // This one is not handled by enableButtons();
			
		}
		
	}

	public void checkStateChanged(CheckStateChangedEvent event) {
		FilterSetElement elem = (FilterSetElement) event.getElement();
		//update any previous selected set  	 		
		for (int i = 0; i < _filterSetList.getSize(); i++) {
			FilterSetElement data = (FilterSetElement) _filterSetList.getElementAt(i);
			
			if (data == elem) {
			
				if(data != null && data.getId() != null) {
					if(data.isFilterAutoPopulated()) {
						showAutoGenFilterMessage(true);
					}  else {
						showAutoGenFilterMessage(false);					
					}
				} else {
					showAutoGenFilterMessage(false);					
				}
				
				// Set corresponding selection.
				if (_filterSetTableViewer.getChecked(data))
					_filterSetTable.setSelection(i);
				continue;
			}
		}

		_filterSetTableViewer.refresh();
	}

	/**
	 * Stores the values of the controls back to the preference store.
	 * 
	 * @deprecated Use TraceFilterManager.getInstance().storeFitlerSetList(items); 
	 */
	public static void storeValues(ArrayList items) 
	{
		TraceFilterManager.getInstance().storeFilterSetList(items);
	}

	/**
	 * Returns the agentTerminated.
	 * @return boolean
	 */
	public boolean isAgentTerminated() {
		return agentTerminated;
	}

	/**
	 * Sets the agentTerminated and disables all ui controls when
	 * <code>false</code> value is set
	 * 
	 * @param agentTerminated The agentTerminated to set
	 */
	public void setAgentTerminated(boolean agentTerminated) {
		this.agentTerminated = agentTerminated;
		if (agentTerminated) {
			enable(false);
		}
	}

	/**
	 * removes a filter set
	 */
	public void removeFilterSet(int idx) {
		if (idx < 0 || idx > _filterSetList.getSize())
			return;
		_filterSetList.removeElementAt(idx);
	}
	
	/*
	 * returns the active filter set
	 * used by ProfileFiltersTab
	 */
	public String getActiveFilterSet()
	{
		if (activeFS == null)
			return null;
		return activeFS.getId();
	}
	
	
	public FilterSetElement getActiveFilterSetElement()
	{
		return activeFS;
	}
	
	public void setActiveFilterSet(FilterSetElement set) {
		activeFS = set;
				
		_filterSetTableViewer.setCheckedElements(new Object[] { set });
		_filterSetTableViewer.setSelection(new StructuredSelection(set));
		showFiltersForSelection(set);
		_filterSetTableViewer.refresh();
		enableButtons();

		// If the filter set is autopopulated, then display the appropriate message (bug 323330)
		if(activeFS != null && activeFS.getId() != null) {
			if(activeFS.isFilterAutoPopulated()) {
				showAutoGenFilterMessage(true);
				return;
			}
		}
	}
	
	public void addFilterSets(FilterSetElement[] list)
	{
		for(int idx=0; idx<list.length; idx++)
			addFilterSet(list[idx]);
	}

	protected void createVerticalIndent(final Composite parent) {
		Label verticalIntent = new Label(parent, SWT.NONE);
		GridData data = new GridData();
		data.horizontalSpan = 2;
		verticalIntent.setLayoutData(data);
	}

	private void setTableCellModifiler() {
		_tableViewer.setCellModifier(new FilterElementModifier(_tableViewer));
	}

	private void setTableCellEditors() {
		CellEditor editors[] = new CellEditor[_table.getColumnCount()];
	
		editors[0] = new TextCellEditor(_table);
		editors[1] = new TextCellEditor(_table);
		editors[2] = new ComboBoxCellEditor(_table, FilterTableElement
				.getVisibilityStates());
	
		_tableViewer.setCellEditors(editors);
	}
}
