/**********************************************************************
 * Copyright (c) 2005, 2011 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TraceProfileUI.java,v 1.34 2011/01/21 15:55:33 mreid Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.core;

import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.hyades.security.internal.util.BaseConnectUtil;
import org.eclipse.hyades.security.internal.util.ConnectUtil;
import org.eclipse.hyades.security.internal.util.NullConnectUtilUI;
import org.eclipse.hyades.trace.internal.ui.PDPluginImages;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.launcher.HyadesTraceUIExtensionSupportUtil;
import org.eclipse.hyades.trace.ui.internal.launcher.HyadesTraceUIExtensionSupportUtil.TraceExtensionUIElementPos;
import org.eclipse.hyades.trace.ui.internal.launcher.ILaunchConfigurationTabUpdater;
import org.eclipse.hyades.trace.ui.internal.launcher.IProfileLaunchConfigurationConstants;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfileDestinationTab;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfileTab;
import org.eclipse.hyades.trace.ui.internal.util.AbstractChangeable;
import org.eclipse.hyades.trace.ui.internal.util.AgentControllerProbe;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSetType;
import org.eclipse.hyades.trace.ui.launcher.ProfilingSetsManagerCopy;
import org.eclipse.hyades.trace.ui.provisional.ITraceUIHelper;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTreeViewer;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.events.ShellAdapter;
import org.eclipse.swt.events.ShellEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.TabFolder;
import org.eclipse.swt.widgets.TabItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIConstants;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIPlugin;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceConstants;
import org.eclipse.tptp.trace.ui.internal.launcher.core.AnalysisType;
import org.eclipse.tptp.trace.ui.internal.launcher.core.AvailabilityTesterDelegate;
import org.eclipse.tptp.trace.ui.internal.launcher.core.DataCollector;
import org.eclipse.tptp.trace.ui.internal.launcher.core.DataCollectorAssociation;
import org.eclipse.tptp.trace.ui.internal.launcher.core.DataCollectorManager;
import org.eclipse.tptp.trace.ui.internal.launcher.core.DataCollectorTreeContentProvider;
import org.eclipse.tptp.trace.ui.internal.launcher.core.DataCollectorTreeContentProvider.DataCollectorTreeInput;
import org.eclipse.tptp.trace.ui.internal.launcher.core.DataCollectorTreeContentProvider.ParentChildNode;
import org.eclipse.tptp.trace.ui.internal.launcher.core.DataCollectorTreeLabelProvider;
import org.eclipse.tptp.trace.ui.internal.launcher.core.EditOptionDelegate;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherConstants;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;

/*
 * CONTEXT_ID top0001 for My application is too slow in profile options page
 * CONTEXT_ID top0002 for My application uses too much memory in profile options page
 * CONTEXT_ID top0003 for Collect execution flow information in profile options page
 * CONTEXT_ID top0004 for Show &instance level information in profile options page
 * CONTEXT_ID top0005 for Show execution flow details in profile options page
 * CONTEXT_ID top0006 for Show heap details in profile options page
 * CONTEXT_ID top0007 for Show execution flow graphical details in profile options page
 * CONTEXT_ID top0008 for Show execution statistic in profile options page
 * CONTEXT_ID top0009 for Boundary class depth: in profile options page
 * CONTEXT_ID top0010 for Automatically start monitoring in profile options page
 */


/**
 * Ali M. -- 93212: The following changes are made to the profile tab:
 * <ul>
 * 	<li> Nested tabs are taken out.  The destination tab will need to appear as another 
 * 		 tab of the launch configuration and the limits options will only appear if the
 * 		 user edits the JVMPI data collector
 * 	</li>
 * 	<li>
 * 		Data collectors and analysis types (formerly known as profiling types are introduced
 * 		in a tree layout.
 * 	</li>
 * </ul>
 */
public class TraceProfileUI extends AbstractChangeable implements 
ICheckStateListener, ISelectionChangedListener, KeyListener, IDoubleClickListener, ILaunchConfigurationTabUpdater
{	

	/* A constant indicating the type DataCollector */
	private static final int DATACOLLECTOR_TYPE = 0;
	
	/* A constant indicating the type AnalysisType */
	private static final int ANALYSIS_TYPE = 1;
	
	/* Represents the parent tab */
	private ProfileTab parentTab;

	/* This is the check box tree displaying the data collectors and their applicable analysis types */
	private CheckboxTreeViewer checkBoxTreeViewer;
	
	/* The launch configuration */
	private ILaunchConfiguration launchConfiguration;
		
	/* A working copy of the profiling set manager -- used for backward compatibility */
	private ProfilingSetsManagerCopy profilingSetsManagerWC;

	/* The test availability button */
	private Button testAvailability;

	/* The edit option button */
	private Button editOptions;
	
	/* Stores an error message */
	private String errorMessage;

	/* Indicates whether the destination tab should be included as a nested tab */
	private boolean includeDestinationTab;
	
	/* The nested destination tab */
	private ProfileDestinationTab destinationTab;
	
	/* Selected element of the check box tree */
	private Object selectedElement;

	/* The edit option button delegate */
	private SelectionListener editOptionDelegate;
	
	/* Indicates if the items of the data collector tree have been fetched yet */
	private boolean initialized;
	
	/* The last host selected */
	private String lastHost;
	/* The last port number detected */
	private int lastPort;
	
	/* Bug 323137 - AC validation job */
	private Job validateAcJob;
	private String lastVerifiedHost;
	private int lastVerifiedPort;
	
	/* A listener used to delegate tasks to the parent folder */
	private org.eclipse.swt.widgets.Listener listener = new org.eclipse.swt.widgets.Listener() {
		public void handleEvent(org.eclipse.swt.widgets.Event event) {
			parentTab.update();
		}
	};
	
	
	/**
	 * The constructor.
	 * 
	 * @param parentTab The parent tab
	 * @param includeDestinationTab true will force the destination tab to appear as a nested
	 * tab of the profile tab.  Clients are encouraged to include the destination tab as a 
	 * configuration tab of its own (i.e. having this flag always set to false)
	 */
	public TraceProfileUI(ProfileTab parentTab, boolean includeDestinationTab) 
	{
		super();
		this.parentTab = parentTab;
		this.includeDestinationTab = includeDestinationTab;

		if (includeDestinationTab)
			destinationTab = new ProfileDestinationTab(parentTab);
	}	
	
	public Composite createControl(Composite parent)
	{
		if (includeDestinationTab)
			return createNestedTabControl(parent);
		return createFlatControl(parent);
	} 
	
	private Composite createNestedTabControl (Composite parent)
	{
		parent.setLayout(new FillLayout());
		TabFolder folder = new TabFolder(parent, SWT.TOP);
		
		TabItem item1 = new TabItem(folder, SWT.TOP);
		item1.setText(UIPlugin.getResourceString(TraceMessages.TB_NVW));
		item1.setControl(createFlatControl(folder));
		
		TabItem item2 = new TabItem(folder, SWT.TOP);
		item2.setText(UIPlugin.getResourceString(TraceMessages.TB_NDEST));
		item2.setImage(PDPluginImages.getImage(PDPluginImages.IMG_UI_FOLDER));
		item2.setControl(destinationTab.createControl(folder));
		destinationTab.addListener(listener);

		return folder;
	}
	
	private Composite createFlatControl (Composite parent)
	{
		Composite content = includeDestinationTab ? new Composite(parent, SWT.NULL) : parent;
		content.setLayout(new GridLayout());
		content.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));			

		/* Provide the profiling dialog UI objects to any interested extension points (Bug 323330) */
		HyadesTraceUIExtensionSupportUtil.callExtensionUIElements(content, launchConfiguration, parentTab, TraceExtensionUIElementPos.TOP);

		/* Create the instruction label */
		Label launchLabel = new Label(content, SWT.NONE);
		launchLabel.setText(TraceMessages.LAUNCH_INSTRUCTION);
		launchLabel.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));

		/* Create the tree composite */
		Composite treeComposite = new Composite(content, SWT.NONE);
		treeComposite.setLayout(new GridLayout(2, false));
		treeComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
				
		/* Create the tree and control buttons */
		createDataCollectorTree(treeComposite);

//		/* Provide the profiling dialog UI objects to any interested extension points (Bug 323330) */
		HyadesTraceUIExtensionSupportUtil.callExtensionUIElements(content, launchConfiguration, parentTab, TraceExtensionUIElementPos.BOTTOM);
		
		return content;
	}
	

	private void createDataCollectorTree(Composite treeComposite)
	{
		/* Create the tree */
		DataCollectorTreeContentProvider provider = new DataCollectorTreeContentProvider(this);
		checkBoxTreeViewer = new CheckboxTreeViewer(treeComposite);
		checkBoxTreeViewer.setContentProvider(provider);
		checkBoxTreeViewer.setSorter(null);		
				
		checkBoxTreeViewer.setLabelProvider(new DataCollectorTreeLabelProvider());
		checkBoxTreeViewer.setInput(null);
		checkBoxTreeViewer.getTree().setLayout(new GridLayout());
		checkBoxTreeViewer.getTree().setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		checkBoxTreeViewer.addCheckStateListener(this);
		checkBoxTreeViewer.addSelectionChangedListener(this);		
		checkBoxTreeViewer.getTree().addKeyListener(this);
		checkBoxTreeViewer.addDoubleClickListener(this);
		
		/* Create the control buttons */
		Composite controlButtonComposite = new Composite(treeComposite, SWT.NONE);
		controlButtonComposite.setLayout(new GridLayout());
		controlButtonComposite.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, false, false));
	
		editOptions = new Button (controlButtonComposite, SWT.NONE);
		editOptions.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, false, false));
		editOptions.setText(TraceMessages.LAUNCH_CONTROL_EDIT);
		editOptionDelegate = EditOptionDelegate.getInstance(this, getProfilingSetWC());
		editOptions.addSelectionListener(editOptionDelegate);
		editOptions.setEnabled(false);
		
		testAvailability = new Button (controlButtonComposite, SWT.NONE);
		testAvailability.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, false, false));
		testAvailability.setText(TraceMessages.LAUNCH_CONTROL_TEST);
		testAvailability.addSelectionListener(AvailabilityTesterDelegate.getInstance());
		testAvailability.setEnabled (false);
	
	}
	
	
	public void checkStateChanged(CheckStateChangedEvent event)
	{
		Object checkedElement = event.getElement();
		if (checkedElement instanceof DataCollector) 
		{
			DataCollector selectedDatacollector = null;
			
			/* Check for the co-existance violators of the selected data collector and de-select
			 * each violator */
			if (event.getChecked())
			{
				selectedDatacollector = (DataCollector)checkedElement;
				deselectTreeItems(checkedElement, selectedDatacollector.getCoexistanceViolators(), 0);
				deselectIndirectViolators(checkedElement, 0);
			}
			
			checkBoxTreeViewer.setSelection(new StructuredSelection(checkedElement));
			checkBoxTreeViewer.setGrayed(checkedElement, false);
			checkBoxTreeViewer.setSubtreeChecked(checkedElement, event.getChecked());
			
			/* We'll need to cause a check state event for the first analysis 
			 * type that is detected as the child of the data collector. */
			if (selectedDatacollector != null)
			{
				Object[] childrenOfSelectedItem = ((DataCollectorTreeContentProvider)checkBoxTreeViewer.getContentProvider()).getChildren(selectedDatacollector);
				if (childrenOfSelectedItem != null)
				{
					for (int i = 0; i < childrenOfSelectedItem.length; i++)
					{
						if (childrenOfSelectedItem[i] instanceof ParentChildNode && ((ParentChildNode)childrenOfSelectedItem[i]).child instanceof AnalysisType)
						{
							ProfileLaunchUtil.changeItemCheckedState(checkBoxTreeViewer, this, childrenOfSelectedItem[i], true);
							break;
						}
					}
				}
			}
		}
		else if (checkedElement instanceof ParentChildNode) 
		{
			/* If the element is an analysis type, then check for its co-existance violators 
			 * and deselect them */
			Object childNode;
			if (event.getChecked() && (childNode = ((ParentChildNode)checkedElement).child) instanceof AnalysisType)
			{
				deselectTreeItems(checkedElement, ((AnalysisType)childNode).getCoexistanceViolators(), 1);
				deselectIndirectViolators(checkedElement, 1);
			}
			
			checkBoxTreeViewer.setSelection(new StructuredSelection(checkedElement));
			Object parent = ((ITreeContentProvider)checkBoxTreeViewer.getContentProvider()).getParent(event.getElement());
			if (ProfileLaunchUtil.changeParentGrayState(checkBoxTreeViewer, event))
			{				
				deselectTreeItems(parent, ((DataCollector)parent).getCoexistanceViolators(), 0);
				deselectIndirectViolators(parent, 0);
			}			
		}		
				
		update();
	}
	

	/**
	 * Deselects indirect co-existance violators.  An indirect violator is an item
	 * that defines a violation on a newly selected item, but the newly selected item
	 * hasn't declared the item as a violator.
	 * <p>
	 * For example: <br/>
	 * <ul>
	 * 	<li> A declares B as a violator </li>
	 * 	<li> B doesn't declare A as a violator </li>
	 * 	<li> User selects A </li>
	 *  <li> User selects B </li>
	 *  <li> A is an indicrect violator of B and should be deselected </li>
	 * </ul>
	 * 
	 * @param newlyCheckedElement The newly checked element
	 * @param itemType The item type (either DATACOLLECTOR_TYPE or ANALYSIS_TYPE)
	 */
	private void deselectIndirectViolators(Object newlyCheckedElement, int itemType)
	{
		Object[] checkedElements = checkBoxTreeViewer.getCheckedElements();
		Object childNode;
		Map violators;
		String currentItemId;
		for (int i = 0; i < checkedElements.length; i++)
		{
			violators = null;
			currentItemId = null;
			if (itemType == DATACOLLECTOR_TYPE && !checkedElements[i].equals(newlyCheckedElement) && checkedElements[i] instanceof DataCollector)
			{				
				violators = ((DataCollector)checkedElements[i]).getCoexistanceViolators();
				currentItemId = ((DataCollector)newlyCheckedElement).getId();
				//deselectTreeItems(newlyCheckedElement, ((DataCollector)checkedElements[i]).getCoexistanceViolators(), itemType);
			}
			else if (itemType == ANALYSIS_TYPE && !checkedElements[i].equals(newlyCheckedElement) && checkedElements[i] instanceof ParentChildNode && (childNode = ((ParentChildNode)checkedElements[i]).child) instanceof AnalysisType)
			{
				AnalysisType analysisType = (AnalysisType)childNode;
				violators = analysisType.getCoexistanceViolators();
				currentItemId = ((AnalysisType)((ParentChildNode)newlyCheckedElement).child).getId();
				//deselectTreeItems(newlyCheckedElement, ((AnalysisType)childNode).getCoexistanceViolators(), itemType);
			}
			
			if (violators != null && (violators.get(LauncherConstants.ALL) != null || violators.get(currentItemId) != null))
				ProfileLaunchUtil.changeItemCheckedState(checkBoxTreeViewer, this, checkedElements[i], false);
				
		}
	}

	/**
	 * Deselects tree items with ids stored in coexistanceViolators.  If
	 * itemType == DATACOLLECTOR_TYPE, then data collectors will be deselected and if 
	 * itemType == ANALYSIS_TYPE analysis types will be deselected.
	 * 
	 * @param checkedElement The checked element
	 * @param violators The violators
	 * @param itemType The type of the items to be deselected
	 */
	private void deselectTreeItems(Object checkedElement, Map violators, int itemType)
	{
		if (violators == null || violators.size() <= 0)
			return;
				
		Object[] selectedItems = checkBoxTreeViewer.getCheckedElements();;
		boolean deselectAll = violators.get(LauncherConstants.ALL) != null;
		
		Object childNode;
		/* Iterate through each selected element and deselect the violators */
		for (int i = 0; i < selectedItems.length; i++)
		{
			/* Deselect all other entries */
			childNode = null;
			boolean isDataCollectorViolationPresent = itemType == DATACOLLECTOR_TYPE && !selectedItems[i].equals(checkedElement) && selectedItems[i] instanceof DataCollector;
			boolean isAnalysisTypeViolationPresent = itemType == ANALYSIS_TYPE && !selectedItems[i].equals(checkedElement) && selectedItems[i] instanceof ParentChildNode && (childNode = ((ParentChildNode)selectedItems[i]).child) instanceof AnalysisType;
			
			/* Deselect specific entries */
			if (!deselectAll)
			{
				isDataCollectorViolationPresent = isDataCollectorViolationPresent && (violators.get(((DataCollector)checkedElement).getId()) != null || violators.get(((DataCollector)selectedItems[i]).getId()) != null);
				isAnalysisTypeViolationPresent = isAnalysisTypeViolationPresent && (violators.get(((AnalysisType)((ParentChildNode)checkedElement).child).getId()) != null || violators.get(((AnalysisType)childNode).getId()) != null);
			}
			
			if (isDataCollectorViolationPresent || isAnalysisTypeViolationPresent)
				ProfileLaunchUtil.changeItemCheckedState(checkBoxTreeViewer, this, selectedItems[i], false);
		}
	}


	/**
	 * Invoked as a result of a selection change in the tree displaying
	 * the data collectors and their associated analysis types.
	 * 
	 * @param event The selection event.
	 */
	public void selectionChanged(SelectionChangedEvent event)
	{
		selectedElement = ProfileLaunchUtil.updateButtonStatus(launchConfiguration, event, editOptions, testAvailability, checkBoxTreeViewer, this);		
	}


	/** Extract the ID of the analysis type from the tree */
	public String getSelectedAnalysisType() {
		try {
			TreeSelection s = (TreeSelection)checkBoxTreeViewer.getSelection();
			ParentChildNode p = (ParentChildNode)s.getFirstElement();
			AnalysisType t = (AnalysisType)p.child;
			return t.getId();
		} catch(Exception e) {
			// Any errors here should return null, where they will be handled gracefully by calling method
			return null;
		}
		
	}
	
	/** In the checkbox viewer, select the given analysis type of the given data collector */
	public void setAnalysisType(String dataCollectorId, String analysisTypeId) {
		
		TreeItem[] dataCollectors = checkBoxTreeViewer.getTree().getItems();
		if (dataCollectors != null)
		{
			for (int i = 0; i < dataCollectors.length; i++)
			{
				if (!(dataCollectors[i].getData() instanceof DataCollector))
					continue;
				
				DataCollector dataCollector = (DataCollector)dataCollectors[i].getData();
				
				if(dataCollector.getId().equalsIgnoreCase(dataCollectorId)) {

					for(AnalysisType at : dataCollector.getApplicableAnalysisTypes()) {
						if(at.getId().equalsIgnoreCase(analysisTypeId)) {
					
							ProfileLaunchUtil.changeItemCheckedState(checkBoxTreeViewer, this, new ParentChildNode(dataCollector, at), true);
						}
					}

					
				}

			}
		}
	}
	
	public void initializeAfterFetch()
	{
		if (checkBoxTreeViewer.getTree().isDisposed())
			return;
		
		getProfilingSetWC().initializeFrom(launchConfiguration);
		
		/* Try to restore the previous selection for the configuration */
		boolean isConfigurationNew = false;
		
		checkBoxTreeViewer.setAllChecked(false);
		Hashtable entitySelection = LauncherUtility.unserializeSelection(launchConfiguration);
		if (entitySelection == null)
			isConfigurationNew = true;
		
		else
		{
			/* Walk through each data collector and select its children */
			Iterator selectedDataCollectors = entitySelection.keySet().iterator();
			DataCollector currentDataCollector;
			while(selectedDataCollectors.hasNext())
			{
				currentDataCollector = (DataCollector)selectedDataCollectors.next();
				List selectedChildren = (List)entitySelection.get(currentDataCollector);
				int chilrenCount = (selectedChildren == null ? 0 : selectedChildren.size());
				
				/* If there are no selected children, then select the data collector itself */
				if (chilrenCount <= 0)
					ProfileLaunchUtil.changeItemCheckedState(checkBoxTreeViewer, this, currentDataCollector, true);
				
				/* Otherwise select every child of the data collector */
				else
					for (int i = 0; i < chilrenCount; ProfileLaunchUtil.changeItemCheckedState(checkBoxTreeViewer, this, new ParentChildNode(currentDataCollector, selectedChildren.get(i++)), true));										
			}
		}
		
		
		/* If the configuration is new then make default selections if the configuration doesn't have an associated
		 * profiling set.  If the configuration happens to have a profiling set, then we'll need to support backward compatibility
		 * and select the profiling types under the profiling set. */
		initialized = true;	
		if (isConfigurationNew)
		{
			try
			{										
				/* Read in the default data collectors */
				DataCollectorAssociation dataCollectorAssociator = DataCollectorManager.getInstance().getDataCollectorAssociator(launchConfiguration.getType().getIdentifier());
				String[] defaultDataCollectors = dataCollectorAssociator.getDefaultDataCollectors();
				if (defaultDataCollectors != null)
				{
					for (int i = 0; i < defaultDataCollectors.length; i++)
						ProfileLaunchUtil.changeItemCheckedState(checkBoxTreeViewer, this, DataCollectorManager.getInstance().getDataCollector(defaultDataCollectors[i]), true);				
				}
				
				/* Read in the default analysis types of the data collectors associated with the launch */
				TreeItem[] dataCollectors = checkBoxTreeViewer.getTree().getItems();
				boolean isPICollectorPresent = false;
				if (dataCollectors != null)
				{
					for (int i = 0; i < dataCollectors.length; i++)
					{
						if (!(dataCollectors[i].getData() instanceof DataCollector))
							continue;
						
						DataCollector dataCollector = (DataCollector)dataCollectors[i].getData();
						if (LauncherConstants.JVMPI_DATA_COLLECTOR_ID.equals(dataCollector.getId()))
							isPICollectorPresent = true;
						
						boolean isChecked = checkBoxTreeViewer.getChecked(dataCollector);
						boolean isGrayed = checkBoxTreeViewer.getGrayed(dataCollector);
						
						/* If the data collector is fully checked, then all of its children (analysis types) are
						 * already selected. */
						if (isChecked && !isGrayed)
							continue;
						
						// Otherwise, use the set default for this data collector
						AnalysisType[] defaultAnalysisTypes = dataCollector.getDefaultAnalysisTypes();
						if (defaultAnalysisTypes != null)
						{
							for (int j = 0; j < defaultAnalysisTypes.length; j++)
							{
								/* Select the analysis type and fire off a change check state*/
								ProfileLaunchUtil.changeItemCheckedState(checkBoxTreeViewer, this, new ParentChildNode(dataCollector, defaultAnalysisTypes[j]), true);													
							}
						}
						
					}
				}	
		
				/* For backward compatability */
				if (isPICollectorPresent)
				{
					String profilingSetID = launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_SET, (String)null);
					if (profilingSetID != null)
					{
						ParentChildNode[] selectedItems = LauncherUtility.getSelectedDataCollectors(profilingSetID);
						for (int i = 0; i < selectedItems.length; ProfileLaunchUtil.changeItemCheckedState(checkBoxTreeViewer, this, selectedItems[i++], true)); 					
					}
				}
				

				
				if (launchConfiguration instanceof ILaunchConfigurationWorkingCopy)
					performApply((ILaunchConfigurationWorkingCopy)launchConfiguration);
				
			}
			catch (Exception e)
			{
				UIPlugin.getDefault().log(e);			
			}
		}		
	}
	
	
	public void initializeFrom(ILaunchConfiguration conf)
	{
		if (includeDestinationTab)
			destinationTab.initializeFrom(conf);
	
		this.launchConfiguration = conf;
		
		lazyUpdateDataCollectors(conf);
		
	}

	/**
	 * Initialize the data collector tree view. Take care to only do this if the host in the launch config differs
	 * from the last time this was initialized. This corrects the accessibility bug 313437.
	 * 
	 * To ensure correct caching behaviour, (e.g. see bug 291702) any calls to checkBoxTreeViewer.setInput() should 
	 * come through here unless setting it to null.
	 * 
	 * @param conf The launch config from which we are (possibly) initializing the data collector
	 */
	private void lazyUpdateDataCollectors(ILaunchConfiguration conf) {
		
		try {
			String currentHost = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_HOSTNAME, CommonUITraceConstants.LOCAL_HOST);
			int currentPort = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PORT, CommonUIPlugin.getDefault().getPreferenceStore().getDefaultInt(CommonUIConstants.LOCALHOST_PORT));
			if( checkBoxTreeViewer.getInput() == null || lastHost == null || !lastHost.equals(currentHost) || lastPort != currentPort )
			{
				initialized = false;
				checkBoxTreeViewer.setInput(new DataCollectorTreeInput(conf));

				// Record the host and port so we don't have to do this again if they haven't changed.
				lastHost = currentHost;
				lastPort = currentPort; 
			}
			else
				// Bug 333908 - If we don't fetch a fresh list of collectors, we still
				// need to initialize the existing list with the current launch config 
				initializeAfterFetch();
		} catch (CoreException e) {
			UIPlugin.getDefault().log(e);
		}
	}
	
	/**
	 * Bug 323137
	 * 
	 * Called when the profiling tab is made active; starts a job to validate the AC to ensure it supports NEF. 
	 */
	public void lazyVerifyAC() {
		
		try {
			final String currentHost = this.launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_HOSTNAME, CommonUITraceConstants.LOCAL_HOST);
			final int currentPort = this.launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PORT, CommonUIPlugin.getDefault().getPreferenceStore().getDefaultInt(CommonUIConstants.LOCALHOST_PORT));
			
			// If we've already verified the host/port combo we don't need to do it again
			if( null == validateAcJob && (lastVerifiedHost == null || !lastVerifiedHost .equals(currentHost) || lastVerifiedPort != currentPort) ) {
				
				lastVerifiedHost = currentHost;
				lastVerifiedPort = currentPort;
				
				validateAcJob = new Job(TraceMessages.TraceProfileUI_verifyACJobName) {
					
					@Override
					public IStatus run(IProgressMonitor progress) {
						
						// Put 15 seconds as upper bound; if it hasn't finished by then there is likely a problem..
						// so go ahead and don't allow the problem to block us indefinitely.
						final int retryCount = 15;
						int attempt = 0;
						
						while( attempt < retryCount && DataCollectorManager.getInstance().isFetching() ) {
							DataCollectorManager.getInstance().waitForDataCollectors(1000);
							attempt++;
						}
						
						// Do not run this against remote hosts
						if( !ConnectUtil.isLocalHost(currentHost) ) {
							validateAcJob = null;
							return Status.OK_STATUS;
						}
						
						// Run the probe against the AC
						ITraceUIHelper traceUIHelper = ITraceUIHelper.INSTANCE;
						IPreferenceStore store = traceUIHelper.getTraceUIPreferenceStore();
						ConnectUtil util = new ConnectUtil(currentHost, "" + currentPort, store.getString(CommonUIConstants.SECURE_USERID), traceUIHelper.getTraceUIApplication()); //$NON-NLS-1$
						AgentControllerProbe acProbe = new AgentControllerProbe(util.getUserId(), traceUIHelper.getTraceUIApplication());
						final AgentControllerProbe.Result res = acProbe.runProbe(currentHost, currentPort);
						final IStatus status = (res.legacyOnly ? new Status(IStatus.WARNING, UIPlugin.PLUGIN_ID, "\n" + TraceMessages.TraceProfileUI_obsoleteDetail) : Status.OK_STATUS); //$NON-NLS-1$
												
						Display.getDefault().syncExec(new Runnable() {						
							public void run() {
								validateAcJob = null;
								if( !parentTab.getControl().isDisposed() ) {
									if( !status.isOK() ) {
										ErrorDialog.openError(parentTab.getControl().getShell(), 
												TraceMessages.CRD_CFWT, // Re-use existing 'Warning' string
												NLS.bind(TraceMessages.TraceProfileUI_obsoleteAC, res.host, res.port), 
												status );
									}
									update();
								}
							}
						});
						
						return status;
					}
					
				};
				validateAcJob.schedule();
			}
		} catch (CoreException e) {
			e.printStackTrace();
		}
		
	}
	
	public void activated(ILaunchConfigurationWorkingCopy conf) 
	{
		if (includeDestinationTab)
			destinationTab.activated(conf);
		
		// If the host/port selection has changed, then update the tree
		try
		{
			String currentHost = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_HOSTNAME, CommonUITraceConstants.LOCAL_HOST);
			int currentPort = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PORT, CommonUIPlugin.getDefault().getPreferenceStore().getDefaultInt(CommonUIConstants.LOCALHOST_PORT));
			
			Label label = null;
			ConnectUtil connect = new ConnectUtil( currentHost, ""+currentPort, ITraceUIHelper.INSTANCE.getTraceUIApplication()); //$NON-NLS-1$
			
			// We don't want to prompt for username/password, we just want to check network connectivity
			connect.setConnectionUI( new NullConnectUtilUI() );
			
			Control[] controls= checkBoxTreeViewer.getTree().getParent().getParent().getChildren();
			
			// Locate the first label in the controls list
			for(int x = 0; x < controls.length; x++) {
				if(controls[x] instanceof Label) {
					label = (Label) controls[x];
					break;
				}
			}
			
			// We're just testing connectivity; if authentication fails we don't care at this point
			int result = connect.connect(false);
			if( BaseConnectUtil.CONNECTION_HOST_ERROR == result ||
				BaseConnectUtil.CONNECTION_PORT_ERROR == result ||
				BaseConnectUtil.CONNECTION_CONTROLLER_ERROR == result ) {
				
				label.setForeground(new Color(null,255, 0, 0));
				label.setText(TraceMessages.CONH_ERROR_);
				
				checkBoxTreeViewer.setInput(null);
				errorMessage =  TraceMessages.CONH_ERROR_;
				
				return;
			}else{
				label.setForeground(new Color(null,0, 0, 0));
				label.setText(TraceMessages.LAUNCH_INSTRUCTION);
			}
						
			// Bug 313437: Lazily update the data collector tree if needed
			lazyUpdateDataCollectors(conf);
			
			// Bug 323137: Lazily verify the AC if needed
			lazyVerifyAC();
		} 
		catch (CoreException e)
		{
			UIPlugin.getDefault().log(e);
		}
	}

	
	public void deactivated(ILaunchConfigurationWorkingCopy wc)
	{
		initialized = false;
	}
	
	
	/**
	 * Invoked when the user has applied the changes.  
	 * 
	 * @param wc An editable copy of the launch configuration.
	 */
	public void performApply(ILaunchConfigurationWorkingCopy wc) 
	{	
		if (!initialized)
			return;
		
		if (includeDestinationTab)
			destinationTab.performApply(wc);
		
		/* In order to support backward compatibility and to make the transition easier, we'll
		 * need to build a dummy profiling set and add it to a working copy of a profiling set manager */
		org.eclipse.hyades.trace.ui.internal.launcher.ProfileLaunchUtil.createDummyProfilingSet(getProfilingSetWC(), wc, checkBoxTreeViewer.getCheckedElements());
		
		String treeSelectionStr = LauncherUtility.serializeSelection(checkBoxTreeViewer);
		wc.setAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_COLLECTOR_AND_ANALYSIS, treeSelectionStr);
	}


	public boolean isValid(ILaunchConfiguration conf) 
	{		
		if (includeDestinationTab && destinationTab.isValid(conf))
			return false;

		// Bug 323137 - Wait until the AC validation job has completed
		if( null != validateAcJob && validateAcJob.getState() != Job.NONE ) {
			errorMessage = TraceMessages.TraceProfileUI_verifyACStatus;
			return false;
		}
		
		IStatus status = ProfileLaunchUtil.checkValidity(conf, checkBoxTreeViewer, getProfilingSetWC());
		errorMessage = status.isOK() ? null : status.getMessage();
		return status.isOK();		
	}
	

	public ProfilingSetsManagerCopy getProfilingSetWC()
	{
		if (profilingSetsManagerWC == null)
			profilingSetsManagerWC = new ProfilingSetsManagerCopy();
		
		return profilingSetsManagerWC;
	}

	public void update()
	{
		parentTab.update();		
	}

	public String getErrorMessage()
	{		
		return errorMessage;
	}


	/**
	 * Used to show a quick description of the selected item in the
	 * monitor tab.
	 */
	public void keyPressed(KeyEvent e)
	{
		final int F2_KEY_CODE = 16777227;
		if (e.keyCode != F2_KEY_CODE || selectedElement == null)
			return;
		
		String descriptionContent = null;
		Object child = null;
		
		if (selectedElement instanceof DataCollector)
			descriptionContent = ((DataCollector)selectedElement).getDescription();
		else if (selectedElement instanceof ParentChildNode)
		{
			child = ((ParentChildNode)selectedElement).child;
			if (child instanceof AnalysisType)
				descriptionContent = ((AnalysisType)child).getDescription();
			
			/* For backward compatibility */
			else if (child instanceof IProfilingSetType)
				descriptionContent = ((IProfilingSetType)child).getDescription();
		}
		
		
		/* Create the quick show description shell if there is a selection 
		 * and a description to show */
		if (descriptionContent == null || descriptionContent.length() <= 0)
			return;
		
		Display display = UIPlugin.getDefault().getWorkbench().getDisplay();
		final Shell quickDescShell = new Shell(display.getActiveShell(), SWT.MODELESS | SWT.RESIZE);
		Color color = new Color(display, 251, 253, 213);
		quickDescShell.setLayout(new GridLayout());
		quickDescShell.setLayoutData(new GridData());
		quickDescShell.setLocation(display.getCursorLocation());		
		quickDescShell.setSize(200, 100);
		quickDescShell.setBackground(color);
		
		final Text descTxt = new Text (quickDescShell, SWT.WRAP);
		descTxt.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		descTxt.setBackground(color);
		descTxt.setText(descriptionContent);
		descTxt.setEditable(false);
		
		/* Destroy the shell once it has been deactivated */
		quickDescShell.addShellListener(new ShellAdapter(){

			public void shellDeactivated(ShellEvent e)
			{
				if (quickDescShell != null && !quickDescShell.isDisposed())
				{
					quickDescShell.close();
					quickDescShell.dispose();				
				}
				
			}			
		});
		
		quickDescShell.open();
	}


	public void keyReleased(KeyEvent e)
	{
		/* Doesn't need to be implemented */
	}


	public void doubleClick(DoubleClickEvent event)
	{		
		ProfileLaunchUtil.handleDoubleClick(editOptions, editOptionDelegate);
	}

	public CheckboxTreeViewer getCheckBoxTreeViewer() {
		return checkBoxTreeViewer;
	}

	public ILaunchConfiguration getLaunchConfiguration() {
		return launchConfiguration;
	}
}
