/**********************************************************************
 * Copyright (c) 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.ui.internal.launcher;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.IJobChangeEvent;
import org.eclipse.core.runtime.jobs.ISchedulingRule;
import org.eclipse.core.runtime.jobs.JobChangeAdapter;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.NotConnectedException;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.trace.internal.ui.PDPluginImages;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.core.TraceUIImages;
import org.eclipse.hyades.trace.ui.internal.util.PDCoreUtil;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSetType;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.AbstractTreeViewer;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.graphics.Image;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceConstants;
import org.eclipse.tptp.platform.execution.client.agent.IAgent;
import org.eclipse.tptp.platform.execution.client.core.IAgentController;
import org.eclipse.tptp.platform.execution.client.core.NodeFactory;
import org.eclipse.tptp.platform.execution.util.internal.AgentControllerPool;
import org.eclipse.tptp.trace.ui.internal.launcher.core.AgentDeclaration;
import org.eclipse.tptp.trace.ui.internal.launcher.core.AgentDeclarationManager;
import org.eclipse.tptp.trace.ui.internal.launcher.core.AnalysisType;
import org.eclipse.tptp.trace.ui.internal.launcher.core.DataCollector;
import org.eclipse.tptp.trace.ui.internal.launcher.core.DataCollectorAssociation;
import org.eclipse.tptp.trace.ui.internal.launcher.core.DataCollectorAssociationData;
import org.eclipse.tptp.trace.ui.internal.launcher.core.DataCollectorManager;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherConstants;
import org.eclipse.tptp.trace.ui.provisional.launcher.ICollectorFiltration;
import org.eclipse.ui.progress.DeferredTreeContentManager;
import org.eclipse.ui.progress.IDeferredWorkbenchAdapter;
import org.eclipse.ui.progress.IElementCollector;
import org.eclipse.ui.progress.PendingUpdateAdapter;
import org.eclipse.ui.progress.WorkbenchJob;

/**
 * Provides the content and label providers for the checkbox tree
 * that appears under the agents tab
 * 
 * @author Ali Mehregani
 */
public class AttachAgentTabProviders
{
	public static class AttachAgentContentProvider implements ITreeContentProvider
	{
		private DeferredTreeContentManager deferredContentManager;
		private AttachAgentsTab agentsTab;
		private TreeViewer treeViewer;
		private AgentTreeInput input;
		
		public AttachAgentContentProvider(TreeViewer treeViewer, AttachAgentsTab agentsTab)
		{
			this.treeViewer = treeViewer;
			this.agentsTab = agentsTab;
		}
		
		public Object[] getChildren(Object parentElement)
		{
			/* We need to determine the analysis types of the agent item */
			AgentTreeItem agentTreeItem = null;
			DataCollector dataCollector = null;			
			if (parentElement instanceof AgentTreeItem && (dataCollector = (agentTreeItem = (AgentTreeItem)parentElement).getDataCollector()) != null)
			{
				AnalysisType[] analysisTypes = dataCollector.getApplicableAnalysisTypes();
				List items = new ArrayList(analysisTypes.length);
				for (int i = 0; i < analysisTypes.length; i++)
				{
					/* Does analysis type have a filter? */
					ICollectorFiltration filter = analysisTypes[i].getFilter();
					if (filter != null &&
						!filter.include(analysisTypes[i].getId(), input == null ? null : input.getConfiguration(), agentTreeItem.getAgent() == null ? (Object)agentTreeItem.getNewAgent() : agentTreeItem.getAgent()))
					{						
						continue;
					}
					
					items.add(new AgentTreeItem(agentTreeItem, analysisTypes[i]));
				}
				
				/* To support backward compatibility, we'll need to associate orphan profiling types to
				 * the JVMPI data collector. */				
				if (LauncherConstants.JVMPI_DATA_COLLECTOR_ID.equals(dataCollector.getId()))
				{
					Object[] profilingTypes = ProfilingSetsManager.instance().getProfilingTypes().values().toArray();
					
					/* We'll need to filter the analysis types */
					for (int i = 0; i < profilingTypes.length; i++)
					{
						if (!(profilingTypes[i] instanceof AnalysisType) && profilingTypes[i] instanceof IProfilingSetType)
							items.add(new AgentTreeItem(agentTreeItem, (IProfilingSetType)profilingTypes[i]));
					}
				}
				
				return (AgentTreeItem[])items.toArray(new AgentTreeItem[items.size()]);
			}
						
			return new Object[0];
		}

		public Object getParent(Object element)
		{
			if (element instanceof AgentTreeItem)
			{
				return ((AgentTreeItem)element).getParent();
			}
			
			return null;
		}

		public boolean hasChildren(Object element)
		{
			if (element instanceof AgentTreeItem)
			{
				return ((AgentTreeItem)element).hasChildren();
			}
			return false;
		}

		public Object[] getElements(Object inputElement)
		{
			return deferredContentManager.getChildren(inputElement);
		}

		public void dispose()
		{			
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput)
		{
			this.input = newInput instanceof AgentTreeInput ? (AgentTreeInput)newInput : null;	
			if (deferredContentManager != null)
				return;			
			deferredContentManager = new DeferredTreeContentManager(this, (AbstractTreeViewer)viewer)
			{
				protected void addChildren(final Object parent, final Object[] children, IProgressMonitor monitor)
				{
			        WorkbenchJob updateJob = new WorkbenchJob(TraceMessages.AttachAdding) 
			        {
			            /** 
			             * @see org.eclipse.ui.progress.UIJob#runInUIThread(org.eclipse.core.runtime.IProgressMonitor)
			             */
			            public IStatus runInUIThread(IProgressMonitor updateMonitor) {
			                //Cancel the job if the tree viewer got closed
			                if (treeViewer.getControl().isDisposed() || updateMonitor.isCanceled()) {
								return Status.CANCEL_STATUS;
							}
			                treeViewer.add(parent, children);
			                return Status.OK_STATUS;
			            }
			        };
			        
			        updateJob.setSystem(true);
			        updateJob.addJobChangeListener(new JobChangeAdapter() {
						public void done(IJobChangeEvent event) {
							agentsTab.initializeAfterFetch();
						}						
			        });
			        updateJob.schedule();
				}
			};
		}		
	}
	
	
	public static class AttachAgentLabelProvider extends LabelProvider
	{		
		public String getText(Object element)
		{
			if (element instanceof AgentTreeItem)
			{
				AgentTreeItem item = (AgentTreeItem)element;	
				String processId = " [PID: " + item.getProcessId() + "]"; //$NON-NLS-1$ //$NON-NLS-2$
				if (item.getDataCollector() != null)
				{
					return item.getDataCollector().getName() + processId;
				}
				else if (item.getAnalysisType() != null)
				{
					return item.getAnalysisType().getName();
				}
				else if (item.getProfilingType() != null)
				{
					return item.getProfilingType().getName();
				}
				else
				{
					Agent agent = item.getAgent();
					IAgent newAgent = item.getNewAgent();
					if (agent != null && agent.getName() != null)
					{
						return agent.getName() + processId;
					}
					else if (newAgent != null && newAgent.getName() != null)
					{
						return newAgent.getName() + processId;
					}
				}
			}
			else if (element instanceof PendingUpdateAdapter)
			{
				return TraceMessages.AttachPendingElement;
			} 
			else if (element instanceof MessageTreeItem)
			{
				MessageTreeItem messageTreeItem = (MessageTreeItem)element;
				return messageTreeItem.getMessage();
			}
			
			return super.getText(element);
		}
		
		
		public Image getImage(Object element)
		{
			if (element instanceof AgentTreeItem)
			{
				AgentTreeItem item = (AgentTreeItem)element;				
				if (item.getDataCollector() != null)
				{
					return item.getDataCollector().getIcon().createImage();
				}
				else if (item.getAnalysisType() != null)
				{
					return item.getAnalysisType().getIcon().createImage();
				}
				else if (item.getProfilingType() != null)
				{
					return item.getProfilingType().getImage().createImage();
				}
				else
				{
					return TraceUIImages.INSTANCE.getImage("c", TraceUIImages.IMG_DETACH_AGENT); //$NON-NLS-1$
				}
			}
			else if (element instanceof MessageTreeItem)
			{
				MessageTreeItem messageTreeItem = (MessageTreeItem)element;
				if (messageTreeItem.getSeverity() == IStatus.ERROR)
				{
					return PDPluginImages.getImage(PDPluginImages.IMG_ERROR_ITEM);
				}
			}
			return super.getImage(element);
		}
	}
	
	
	/**
	 * This object will act as the input of the agent tree.
	 * It will need to adapt to the IDeferredWorkbenchAdapter.
	 */
	public static class AgentTreeInput implements IAdaptable
	{
		private DeferredAgentAdapter deferredAgentAdapter;
		private ILaunchConfiguration configuration;
		
		public AgentTreeInput (ILaunchConfiguration configuration, AttachAgentsTab agentsTab)
		{
			deferredAgentAdapter = new DeferredAgentAdapter(configuration, agentsTab);
			this.configuration = configuration;
		}
		public Object getAdapter(Class adapter)
		{
			if (adapter == IDeferredWorkbenchAdapter.class)
				return deferredAgentAdapter;
			
			return null;
		}
		/**
		 * @return the configuration
		 */
		public ILaunchConfiguration getConfiguration()
		{
			return configuration;
		}
		/**
		 * @param configuration the configuration to set
		 */
		public void setConfiguration(ILaunchConfiguration configuration)
		{
			this.configuration = configuration;
		}		
		
		
	}
	
	
	/**
	 * Represents the items that appears in the check box tree
	 * of the agents tab
	 */
	public static class AgentTreeItem
	{
		private AgentTreeItem parent;
		private DataCollector dataCollector;
		private Agent agent;
		private IAgent newAgent;
		private AnalysisType analysisType;
		private IProfilingSetType profilingType;
		
		public AgentTreeItem (DataCollector dataCollector, Agent agent)
		{
			this.dataCollector = dataCollector;
			this.agent = agent;			
		}

		public AgentTreeItem (DataCollector dataCollector, IAgent newAgent)
		{
			this.dataCollector = dataCollector;
			this.newAgent = newAgent;			
		}
		
		public AgentTreeItem(AgentTreeItem parent, AnalysisType analysisType)
		{
			this.parent = parent;
			this.analysisType = analysisType;
		}
		
		public AgentTreeItem(AgentTreeItem parent, IProfilingSetType profilingType)
		{
			this.parent = parent;
			this.profilingType = profilingType;
		}

		public boolean hasChildren()
		{
			if (dataCollector != null)
				return true;
			return false;
		}

		public AgentTreeItem getParent()
		{			
			return parent;
		}
		
		public Agent getAgent()
		{
			return agent;
		}
		
		public DataCollector getDataCollector()
		{
			return dataCollector;
		}
		
		public AnalysisType getAnalysisType()
		{
			return analysisType;
		}
		
		
		/**
		 * @return the newAgent
		 */
		public IAgent getNewAgent()
		{
			return newAgent;
		}

		public boolean equals(Object o)
		{
			if (!(o instanceof AgentTreeItem))
				return false;
			
			AgentTreeItem item = (AgentTreeItem)o;
			String itemProcessId = null;
			String processId = null;
			
			itemProcessId = item.getProcessId();
			processId = getProcessId();
			
			/* To avoid an NPE */
			if (processId == null)
				processId = ""; //$NON-NLS-1$
			
			return 	processId.equals(itemProcessId) && 								/* Processes must be the same */
					equalCheck(parent, item.getParent()) &&							/* Check to make sure the parents are the same */
					equalCheck(agent, item.getAgent()) &&							/* The agents are equal */
					equalCheck(newAgent, item.getNewAgent()) &&						/* The agents are equal */
					equalCheck(item);												/* The entities are equal */
		}
		
		private boolean equalCheck(AgentTreeItem parent1, AgentTreeItem parent2)
		{
			if (parent1 == null)
				return parent2 == null;
			
			return parent1.equals(parent2);
		}

		private boolean equalCheck(AgentTreeItem item)
		{
			if (dataCollector != null)
			{
				DataCollector itemDataCollector = item.getDataCollector();
				return itemDataCollector != null && dataCollector.getId().equals(itemDataCollector.getId());
			}
			else if (analysisType != null)
			{
				AnalysisType itemAnalysisType = item.getAnalysisType();
				return itemAnalysisType != null && analysisType.getId().equals(itemAnalysisType.getId());
			}
			else if (profilingType != null)
			{
				IProfilingSetType itemProfilingType = item.getProfilingType();
				return itemProfilingType != null && profilingType.getId().equals(itemProfilingType.getId());
			}
			
			return item.getDataCollector() == null && item.getAnalysisType() == null && item.getProfilingType() == null;
		}

		private boolean equalCheck(IAgent newAgent1, IAgent newAgent2)
		{			
			if (newAgent1 == null)
				return newAgent2 == null;
			else if (newAgent2 == null)
				return false;
			
			return 	newAgent.getName() == null ? newAgent2.getName() == null : newAgent.getName().equals(newAgent2.getName());
		}

		private boolean equalCheck(Agent agent1, Agent agent2)
		{
			if (agent1 == null)
				return agent2 == null;
			else if (agent2 == null)
				return false;
			
			return 	agent1.getName() == null ? agent2.getName() == null : agent1.getName().equals(agent2.getName()) &&
					agent1.getType() == null ? agent2.getType() == null : agent1.getType().equals(agent2.getType());
		}

		public String getProcessId()
		{
			try
			{
				return (agent == null ? (newAgent == null ? "" : String.valueOf(newAgent.getProcess().getProcessId())) : agent.getProcess().getProcessId()); //$NON-NLS-1$
			}
			catch (Exception e)
			{
				return null;
			}
		}
		
		public boolean isNew()
		{
			return newAgent != null;
		}

		/**
		 * @return the profilingType
		 */
		public IProfilingSetType getProfilingType()
		{
			return profilingType;
		}

		/**
		 * @param profilingType the profilingType to set
		 */
		public void setProfilingType(IProfilingSetType profilingType)
		{
			this.profilingType = profilingType;
		}
				
		public String toString()
		{
			StringBuffer sb = new StringBuffer();
			sb.append(dataCollector == null ? "" : dataCollector.getId()).
			append(analysisType == null ? "" : analysisType.getId()).
			append(profilingType == null ? "" : profilingType.getId());
			
			return sb.toString();
		}
	}
	
	
	/**
	 * A message tree item is used to display a message in the agent
	 * check box tree
	 */
	public static class MessageTreeItem
	{
		private String message;
		private int severity;
		
		public MessageTreeItem(String message, int severity)
		{
			this.message = message;
			this.severity = severity;
		}

		/**
		 * @return the message
		 */
		public String getMessage()
		{
			return message;
		}

		/**
		 * @param message the message to set
		 */
		public void setMessage(String message)
		{
			this.message = message;
		}

		/**
		 * @return the severity
		 */
		public int getSeverity()
		{
			return severity;
		}

		/**
		 * @param severity the severity to set
		 */
		public void setSeverity(int severity)
		{
			this.severity = severity;
		}
				
	}
	
	/**
	 * This deferred adapter is used to retrieve the agents that are 
	 * running on a particular host.
	 */
	public static class DeferredAgentAdapter implements IDeferredWorkbenchAdapter 
	{
		/**
		 * The launch configuration
		 */
		private ILaunchConfiguration configuration;

		/**
		 * The data collector associations with the attach launch configuration type
		 */
		private DataCollectorAssociation dataCollectorAssociations;

		private AttachAgentsTab agentsTab;

	
		public DeferredAgentAdapter(ILaunchConfiguration configuration, AttachAgentsTab agentsTab)
		{
			this.configuration = configuration;
			this.agentsTab = agentsTab;
		}

		public void fetchDeferredChildren(Object parent, IElementCollector collector, IProgressMonitor monitor)
		{
			/* We need to fetch the agents running on the selected host */
			if (parent instanceof AgentTreeInput)
			{
				Node node = null;				
				try
				{					
					String hostname = configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_HOSTNAME, CommonUITraceConstants.LOCAL_HOST); //$NON-NLS-1$
					int port = configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PORT, -1);
					monitor.beginTask(TraceMessages.AttachRetrievingAgents, 5); //$NON-NLS-1$
					
					/* Connect to the Agent Controller using the old execution framework to retrieve old agents */ 
					node = PDCoreUtil.profileConnect(hostname, String.valueOf(port), false);
					if (node == null)
						return;
					
					monitor.worked(1);
					Enumeration processes = node.listProcesses();
					if (processes == null)
						return;
					
					monitor.worked(1);
					AgentTreeItem[] items = new AgentTreeItem[0];
					/* For every process running */
					while (processes.hasMoreElements())
					{
						Process process = (Process)processes.nextElement();
						Enumeration agents = process.listAgents();
						
						if (agents == null)
							continue;
						
						/* For every agent */
						while (agents.hasMoreElements())
						{
							Agent agent = (Agent)agents.nextElement();	
							if (agent.isAttached())
								continue;
							items = determineDataCollectorMapping(agent);
							collector.add(items, monitor);
						}
					}					
					monitor.worked(1);				
					
					/* Connect to the agent controller via the new execution framework to retrieve the new agents */					
					try
					{
						if (port < 0)
							return;
						
						

						IAgentController ac = AgentControllerPool.getInstance().getConnection(hostname, port);

						NodeFactory.createNode(hostname);
							
						monitor.worked(1);
						
						IAgent[] runningAgents = ac.queryRunningAgents();
						for (int i = 0; i < runningAgents.length; i++)
						{
							if (runningAgents[i].isMonitored())
								continue;
							items = determineDataCollectorMapping(runningAgents[i]);
							collector.add(items, monitor);
						}
						
						monitor.worked(1);
					}
					catch (Exception e)
					{
						collector.add(new MessageTreeItem(TraceMessages.AttachAttachErrorRetrievingAgents, IStatus.ERROR), monitor); //$NON-NLS-1$
						UIPlugin.getDefault().log(e);
					}
				} 
				catch (CoreException e)
				{
					collector.add(new MessageTreeItem(TraceMessages.AttachAttachErrorFindingHostInfo, IStatus.ERROR), monitor); //$NON-NLS-1$
					UIPlugin.getDefault().log(e);
				} 
				catch (NotConnectedException e)
				{
					collector.add(new MessageTreeItem(TraceMessages.CONH_ERROR_, IStatus.ERROR), monitor);
					UIPlugin.getDefault().log(e);
				}
				finally
				{			
					monitor.done();
					collector.done();	
				}
			}
		}

		/**
		 * Determine the set of data collectors that are bound to the agent
		 * name and type passed in.
		 * 
		 * @param agent The agent to find the binding from
		 * @return The data collectors of 'agent'
		 */
		private AgentTreeItem[] determineDataCollectorMapping(Object agent)
		{
			List mappedDataCollectors = new ArrayList();
			Agent oldAgent = null;
			IAgent newAgent = null;
			
			if (agent instanceof Agent)
				oldAgent = (Agent) agent;
			else if (agent instanceof IAgent)
				newAgent = (IAgent)agent;
			else
				return new AgentTreeItem[0];
			
			try
			{				
				if (dataCollectorAssociations == null)
				{
					dataCollectorAssociations = DataCollectorManager.getInstance().getDataCollectorAssociator(configuration.getType().getIdentifier());
				}
				
				DataCollector[] dataCollectors = dataCollectorAssociations.getDataCollectors();
				for (int i = 0; i < dataCollectors.length; i++)
				{
					DataCollectorAssociationData associationData = dataCollectorAssociations.getDataCollectorAssociationData(dataCollectors[i].getId());
					String[] applicableAgents = associationData.getApplicableAgents();
					
					for (int j = 0; j < applicableAgents.length; j++)
					{
						AgentDeclaration agentDeclaration = AgentDeclarationManager.getInstance().getAgentDeclaration(applicableAgents[j]);
						boolean matched = oldAgent == null ? 	agentDeclaration.getName().equals(newAgent.getName()) :
																("*".equals(agentDeclaration.getName()) || agentDeclaration.getName().equals(oldAgent.getName())) && agentDeclaration.getType().equals(oldAgent.getType()); //$NON-NLS-1$
						if (matched)
						{
							AgentTreeItem agentTreeItem = oldAgent == null ? new AgentTreeItem(dataCollectors[i], newAgent) : new AgentTreeItem(dataCollectors[i], oldAgent);
							mappedDataCollectors.add(agentTreeItem);
							break;
						}
					}
				}
				
				/* For backward compatibility */
				if (mappedDataCollectors.size() <= 0 && oldAgent != null)
				{
					AgentTreeItem agentTreeItem = new AgentTreeItem(null, oldAgent);
					mappedDataCollectors.add(agentTreeItem);
				}
			} 
			catch (CoreException e)
			{
				UIPlugin.getDefault().log(e);
			}
			
			return (AgentTreeItem[])mappedDataCollectors.toArray(new AgentTreeItem[mappedDataCollectors.size()]);
		}

		public ISchedulingRule getRule(Object object)
		{			
			return null;
		}

		public boolean isContainer()
		{			
			return true;
		}

		public Object[] getChildren(Object o)
		{
			return null;
		}

		public ImageDescriptor getImageDescriptor(Object object)
		{			
			return null;
		}

		public String getLabel(Object o)
		{
			return TraceMessages.AttachAttachHost; //$NON-NLS-1$
		}

		public Object getParent(Object child)
		{
			if (child instanceof AgentTreeItem)
			{
				return ((AgentTreeItem)child).getParent();
			}
			
			return null;
		}
		
	}
}
