/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.ui.internal.launcher;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.AbstractLaunchConfigurationTab;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.trace.internal.ui.PDPluginImages;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.core.ProfileLaunchUtil;
import org.eclipse.hyades.trace.ui.internal.launcher.AttachAgentTabProviders.AgentTreeInput;
import org.eclipse.hyades.trace.ui.internal.launcher.AttachAgentTabProviders.AgentTreeItem;
import org.eclipse.hyades.trace.ui.internal.launcher.AttachAgentTabProviders.AttachAgentContentProvider;
import org.eclipse.hyades.trace.ui.internal.launcher.AttachAgentTabProviders.AttachAgentLabelProvider;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.hyades.trace.ui.launcher.ProfilingSetsManagerCopy;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTreeViewer;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.tptp.platform.common.internal.ICommonConstants;
import org.eclipse.tptp.trace.ui.internal.launcher.core.AnalysisType;
import org.eclipse.tptp.trace.ui.internal.launcher.core.DataCollectorAssociation;
import org.eclipse.tptp.trace.ui.internal.launcher.core.DataCollectorAssociationData;
import org.eclipse.tptp.trace.ui.internal.launcher.core.DataCollectorManager;
import org.eclipse.tptp.trace.ui.internal.launcher.core.EditOptionDelegate;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherConstants;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LightConfigurationLoader;
import org.eclipse.tptp.trace.ui.provisional.launcher.IConfiguration;
import org.eclipse.tptp.trace.ui.provisional.launcher.IConfigurationPage;

/**
 * The attach agent tab used to select the agents to attach to
 * 
 * @author Ali Mehregani
 */
public class AttachAgentsTab extends AbstractLaunchConfigurationTab implements 
	ICheckStateListener, 
	ISelectionChangedListener, 
	IDoubleClickListener, 
	SelectionListener,
	ILaunchConfigurationTabUpdater
{	
	/**
	 * The check box tree used to display the available agents
	 */
	private CheckboxTreeViewer agentsCheckBoxTree;
	
	/**
	 * The edit option that appears on the right side of the available agents
	 */
	private Button editOptions;
	
	/**
	 * The edit option selection listener
	 */
	private SelectionListener editOptionDelegate;

	/**
	 * The launch configuration that this tab is initialized with
	 */
	private ILaunchConfiguration configuration;
	
	/**
	 * Used to indicate if the tab has been initialized yet
	 */
	private boolean initialized;

	/**
	 * The error message of this tab
	 */
	private String errorMessage;

	/**
	 * Indicates the enable status of the validation
	 */
	private boolean enableValidation;

	/**
	 * The profiling set manager
	 */
	private ProfilingSetsManagerCopy profilingSetsManagerWC;
	

	public AttachAgentsTab()
	{
		enableValidation = true;
	}
	
	
	public void createControl(Composite parent)
	{
		Composite content = new Composite(parent, SWT.NULL);		
		content.setLayout(new GridLayout());
		setControl(content);
		createVerticalSpacer(content, 1);
		content.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

		Label agentsLabel = new Label(content, SWT.NONE);
		GridData agentsLabelGD = new GridData(SWT.FILL, SWT.DEFAULT, true, false);
		agentsLabelGD.horizontalIndent = 10;
		agentsLabel.setLayoutData(agentsLabelGD);
		agentsLabel.setText(TraceMessages.LST_AG);
		
		Composite treeComposite = new Composite(content, SWT.NONE);
		treeComposite.setLayout(new GridLayout(2, false));
		treeComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		
		agentsCheckBoxTree = new CheckboxTreeViewer(treeComposite);
		agentsCheckBoxTree.setContentProvider(new AttachAgentContentProvider(agentsCheckBoxTree, this));
		agentsCheckBoxTree.setSorter(null);		
				
		agentsCheckBoxTree.setLabelProvider(new AttachAgentLabelProvider());
		agentsCheckBoxTree.setInput(null);
		agentsCheckBoxTree.getTree().setLayout(new GridLayout());
		agentsCheckBoxTree.getTree().setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		agentsCheckBoxTree.addCheckStateListener(this);
		agentsCheckBoxTree.addSelectionChangedListener(this);		
		agentsCheckBoxTree.addDoubleClickListener(this);		
		
		Composite buttonComposite = new Composite(treeComposite, SWT.NONE);
		buttonComposite.setLayout(new GridLayout());
		buttonComposite.setLayoutData(new GridData(SWT.DEFAULT, SWT.TOP, false, true));
		
		editOptions = createButton(buttonComposite, TraceMessages.LAUNCH_CONTROL_EDIT, null);
		editOptionDelegate = EditOptionDelegate.getInstance(this, getProfilingSetWC());
		editOptions.setEnabled(false);
		editOptions.addSelectionListener(editOptionDelegate);
		
		createButton(buttonComposite, TraceMessages.AttachRefresh, this);		 //$NON-NLS-1$
	}

	private Button createButton(Composite parent, String text, SelectionListener listener)
	{
		Button button = new Button (parent, SWT.NONE);
		GridData buttonGD = new GridData(SWT.FILL, SWT.DEFAULT, true, false);
		buttonGD.widthHint = 100;
		button.setLayoutData(buttonGD);
		button.setText(text);
		
		if (listener != null)
			button.addSelectionListener(listener);
		
		return button;
	}

	public String getName() 
	{
		return TraceMessages.TB_NAGENT;
	}

	public Image getImage() 
	{
		return PDPluginImages.getImage(PDPluginImages.IMG_UI_AGENT);
	}

	public ProfilingSetsManagerCopy getProfilingSetWC()
	{
		if (profilingSetsManagerWC == null)
			profilingSetsManagerWC = new ProfilingSetsManagerCopy();
		
		return profilingSetsManagerWC;
	}
	/**
	 * This method is used to initialize the agents tab after the
	 * available agents have been retrieved from the selected host.
	 */
	public void initializeAfterFetch ()
	{
		try
		{
			SelectedAgent[] selectedAgents = LauncherUtility.retrieveSelectedAgents(configuration);			
			ITreeContentProvider contentProvider = ((ITreeContentProvider)agentsCheckBoxTree.getContentProvider());
			Tree tree = agentsCheckBoxTree.getTree();
			if (tree.isDisposed())
				return;
			
			TreeItem[] treeElements = tree.getItems();
			
			/* Index using the process id */
			Map availableAgents = new Hashtable(); 
			for (int i = 0; i < treeElements.length; i++)
			{
				if (!(treeElements[i].getData() instanceof AgentTreeItem))
					continue;
				AgentTreeItem item = (AgentTreeItem)treeElements[i].getData();
				availableAgents.put(item.getProcessId(), item);				
			}
			
			/* Read in each selection and select the appropriate item */
			for (int i = 0; i < selectedAgents.length; i++)
			{				
				AgentTreeItem item = (AgentTreeItem)availableAgents.get(selectedAgents[i].getPid());
				if (item == null || (item.getDataCollector() != null && !selectedAgents[i].getCollectorId().equals(item.getDataCollector().getId())))
					continue;
					
				String[] analysisTypes = selectedAgents[i].getAnalysisTypes();
				if (analysisTypes == null || analysisTypes.length <= 0)
				{
					agentsCheckBoxTree.setChecked(item, true);
					continue;
				}
				
				Object[] availableAnalysisTypes = contentProvider.getChildren(item);
				
				/* For each available analysis type */
				for (int j = 0; j < availableAnalysisTypes.length; j++)
				{
					/* For each selected analysis type */
					for (int k = 0; k < analysisTypes.length; k++)
					{
						AgentTreeItem currentItem = (AgentTreeItem)availableAnalysisTypes[j];
						String id = "";
						if (currentItem.getAnalysisType() != null)
						{
							id = currentItem.getAnalysisType().getId();
						}
						else if (currentItem.getProfilingType() != null)
						{
							id = currentItem.getProfilingType().getId();
						}
						
						if (id.equals(analysisTypes[k]) && !agentsCheckBoxTree.getChecked( availableAnalysisTypes[j]))
						{
							ProfileLaunchUtil.changeItemCheckedState(agentsCheckBoxTree, AttachAgentsTab.this, availableAnalysisTypes[j], true);
						}
					}
				}
			}							
		} 
		catch (CoreException e)
		{
			UIPlugin.getDefault().log(e);			
		}	

		initialized = true;	
	}
	
	public void initializeFrom(ILaunchConfiguration configuration)
	{	
		if (this.configuration == null || !this.configuration.equals(configuration))
		{
			this.configuration = configuration;
			agentsCheckBoxTree.setInput(new AgentTreeInput (configuration, this));
			initialized = false;
		}		
	}

	public void performApply(ILaunchConfigurationWorkingCopy configuration)
	{
		if (!initialized)
			return;
						
		/* Read in the checked elements and serialize it into string */
		Object[] checkedElements = agentsCheckBoxTree.getCheckedElements();
		
		String selectionSerailized = ""; //$NON-NLS-1$
		for (int i = 0; i < checkedElements.length; i++)
		{
			if (!(checkedElements[i] instanceof AgentTreeItem))
				continue;
			
			AgentTreeItem agentTreeItem = (AgentTreeItem)checkedElements[i];
			boolean analysisTypePresent = agentTreeItem.getAnalysisType() != null;
			boolean profilingTypePresent = agentTreeItem.getProfilingType() != null;
				
			if (!analysisTypePresent && !profilingTypePresent)
			{
				if (selectionSerailized.length() > 0)
					selectionSerailized += ";"; //$NON-NLS-1$
				
				String processId = agentTreeItem.getProcessId();
				String dataCollectorID = (agentTreeItem.getDataCollector() == null ? " " : agentTreeItem.getDataCollector().getId()); //$NON-NLS-1$ //$NON-NLS-2$
				
				//for backward compatibility purpose
				if (dataCollectorID.equals(" ") && 
						!agentTreeItem.isNew() &&
						agentTreeItem.getAgent() instanceof Agent && 
						agentTreeItem.getAgent().getType().equals(ICommonConstants.PROFILE_AGENT_TYPE)) { //$NON-NLS-1$ //$NON-NLS-2$
					dataCollectorID="org.eclipse.tptp.trace.ui.jvmpiMechanism"; //$NON-NLS-1$ //$NON-NLS-2$
				}
				
				selectionSerailized += 	processId + ":" +  //$NON-NLS-1$
										(agentTreeItem.isNew() ? agentTreeItem.getNewAgent().getName() : agentTreeItem.getAgent().getName()) + ":" +  //$NON-NLS-1$
										dataCollectorID + ":"; //$NON-NLS-1$ //$NON-NLS-2$
			}
			else
			{
				selectionSerailized += (analysisTypePresent ? agentTreeItem.getAnalysisType().getId() : agentTreeItem.getProfilingType().getId()) + ",";  //$NON-NLS-1$
			}
		}
		
		
		/* Store the serialized selection */
		configuration.setAttribute(IProfileLaunchConfigurationConstants.ATTR_AGENTS, selectionSerailized);		
		org.eclipse.hyades.trace.ui.internal.launcher.ProfileLaunchUtil.createDummyProfilingSet(getProfilingSetWC(), configuration, agentsCheckBoxTree.getCheckedElements());
		
	}

	public void deactivated(ILaunchConfigurationWorkingCopy workingCopy)
	{
		initialized = false;
	}
	
	public void setDefaults(ILaunchConfigurationWorkingCopy configuration)
	{
			
	}

	public void checkStateChanged(CheckStateChangedEvent event)
	{
		if (!(event.getElement() instanceof AgentTreeItem))
			return;
		
		AgentTreeItem item = (AgentTreeItem)event.getElement();
		agentsCheckBoxTree.setSelection(new StructuredSelection(item));
		boolean analysisTypeSelection = item.getAnalysisType() != null;
		boolean profilingTypeSelection = item.getProfilingType() != null;
		
		
		/* Root item */
		if (item.getDataCollector() != null)
		{			
			if (event.getChecked())
			{
				ITreeContentProvider provider = (ITreeContentProvider)agentsCheckBoxTree.getContentProvider();
				Object[] children = provider.getChildren(item);				
				if (children != null)
				{
					enableValidation = false;
					for (int i = children.length - 1; i >= 0; i--)
					{
						ProfileLaunchUtil.changeItemCheckedState(agentsCheckBoxTree, this, children[i], true);
					}
					enableValidation = true;					
				}	
				
			}
			else
			{
				agentsCheckBoxTree.setSubtreeChecked(item, false);
			}
		}
		/* Analysis type is selected */
		else if (analysisTypeSelection || profilingTypeSelection)
		{
			/* Update the state of items based on coexistance violation rules */
			if (event.getChecked())
			{
				if (analysisTypeSelection)
					updateViolators(agentsCheckBoxTree, event);				
				
								
				/* We need to reset the configuration pages */
				try
				{
					DataCollectorAssociation association = DataCollectorManager.getInstance().getDataCollectorAssociator(this.configuration.getType().getIdentifier());
					DataCollectorAssociationData associationData = association.getDataCollectorAssociationData(((AgentTreeItem)item.getParent()).getDataCollector().getId());
					LightConfigurationLoader[] configs = new LightConfigurationLoader[analysisTypeSelection ? 2 : 1];
					configs[0] = associationData.getConfigurationLoader();
					
					if (analysisTypeSelection)
					{
						configs[1] = item.getAnalysisType().getConfigurationLoader();
						
						for (int i = 0; i < configs.length; i++)
						{
							if (configs[i] != null && configs[i].isEditable())
							{
								IConfiguration configuration = (IConfiguration)configs[i].getConfigurationClass();
								IConfigurationPage[] pages = configuration.getConfigurationPages();
								for (int j = 0; j < pages.length; j++)
								{
									pages[j].reset(this.configuration);
								}
							}
						}
					}					
				}
				catch (CoreException e)
				{
					UIPlugin.getDefault().log(e);
				}				
			}
			
			ProfileLaunchUtil.changeParentGrayState(agentsCheckBoxTree, event);
		}
		
		update();
	}

	private void updateViolators(CheckboxTreeViewer treeViewer, CheckStateChangedEvent event)
	{
		/* Get the sibilings of the analysis type selected */
		AgentTreeItem selectedItem = (AgentTreeItem)event.getElement();
		ITreeContentProvider contentProvider = ((ITreeContentProvider)treeViewer.getContentProvider());
		Object parent = contentProvider.getParent(selectedItem);
		Object[] children = contentProvider.getChildren(parent);
		
		Map violators = selectedItem.getAnalysisType().getCoexistanceViolators();
		if (violators == null)
			return;
		
		
		String selectedAnalysisTypeId = selectedItem.getAnalysisType().getId();
		List selectedSibilings = new ArrayList();
		Object[] selectedItems = treeViewer.getCheckedElements();
		
		for (int i = 0; i < selectedItems.length; i++)
		{
			for (int j = 0; j < children.length; j++)
			{
				if (selectedItems[i].equals(children[j]))
					selectedSibilings.add(children[j]);
			}
		}
		
		for (int i = 0, selectedSibilingCount = selectedSibilings.size(); i < selectedSibilingCount; i++)
		{
			AgentTreeItem item = (AgentTreeItem)selectedSibilings.get(i);
			AnalysisType analysisType = item.getAnalysisType();
			if (analysisType != null && (violators.get(LauncherConstants.ALL) != null || violators.get(analysisType.getId()) != null || analysisType.getCoexistanceViolators().get(selectedAnalysisTypeId) != null))
			{
				treeViewer.setChecked(item, false);
			}
		}
	}

	public void selectionChanged(SelectionChangedEvent event)
	{
		ProfileLaunchUtil.updateButtonStatus(configuration, event, editOptions, null, agentsCheckBoxTree, this);
	}

	public void doubleClick(DoubleClickEvent event)
	{
		ProfileLaunchUtil.handleDoubleClick(editOptions, editOptionDelegate);
	}

	public void widgetDefaultSelected(SelectionEvent e)
	{
		/* Doesn't need to be implemented */
	}

	public void widgetSelected(SelectionEvent event)
	{
		agentsCheckBoxTree.refresh();
	}

	public void update()
	{
		updateLaunchConfigurationDialog();
	}

	public boolean isValid(ILaunchConfiguration launchConfig)
	{
		if (!enableValidation)
			return true;
		
		IStatus status = ProfileLaunchUtil.checkValidity(launchConfig, agentsCheckBoxTree, getProfilingSetWC());
		errorMessage = status.isOK() ? null : status.getMessage();
		return status.isOK();		
	}
	
	public String getErrorMessage()
	{
		return errorMessage;
	}

	public CheckboxTreeViewer getAgentsCheckBoxTree() {
		return agentsCheckBoxTree;
	}
}
