/**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AttachDelegate.java,v 1.9 2008/06/06 00:17:41 jkubasta Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.launcher;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.model.ILaunchConfigurationDelegate;
import org.eclipse.debug.ui.DebugUITools;
import org.eclipse.debug.ui.IDebugUIConstants;
import org.eclipse.hyades.internal.execution.local.common.Options;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.AgentConfigurationEntry;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.models.hierarchy.HierarchyFactory;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCConfiguration;
import org.eclipse.hyades.models.hierarchy.TRCOption;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.util.AttachConfigurationDetail;
import org.eclipse.hyades.trace.ui.internal.util.PDCoreUtil;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.hyades.trace.ui.launcher.ProfilingAttribute;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.tptp.platform.common.internal.CommonPlugin;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIConstants;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIPlugin;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceConstants;
import org.eclipse.tptp.platform.execution.client.agent.IAgent;
import org.eclipse.tptp.platform.execution.client.core.IAgentController;
import org.eclipse.tptp.platform.execution.client.core.IProcess;
import org.eclipse.tptp.platform.execution.util.internal.AgentControllerPool;
import org.eclipse.tptp.trace.ui.internal.launcher.core.AnalysisType;
import org.eclipse.tptp.trace.ui.internal.launcher.core.AnalysisTypeManager;
import org.eclipse.tptp.trace.ui.internal.launcher.core.DataCollectorAssociation;
import org.eclipse.tptp.trace.ui.internal.launcher.core.DataCollectorManager;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherConstants;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherMessages;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LightConfigurationLoader;
import org.eclipse.tptp.trace.ui.provisional.launcher.AgentSelection;
import org.eclipse.tptp.trace.ui.provisional.launcher.DataCollectionEngineAttribute;
import org.eclipse.tptp.trace.ui.provisional.launcher.IConfiguration;
import org.eclipse.tptp.trace.ui.provisional.launcher.IConfigurationPage;
import org.eclipse.tptp.trace.ui.provisional.launcher.IDataCollectorAgentAttach;
import org.eclipse.tptp.trace.ui.provisional.launcher.IDataCollectorBaseLauncher;
import org.eclipse.tptp.trace.ui.provisional.launcher.IDataCollectorMutualLauncher;
import org.eclipse.tptp.trace.ui.provisional.launcher.ILightConfiguration;

import com.ibm.icu.util.StringTokenizer;

/**
 * The attach agent launch delegate.  This delegate gets invoked when users attach to an
 * agent.
 * 
 * @author znensi
 * @author Ali Mehregani
 */
public class AttachDelegate implements ILaunchConfigurationDelegate
{
	/**
	 * @see org.eclipse.debug.core.model.ILaunchConfigurationDelegate#launch(org.eclipse.debug.core.ILaunchConfiguration, java.lang.String, org.eclipse.debug.core.ILaunch, org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void launch(final ILaunchConfiguration conf, final String mode, ILaunch launch, IProgressMonitor monitor) throws CoreException 
	{		
		Map launchDelegates = null;
		try 
		{
			List selectedAgents = retrieveSelectedAgents(conf);
			monitor.worked(1);
			
			/* Display the attach dialog if no detached agent selections are found */
			if (selectedAgents.size() <= 0 || !unattachedAgents(selectedAgents)) 
			{
				monitor.setCanceled(true);
				Display.getDefault().asyncExec(new Runnable() 
				{
					public void run() 
					{
						Shell shell = UIPlugin.getActiveWorkbenchShell();
						DebugUITools.openLaunchConfigurationDialog(shell, conf, IDebugUIConstants.ID_PROFILE_LAUNCH_GROUP, new Status(Status.OK, UIPlugin.getPluginId(), RunAttachProfileStatusHandler.CODE, "", null)); //$NON-NLS-1$
					}
				});
				
				return;
			}
			
		
			monitor.beginTask(LauncherMessages.LAUNCHING_BEGIN_TASK, selectedAgents == null ? 4 : selectedAgents.size() + 4);
						
			/* Get the launch delegates of the selected data collectors */
			ArrayList launchDelegateList = new ArrayList();
			launchDelegates = findLaunchDelegates(conf, selectedAgents, launchDelegateList);
			LauncherUtility.delegateInit(launchDelegateList, conf, mode, launch, monitor);			
						
			
			monitor.subTask(TraceMessages.LNCH_MSGV);			
			monitor.worked(1);

			boolean success = ProfileLaunchUtil.performProfilingTypesLaunch(conf);		
			if (!success) 
			{
				monitor.setCanceled(true);
				return;
			}

			monitor.worked(1);
				
			
			/* Attach to the agents and assocatie the analysis types */
			attachToAgents(selectedAgents, launchDelegates, conf, launch);				
			
			/* We need to invoke the postLaunch methods of the secondary launch delegates */
			postLaunch(launchDelegates, Status.OK_STATUS);			
			monitor.worked(1);
		}
		catch (CoreException e)
		{
			monitor.setCanceled(true);			
			postLaunch (launchDelegates, new Status(IStatus.ERROR, UIPlugin.getPluginId(), IStatus.ERROR, e.getMessage(), e));
			throw e;
		}
	}
	
	
	private void attachToAgents(List selectedAgents, Map launchDelegates, ILaunchConfiguration conf, ILaunch launch)
	{
		try
		{		
			Vector agents = new Vector();
			for (int i = 0, selectedAgentCount = selectedAgents.size(); i < selectedAgentCount; i++)
			{
				SelectedAgent agentSelection = (SelectedAgent)selectedAgents.get(i);				
				agents.add(agentSelection.getAgent() == null ? (Object)agentSelection.getNewAgent() : agentSelection.getAgent());
			}
			
			/* Find the options */
			Vector options = retrieveOptions(selectedAgents, conf);
			
			/* Do the attach */
	    	AttachConfigurationDetail configurationDetail = new AttachConfigurationDetail (conf , launch);
	    	configurationDetail.setSwitchPerspective(true); 	
	    	configurationDetail.setOptions(options);
	    	configurationDetail.setAgents(agents);
	    	Vector attachedAgents = PDCoreUtil.attach(configurationDetail);
	    	
	    	if (attachedAgents.size() != agents.size())
	    		LauncherUtility.createCoreException(IStatus.ERROR, LauncherMessages.AttachErrorAttaching);
	    	
	    	List agentSelection = new ArrayList();
	    	for (int i = 0, agentCount = attachedAgents.size(); i < agentCount; i++)
			{
	    		AgentSelection selection = new AgentSelection();
				selection.setAgent(agents.get(i));
				selection.setAgentProxy((TRCAgentProxy)attachedAgents.get(i));
				agentSelection.add(selection);
			}
	    	
	    	/* Associate analysis types */
	    	associateAnalysisTypes(attachedAgents, selectedAgents);
	    	
	    	
	    	/* If in case the agent does not have an associated launch delegate, then use the PIProcessListener
	    	 * to 
	    	 */ 
			if (launchDelegates == null)
				return;
						
			
			for (Iterator agentNames = launchDelegates.keySet().iterator(); agentNames.hasNext();)
			{
				String agentName = (String) agentNames.next();
				List applicableAgents = new ArrayList();
				for (int i = 0, agentSelectionCount = agentSelection.size(); i < agentSelectionCount; i++)
				{
					AgentSelection selection = (AgentSelection)agentSelection.get(i);					
					String type = selection.getAgent() instanceof Agent ? ((Agent)selection.getAgent()).getType() : "";
					if (agentName.equals(selection.getAgentProxy().getName() + type))
						applicableAgents.add(selection);						
				}
				
				IDataCollectorAgentAttach launchDelegate = (IDataCollectorAgentAttach)launchDelegates.get(agentName);
				launchDelegate.launch((AgentSelection[])applicableAgents.toArray(new AgentSelection[applicableAgents.size()]));
			}			
		} 
		catch (CoreException e)
		{
			UIPlugin.getDefault().log(e);
		}	
	}


	private Vector retrieveOptions(List selectedAgents, ILaunchConfiguration config)
	{
		final Vector options = new Vector();
		
		addAttributes(options, LauncherUtility.getDefaultProfilingOptionsAttributes());
		
		try
		{			
			for (int i = 0, selectedAgentCount = selectedAgents.size(); i < selectedAgentCount; i++)
			{
				SelectedAgent agentSelection = (SelectedAgent)selectedAgents.get(i);
				if (agentSelection.getCollectorId() != null)
				{
					DataCollectorAssociation associator = DataCollectorManager.getInstance().getDataCollectorAssociator(config.getType().getIdentifier());
					if (agentSelection.getCollectorId().length() > 0)
					{
						LightConfigurationLoader configLoader = associator.getDataCollectorAssociationData(agentSelection.getCollectorId()).getConfigurationLoader();						
						addAttributes (options, configLoader, config);
					}

					String[] analysisTypeIds = agentSelection.getAnalysisTypes();					
					if (analysisTypeIds == null)
						continue;
					
					for (int j = 0; j < analysisTypeIds.length; j++)
					{
						AnalysisType analysisType = AnalysisTypeManager.getInstance().getAnalysisType(analysisTypeIds[j]);
						if (analysisType != null)
						{
							LightConfigurationLoader configLoader = analysisType.getConfigurationLoader();
							addAttributes (options, configLoader, config);
						}
						/* This might be a profiling type */
						else
						{
							Vector profilingSetOptions = ProfilingSetsManager.instance().getOptions(config);
							ProfilingAttribute[] attributes = new ProfilingAttribute[profilingSetOptions.size()];
							
							for (int k = 0; k < attributes.length; k++)
							{
								AgentConfigurationEntry entry = (AgentConfigurationEntry)profilingSetOptions.get(k);
								attributes[k] = new ProfilingAttribute(entry.getName(), entry.getValue());								
							}
							addAttributes(options, attributes);

//							final IProfilingSetType profilingSetType = (IProfilingSetType)ProfilingSetsManager.instance().getProfilingTypes().get(analysisTypeIds[j]);
//							if (profilingSetType != null)
//							{
//								UIPlugin.getDefault().getWorkbench().getDisplay().syncExec(new Runnable(){
//
//									public void run()
//									{
//										IProfilingType profilingType = profilingSetType.getProfilingType();
//										Composite composite = new Composite(UIPlugin.getActiveWorkbenchShell(), SWT.NONE);
//										profilingType.createControl(composite, new ProfilingSetsManagerCopy());
//										addAttributes(options, profilingSetType.getProfilingType().getAttributes());
//										composite.dispose();									
//									}
//								});
//							}
						}
					}					
				}
			}			
		}
		catch (CoreException e)
		{
			UIPlugin.getDefault().log(e);
		}
		return options;
	}


	private void addAttributes(Vector options, LightConfigurationLoader configLoader, ILaunchConfiguration configuration)
	{
		if (configLoader == null)
			return;
		
		ILightConfiguration lightConfiguration = (ILightConfiguration)configLoader.getConfigurationClass();
		if (lightConfiguration instanceof IConfiguration)
		{
			IConfigurationPage[] pages = ((IConfiguration)lightConfiguration).getConfigurationPages();
			for (int i = 0; i < pages.length; i++)
			{
				pages[i].reset(configuration);
			}
		}
		DataCollectionEngineAttribute[] attributes = lightConfiguration.getAttributes();
		if (attributes == null)
			return;
		
		addAttributes(options, attributes);
	}
	

	private void addAttributes(Vector options, ProfilingAttribute[] attributes)
	{
		boolean foundOption;
		for (int i = 0; i < attributes.length; i++)
		{
			if (attributes[i].getValue() == null)
			{
				continue;
			}
			foundOption = false;
			String attributeName = attributes[i].getName();
			boolean option = attributeName.startsWith(ILightConfiguration.AC_OPT_PREFIX_NAME);
			boolean filter = attributeName.startsWith(LauncherConstants.SET_FILTER_PREFIX);
			boolean methodFilter = attributeName.startsWith(LauncherConstants.SET_METHOD_FILTER_PREFIX);
			
			attributeName = option ? attributeName.substring(ILightConfiguration.AC_OPT_PREFIX_NAME.length()) : 
							filter ? attributeName.substring(LauncherConstants.SET_FILTER_PREFIX.length()) : 
							methodFilter ? attributeName.substring(LauncherConstants.SET_METHOD_FILTER_PREFIX.length()) : attributeName;							
				
			for(int j = 0, optionCount = options.size(); j < optionCount; j++)
			{
				AgentConfigurationEntry pOption = (AgentConfigurationEntry)options.get(j);
				if(pOption.getName().equals(attributeName))
				{
					if(pOption.getValue().equals(Options.OPTION_VALUE_FALSE))
					{
						//false value is overridden by any other value
						pOption.setValue(attributes[i].getValue());
					}
					else if(pOption.getValue().equals(Options.OPTION_VALUE_NONE))
					{
						//none value is overridden by any other value
						pOption.setValue(attributes[i].getValue());
					}
					else if(attributes[i].getValue().equals(Options.OPTION_VALUE_FALSE))
					{
						//false value does not override anything
						return;
					}
					else if(pOption.getValue().equals(Options.OPTION_VALUE_NONE))
					{
						//none value does not override anything
						return;
					}
					else if(pOption.getName().equals(Options.OPTION_STACK_INFORMATION))
					{
						//OPTION_VALUE_NORMAL take precedence to OPTION_VALUE_NONE
						//OPTION_VALUE_BOUNDARY take precedence to OPTION_VALUE_NORMAL
						if(pOption.getValue().equals(Options.OPTION_VALUE_BOUNDARY))
							 return; //higher precedence
						
						if(!attributes[i].getValue().equals(Options.OPTION_VALUE_NONE))
							 pOption.setValue(attributes[i].getValue());
					}
					else if(pOption.getName().equals(Options.OPTION_TRACE_MODE))
					{
						if(pOption.getValue().equals(Options.OPTION_VALUE_NOOBJECTCORRELATION))
							pOption.setValue(attributes[i].getValue()); // override no object correlation value with any other value				
					}
					
					foundOption = true;
					break;
				}
			}
			
			if (!foundOption)
			{
				AgentConfigurationEntry entry = new AgentConfigurationEntry();
												
				entry.setName(attributeName);
				entry.setValue(attributes[i].getValue());
				entry.setType(filter ? LauncherConstants.SET_FILTER_PREFIX :
							  methodFilter ? LauncherConstants.SET_METHOD_FILTER_PREFIX : LauncherConstants.SET_OPTION_PREFIX);
				options.add(entry);
			}
		}	
	}


	private void associateAnalysisTypes(List attachedAgents, List selectedAgents)
	{		
		/* The number of attached agents is expected to equal the number of selected agents */
		if (attachedAgents.size() != selectedAgents.size())
			return;
		
		
		for (int i = 0, attachedAgentCount = attachedAgents.size(); i < attachedAgentCount; i++)
		{
			TRCAgentProxy agentProxy = (TRCAgentProxy)attachedAgents.get(i);
			SelectedAgent agentSelection = (SelectedAgent)selectedAgents.get(i);
			
			String[] analysisTypes = agentSelection.getAnalysisTypes();
			if (analysisTypes != null)						
			{
				TRCConfiguration configuration = HierarchyFactory.eINSTANCE.createTRCConfiguration();
				configuration.setActive(true);
				configuration.setAgentProxy(agentProxy);
				configuration.setName(LauncherConstants.SET_OPTION_PREFIX);
				
				for (int j = 0; j < analysisTypes.length; j++)
				{
					TRCOption option =  HierarchyFactory.eINSTANCE.createTRCOption();
					option.setConfiguration(configuration);
					
					AgentConfigurationEntry confEntry = LauncherUtility.associateAnalysisType(null, analysisTypes[j]);
					option.setKey(confEntry.getName());
					option.setValue(confEntry.getValue());
				}
			}
		}
	}

	/**
	 * Returns a map where KEY = agent name+type VALUE = an object of type IDataCollectorBaseLauncher representing
	 * the launch delegate of the agent
	 * 
	 * @param conf The launch configuration
	 * @param selectedAgents The selected agents
	 * @return The delegates indexed using the associated agent name
	 */
	private Map findLaunchDelegates(ILaunchConfiguration conf, List selectedAgents, List launchDelegateList)
	{
		Map launchDelegates = new Hashtable();
		
		try
		{
			for (int i = 0, selectedAgentsCount = selectedAgents.size(); i < selectedAgentsCount; i++)
			{
				SelectedAgent agentSelection = (SelectedAgent)selectedAgents.get(i);
				String dataCollectorId = agentSelection.getCollectorId();
				IDataCollectorBaseLauncher delegate = DataCollectorManager.getInstance().getDataCollectorLaunchDelegate(dataCollectorId, conf);
				if (delegate != null)
				{
					launchDelegateList.add(delegate);
					if (delegate.isMutualLauncher())
						launchDelegates.put(agentSelection.getAgentName() + (agentSelection.getAgent() == null ? "" : agentSelection.getAgent().getType()), delegate);
				}
			}
		}
		catch (CoreException e)
		{
			/* Ignore the exception */
		}
		
		return launchDelegates;
	}


	private boolean unattachedAgents(List agents)
	{
		if (agents != null)
		{
			for (int i = 0, agentCount = agents.size(); i < agentCount; i++)
			{
				SelectedAgent agentSelection = (SelectedAgent)agents.get(i);
				if ((agentSelection.getAgent() != null && !agentSelection.getAgent().isAttached()) ||
					(agentSelection.getNewAgent() != null && !agentSelection.getNewAgent().isMonitored()))
					return true;
			}
		}
		
		return false;
	}
	
	
	private void postLaunch(Map launchDelegates, IStatus status) throws CoreException
	{
		if (launchDelegates != null)
		{
			for (Iterator delegates = launchDelegates.values().iterator(); delegates.hasNext();)
			{
				((IDataCollectorMutualLauncher)delegates.next()).postLaunch(status);
			}	
		}		
	}
	

	protected static Process retrieveProcess(ILaunchConfiguration conf, String processID, boolean showErrorMsgs) {
		if (processID==null || conf==null)
			return null;
		
		try
		{
			IPreferenceStore store = CommonUIPlugin.getDefault().getPreferenceStore();
			String host = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_HOSTNAME, CommonUITraceConstants.LOCAL_HOST); //$NON-NLS-1$
			int port = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PORT, store.getDefaultInt(CommonUIConstants.LOCALHOST_PORT));
			Enumeration enum1 = PDCoreUtil.getNodeConfiguration(host, port, showErrorMsgs); 
			if (enum1 == null)
				return null;
			
			if (enum1 != null)
			{
				/* Index proceses using their PID */				
				while (enum1.hasMoreElements())
				{
					Process p = (Process)enum1.nextElement();
					if (p.getProcessId().equals(processID))
						return p;
				}				
			}			
		}
		catch (Exception e)
		{
			CommonPlugin.logError(e);
		} 
		return null;
	}
	
	protected static IAgent retrieveIAgent(ILaunchConfiguration conf, String processID, SelectedAgent agentSelection) {
		if (processID==null || conf==null)
			return null;
		
		try {
			IPreferenceStore store = CommonUIPlugin.getDefault().getPreferenceStore();
			String host = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_HOSTNAME, CommonUITraceConstants.LOCAL_HOST); //$NON-NLS-1$
			int port = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PORT, store.getDefaultInt(CommonUIConstants.LOCALHOST_PORT));
			
			IAgentController ac = AgentControllerPool.getInstance().getConnection(host, port);
	
			IAgent[] runningAgents = ac.queryRunningAgents();
			if (runningAgents != null)
			{
				for (int i = 0; i < runningAgents.length; i++)
				{
					IAgent agent = runningAgents[i];
					String pId = String.valueOf(agent.getProcess().getProcessId());
					String name = runningAgents[i].getName();
					if (pId.equals(processID) && name.equals(agentSelection.getAgentName()))
						return agent;
				}
			}	
		}
		catch (Exception e)
		{
			CommonPlugin.logError(e);
		} 		
		return null;
	}
	
	/**
	 * Retrieves active processes from the Agent Controller and returns a map
	 * with key = process id and value = process object
	 * 
	 * @param conf The launch configuration
	 * @param showErrorMsgs Indicates whether to show an error if there are problems
	 * connecting to an agent controller
	 * @return Active processes indexed using their PID.
	 */
	protected static Map retrieveProcesses (ILaunchConfiguration conf, boolean showErrorMsgs)
	{
		IPreferenceStore store = CommonUIPlugin.getDefault().getPreferenceStore();
		Map processes = new Hashtable();
	
		try
		{
			String host = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_HOSTNAME, CommonUITraceConstants.LOCAL_HOST); //$NON-NLS-1$
			int port = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PORT, store.getDefaultInt(CommonUIConstants.LOCALHOST_PORT));
			Enumeration enum1 = PDCoreUtil.getNodeConfiguration(host, port, showErrorMsgs); 
			if (enum1 == null)
				return null;
			
			if (enum1 != null)
			{
				/* Index proceses using their PID */				
				while (enum1.hasMoreElements())
				{
					Process process = (Process)enum1.nextElement();
					processes.put(process.getProcessId(), process);
				}				
			}			
					
			IAgentController ac = AgentControllerPool.getInstance().getConnection(host, port);

			IAgent[] runningAgents = ac.queryRunningAgents();
			if (runningAgents != null)
			{
				for (int i = 0; i < runningAgents.length; i++)
				{
					IProcess process = runningAgents[i].getProcess();
					String processId = String.valueOf(process.getProcessId());
					if (processes.get(processId) == null)
						processes.put(processId, process);
				}
			}			
		}
		catch (Exception e)
		{
			CommonPlugin.logError(e);
		}
		
		return processes;
	}
	
	
	/**
	 * Equivalent to retrieveProcesses (conf, true); 
	 */
	protected static Map retrieveProcesses(ILaunchConfiguration conf)
	{
		Map processes = retrieveProcesses (conf, true);
		return processes;
	}
	
	public static List retrieveSelectedAgents(ILaunchConfiguration conf)
	{
		try
		{
			String agents = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_AGENTS, ""); //$NON-NLS-1$
			
			if (agents.length() <= 0)
				return new ArrayList();
			
			StringTokenizer st = new StringTokenizer (agents, ";"); //$NON-NLS-1$
			ArrayList agentSelections = new ArrayList();
			while (st.hasMoreTokens())
			{
				agentSelections.add(new SelectedAgent(st.nextToken()));				
			}
			
			
			for (int i = 0, agentSelectionsCount = agentSelections.size(); i < agentSelectionsCount; i++)
			{
				SelectedAgent agentSelection = (SelectedAgent)agentSelections.get(i);		

				Process p = retrieveProcess(conf, agentSelection.getPid(), true);
				IAgent iA = retrieveIAgent(conf,agentSelection.getPid(),agentSelection);
				if (p!=null) {
					Agent agent = p.getAgent(agentSelection.getAgentName());
					if (agent!=null) {
						agentSelection.setAgent(agent);
					}
					else {
						if (iA!=null) {
							agentSelection.setNewAgent(iA);
						}
					}
				}
				else if (iA!=null) {
						agentSelection.setNewAgent(iA);
				}
			}

			return agentSelections;
		}
		catch (Exception e)
		{
			CommonPlugin.logError(e);
		}
		
		return new ArrayList();
	}	

	// parse and organize the agents in a more accessible data structure
	public static Map createMap(List list)
	{
		Map map = new HashMap();
		
		Iterator iter = list.iterator();
		while (iter.hasNext())
		{
			SelectedAgent agentAttribute = (SelectedAgent) iter.next();
			Vector agents = (Vector) map.get(agentAttribute.getPid());

			if (agents == null)
				agents = new Vector();
			
			agents.addElement(agentAttribute.getAgentName());
			map.put(agentAttribute.getPid(), agents);
		}
		
		Iterator keys = map.keySet().iterator();
		String key = null;
		Vector agents = null;		
		String[] stringArray = null; 
		
		while (keys.hasNext())
		{
			key = (String)keys.next();
			agents = (Vector) map.get(key);
			stringArray = new String[agents.size()];
			agents.toArray(stringArray);

			map.put(key, stringArray);
		}
		
		return map;
	}
}

