/**********************************************************************
 * Copyright (c) 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.launcher;

import java.io.File;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.AbstractLaunchConfigurationTab;
import org.eclipse.hyades.trace.internal.ui.PDPluginImages;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.core.TraceFileUI;
import org.eclipse.hyades.trace.ui.internal.core.TraceLocationUI;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceConstants;

/**
 * Represents the destination tab
 * 
 * @author znensi
 * @author Ali Mehregani
 */
public class DestinationTab extends AbstractLaunchConfigurationTab
{
	/** Used to create the control of the destination tab */
	private TraceLocationUI destinationUI;
	
	/** Used to create the controls for accepting a file name */
	private TraceFileUI fileUI;
	
	
	/** 
	 * The constructor
	 */
	public DestinationTab()
	{
		destinationUI = new TraceLocationUI();
		fileUI = new TraceFileUI();
		
		Listener listener = new Listener(){

			public void handleEvent(Event event)
			{
				updateLaunchConfigurationDialog();
			}};
		
		destinationUI.addListener(listener);
		fileUI.addListener(listener);
	}
	
	public void createControl(Composite parent)
	{
		Composite content = new Composite(parent, SWT.NULL);
		setControl(content);
		content.setLayout(new GridLayout());

		destinationUI.createControl(content);
		fileUI.createControl(content);		
	}

	public void setDefaults(ILaunchConfigurationWorkingCopy wc)
	{
		IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();
		
		wc.setAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_PROJECT, store.getString(CommonUITraceConstants.TRACE_PROJECT_NAME));
		wc.setAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_MONITOR, store.getString(CommonUITraceConstants.TRACE_MONITOR_NAME));
		wc.setAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_FILE, (String) null);
	}

	public void initializeFrom(ILaunchConfiguration conf)
	{
		try
		{
			IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();
			String location = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_PROJECT, "");
			String monitor = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_MONITOR, "");
			String file = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_FILE, "");
			boolean profileToFile = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILE_TO_FILE, false);
			if (location.length() == 0) location = store.getString(CommonUITraceConstants.TRACE_PROJECT_NAME);
			if (monitor.length() == 0) monitor = store.getString(CommonUITraceConstants.TRACE_MONITOR_NAME);
			
			destinationUI.setLocation(location);
			destinationUI.setMonitor(monitor);
			fileUI.setFile(file);
			fileUI.setEnabled(profileToFile);
			fileUI.getExternalBox().setSelection(profileToFile);
		}
		catch (CoreException e)
		{
			e.printStackTrace();
		}
	}

	public void performApply(ILaunchConfigurationWorkingCopy wc)
	{
		wc.setAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_PROJECT, destinationUI.getLocation().getText().trim());
		wc.setAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_MONITOR, destinationUI.getMonitor().getText().trim());
		wc.setAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILE_TO_FILE, fileUI.getExternalBox().getSelection());
		wc.setAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_FILE, fileUI.getFile());
	}

	public boolean isValid(ILaunchConfiguration conf)
	{
		setErrorMessage(null);
		setMessage(null);

		if (destinationUI.getLocation().getText().trim().length() == 0)
		{
			setErrorMessage(TraceMessages.TAB_ERRDP);
			return false;
		}

		if (destinationUI.getMonitor().getText().trim().length() == 0)
		{
			setErrorMessage(TraceMessages.TAB_ERRDM);
			return false;
		}

		if (fileUI.getSelection() && fileUI.getFile().trim().length() == 0)
		{
			setErrorMessage(TraceMessages.TAB_ERRDF);
			return false;
		}
		
		if (fileUI.getSelection() && fileUI.getFile().trim().length() > 0)
		{
			String fName = fileUI.getFile().trim();	
			String dirName = getDirectoryName(fName);
			String fileName = getFileName(fName);
					
			
			if (!(new File(dirName)).exists()){		
				setErrorMessage(TraceMessages.TAB_ERRDR);
				return false;
			}		
			
			if (fileName.equals("")|| fileName.length() == 0){		
				setErrorMessage(TraceMessages.TAB_ERRDR);
				return false;
			}
			
			if((new File(fName)).isDirectory()){
				setErrorMessage(TraceMessages.TAB_ERRDR);
				return false;
			}				
		}	

		return true;
	}
	
	public String getName()
	{
		return TraceMessages.TB_NDEST;
	}
	
	public Image getImage()
	{
		return PDPluginImages.getImage(PDPluginImages.IMG_UI_FOLDER);
	}
	
	private String getDirectoryName(String fName) {
		int i = fName.lastIndexOf(File.separatorChar);
		if (i >= 0) {
			fName = fName.substring(0, i);
		}
		return fName;
	}
	
	private String getFileName(String fName) {
		int i = fName.lastIndexOf(File.separatorChar);
		if (i >= 0) {
			fName = fName.substring(i + 1);
		}
		return fName;
	}

}
