/**********************************************************************
 * Copyright (c) 2010 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.launcher;

import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Platform;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.ILaunchConfigurationTab;
import org.eclipse.hyades.trace.ui.IProfilerFilterProcessorExtension;
import org.eclipse.hyades.trace.ui.IProfilerTabExtension;
import org.eclipse.hyades.trace.ui.IProfilerUIAdditionExtension;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IWorkingSet;

/**
 * Provides support for the extension points that were introduced as part of bug 323330.
 */
public class HyadesTraceUIExtensionSupportUtil {
	
	public enum TraceExtensionUIElementPos { TOP, BOTTOM };
	
	private final static String TEXT_TOP = "top";
	private final static String TEXT_BOTTOM = "bottom";
	private final static String ATTRIBUTE_POSITION = "position";
	
	private static TraceExtensionUIElementPos convertTraceExtensionUIElementPos(String str) {
		if(str == null) return null;
		
		str = str.trim();
		
		if(str.equalsIgnoreCase(TEXT_TOP)) return TraceExtensionUIElementPos.TOP;
		if(str.equalsIgnoreCase(TEXT_BOTTOM)) return TraceExtensionUIElementPos.BOTTOM;

		return null;
		
	}
	
	/** Provides extensions the parent composite of the profiling tab, the current launch configuration,
	 * and a reference to the profile tab from which the composite is coming from. (Bug 323330) */
	public static void callExtensionUIElements(Composite composite, ILaunchConfiguration launchConf, ProfileTab profileTab, TraceExtensionUIElementPos position) {

		IConfigurationElement[] config = Platform.getExtensionRegistry()
			.getConfigurationElementsFor(
					UIPlugin.PLUGIN_ID+".profilingTabUIAddition");

		for (IConfigurationElement e : config) {
			
			String extensionPosition = e.getAttribute(ATTRIBUTE_POSITION);
			TraceExtensionUIElementPos extPosConst = convertTraceExtensionUIElementPos(extensionPosition);

			// Ignore unmatched text
			if(extPosConst == null) continue;

			// Ignore text that doesn't match the current position of the calling method
			if(extPosConst != position) continue;
			
			try {
				final Object o = e.createExecutableExtension("class");
				if (o instanceof IProfilerUIAdditionExtension) {
					IProfilerUIAdditionExtension i = (IProfilerUIAdditionExtension)o;
					i.addExtensionUIElements(composite, launchConf, profileTab);
				}
			} catch (CoreException e1) {
				e1.printStackTrace();
			}
		}

	}

	/** Provides the list of tabs from the particular profiling dialog tab group, and the class
	 * from which the extension is being called. (Bug 323330) */
	public static void addOrInformExtensionTabs(List<ILaunchConfigurationTab> l, Class src) {
		
		IConfigurationElement[] config = Platform.getExtensionRegistry()
			.getConfigurationElementsFor(UIPlugin.PLUGIN_ID+".profilingTabList");
		
		for(IConfigurationElement e : config) {
			try {
				final Object o = e.createExecutableExtension("class");
				if(o instanceof IProfilerTabExtension) {
					IProfilerTabExtension i = (IProfilerTabExtension)o;
					i.updateOrAddTabs(l, src);
				}
			} catch (CoreException e1) {
				e1.printStackTrace();
			}
		}
	}

	/** Provides the extension with the launch configuration and working sets, which contain
	 * the filter and filter settings to be processed. (Bug 323330) */
	public static void applyExtensionFilterProcessing(ILaunchConfigurationWorkingCopy launchConf, IWorkingSet[] wsArr) {

		IConfigurationElement[] config = Platform.getExtensionRegistry().getConfigurationElementsFor(UIPlugin.PLUGIN_ID+".profilingFilterProcessor");

		for (IConfigurationElement e : config) {

			try {
				final Object o = e.createExecutableExtension("class");
				
				if (o instanceof IProfilerFilterProcessorExtension) {
					IProfilerFilterProcessorExtension i = (IProfilerFilterProcessorExtension)o;
					i.processFilter(launchConf, wsArr);
				}
			} catch (CoreException e1) {
				e1.printStackTrace();
			}
		}

	}

}
