/**********************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ProfileLaunchUtil.java,v 1.8 2007/07/11 14:27:06 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.launcher;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.launcher.AttachAgentTabProviders.AgentTreeItem;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSet;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSetType;
import org.eclipse.hyades.trace.ui.launcher.IProfilingType;
import org.eclipse.hyades.trace.ui.launcher.ProfilingAttribute;
import org.eclipse.hyades.trace.ui.launcher.ProfilingSetsManagerCopy;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.tptp.platform.common.internal.CommonPlugin;
import org.eclipse.tptp.trace.ui.internal.launcher.core.AnalysisType;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherConstants;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LightConfigurationLoader;
import org.eclipse.tptp.trace.ui.internal.launcher.core.DataCollectorTreeContentProvider.ParentChildNode;
import org.eclipse.tptp.trace.ui.provisional.launcher.IConfiguration;
import org.eclipse.tptp.trace.ui.provisional.launcher.IConfigurationPage;
import org.eclipse.tptp.trace.ui.provisional.launcher.ILaunchValidator;
import org.eclipse.tptp.trace.ui.provisional.launcher.ILightConfiguration;

/**
 * Utility methods shared by the profile launchers.
 */
public class ProfileLaunchUtil {
	
	private static final String[] EMPTY = new String[0];
	
	/**
	 * Retrieves the environment variables from the given launch
	 * configuration, and returns them in the form of Strings
	 * in the form "name=value".
	 */
	public static String[] getEnvironmentVariables(ILaunchConfiguration conf) throws CoreException {
		Map envMap = conf.getAttribute(ILaunchManager.ATTR_ENVIRONMENT_VARIABLES, (Map)null);
		if (envMap != null) {
			List list = new ArrayList();
			Iterator iter = envMap.keySet().iterator();
			while (iter.hasNext()) {
				String key = (String)iter.next();
				list.add(key + "=" + envMap.get(key));
			}
			String[] array = new String[list.size()];
			list.toArray(array);
			return array;
		}
		else {
			return EMPTY;
		}
	}
	
	/**
	 * Notifies this configuration's profiling types that the
	 * launch is about to begin.
	 */
	public static boolean performProfilingTypesLaunch(ILaunchConfiguration config) {

		ProfilingSetsManager manager = ProfilingSetsManager.instance();
		IProfilingSet set = null;
		try {
			String setID = config.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_SET, (String)null);
			set = (IProfilingSet)manager.getProfilingSets().get(setID);
		}
		catch (CoreException e) {
			CommonPlugin.logError(e);
		}

		if (set == null) {
			return true;
		}

		IProfilingSetType[] profilingTypes = new IProfilingSetType[manager.getProfilingTypes().size()];
		profilingTypes = (IProfilingSetType[]) manager.getProfilingTypes().values().toArray(profilingTypes);

		List types = set.getProfilingTypes();
		for (int i = 0; i < profilingTypes.length; i++) {
			if (types.contains(profilingTypes[i].getId())) {
				IProfilingType type = profilingTypes[i].getProfilingType();
				final String msg = type.launch(config);
				if (msg != null) {
					Display.getDefault().asyncExec(new Runnable() {
						public void run() {
							MessageDialog.openError(Display.getDefault().getActiveShell(), UIPlugin.getResourceString("ERROR_DLG_CONFLICTS_TITLE"), msg);
						}
					});
					return false;
				}
			}
		}
		return true;
	}

	/**
	 * Converts the given array into a delimited String with
	 * the given delimiter character.
	 */
	public static String convertToDelimitedString(String[] array, char delimiter) {
		return convertToDelimitedString(array, delimiter, true);
	}	
	
	/**
	 * Converts the given array into a delimited String with
	 * the given delimiter character.
	 */
	public static String convertToDelimitedString(String[] array, char delimiter, boolean addQuotes) {
		StringBuffer buf = new StringBuffer();
		if (array.length > 0) {
			for (int i=0;i<array.length;++i) {
				if (i != 0) {
					buf.append(delimiter);
				}

				if (array[i].startsWith("\"") && array[i].endsWith("\"")) {
					buf.append(array[i]);
				}
				else if (!addQuotes || array[i].indexOf(' ') == -1) {
					buf.append(array[i]);
				}
				else {
					buf.append('"');
					buf.append(array[i]);
					buf.append('"');
				}
			}
		}
		return buf.toString();
	}
	
	
	/**
	 * In order to support backward compatibility and to make the transition easier, we'll
	 * need to build a dummy profiling set and add it to the configuration 
	 *
	 * @param wc The working copy of the launch configuration
	 */
	public static void createDummyProfilingSet(ProfilingSetsManagerCopy profilingSetManager, ILaunchConfigurationWorkingCopy wc, Object[] selectedElements)
	{		
		Object childNode;
		AnalysisType analysisType;
		ILaunchValidator analysisTypeSupp;		 
		List optionsList = null;

		/* A good opportunity to clean off the dummy profiling sets that are no longer in use */
		String configurationName = wc.getName();
		cleanProfilingSets(profilingSetManager, configurationName);
		
		/* Create the dummy profiling set */
		final String dummyProfilingSetID = LauncherConstants.DUMMY_PROFILING_SET_PREFIX + configurationName;
		ProfilingSet dummyProfilingSet = (ProfilingSet) profilingSetManager.getProfilingSets().get(dummyProfilingSetID);
		if (dummyProfilingSet == null)
			dummyProfilingSet = new ProfilingSet(dummyProfilingSetID, "DummyProfilingSet", "Programmatically Created");

		dummyProfilingSet.getProfilingTypes().clear();
		optionsList = LauncherUtility.getDefaultProfilingOptions();
		
		Hashtable addedAttributes = new Hashtable();
		
		/* The configurations */
		ArrayList configurations = new ArrayList();
		
		/* Traverse through the selected elements */
		ParentChildNode currentParentChildItem = null;
		AgentTreeItem currentAgentTreeItem = null;
		for (int i = 0; i < selectedElements.length; i++)
		{									
			if (selectedElements[i] instanceof ParentChildNode)
			{
				currentParentChildItem = (ParentChildNode) selectedElements[i];
			}
			else if (selectedElements[i] instanceof AgentTreeItem)
			{
				currentAgentTreeItem = (AgentTreeItem) selectedElements[i];
			}
			else
			{
				continue;
			}
				
			childNode = currentAgentTreeItem == null ? currentParentChildItem.child : 
						currentAgentTreeItem.getAnalysisType() == null ? currentAgentTreeItem.getProfilingType() : currentAgentTreeItem.getAnalysisType();
			if (childNode instanceof AnalysisType)
			{
				analysisType = (AnalysisType)childNode;
				
				/* Add the analysis type */					
				dummyProfilingSet.getProfilingTypes().add(analysisType.getId());
				
				
				/* Collect the attributes of the analysis type */				
				LightConfigurationLoader lightConfigurationLoader = analysisType.getConfigurationLoader();
				
				if (lightConfigurationLoader != null)
				{
					ILightConfiguration lightConfigClass = lightConfigurationLoader.getConfigurationClass();
					
					/* To retrieve the attributes of the analysis types, we'll first need to 
					 * reset each page with the right launch configuration if the configuration is editable. */
					if (lightConfigClass != null)
					{						
						if (lightConfigurationLoader.isEditable())
						{
							IConfiguration configuration = (IConfiguration)lightConfigClass;
							IConfigurationPage[] configurationPages = configuration.getConfigurationPages();
							configurations.add(configuration);
							for (int j = 0; j < configurationPages.length; configurationPages[j++].reset(wc.getOriginal()));		
						}
						
						ProfilingAttribute[] attributes = lightConfigClass.getAttributes();
						if (attributes != null)
						{
							for (int j = 0; j < attributes.length; j++)
							{
								optionsList.add(attributes[j]);
								addedAttributes.put(attributes[j].getName(), Boolean.TRUE);
							}
								
						}
					}
				}							

			}
			
			/* For backward compatibility */
			else if (childNode instanceof IProfilingSetType)
			{
				IProfilingSetType profilingSetType = (IProfilingSetType)childNode;
				dummyProfilingSet.getProfilingTypes().add(profilingSetType.getId());
				IProfilingType profilingType = profilingSetType.getProfilingType();
				
				/* Profiling types expect their controls to be created prior to the attributes */
				Control control = null;
				ProfilingAttribute[] attributes = null;
				try
				{
					attributes = profilingType.getAttributes();
				}
				catch (Exception e)
				{
					control = org.eclipse.hyades.trace.ui.internal.core.ProfileLaunchUtil.createProfilingTypeControls(profilingType, profilingSetManager);
					attributes = profilingType.getAttributes();
				}
			
				
				/* Augment attributes with corresponding contributor id and add it to the option's list */
				for (int j = 0; j < attributes.length; j++)
				{
					attributes[j].setContributorId(profilingSetType.getId());
					optionsList.add(attributes[j]);
					addedAttributes.put(attributes[j].getName(), Boolean.TRUE);
				}		
				
				if (control != null)
					org.eclipse.hyades.trace.ui.internal.core.ProfileLaunchUtil.destroyControl(control);
			}							
		}
		
		/* Remove the attributes that weren't explicitly added by profiling/analysis types */
		Map currentAttributes = dummyProfilingSet.getAttributes();
		Object[] keys = currentAttributes.keySet().toArray();
		for (int i = 0; i < keys.length; i++)
		{
			Object currentKey = keys[i];
			if (addedAttributes.get(currentKey) == null)
				currentAttributes.remove(currentKey);
		}
		
		/* remove the duplicate entries (with precedence) */
		optionsList = LauncherUtility.filterDuplicateOptions(optionsList);
		
		/* apply attributes to the manager copy */		
		profilingSetManager.getProfilingSets().put(dummyProfilingSetID, dummyProfilingSet);
		profilingSetManager.setDefaultSet(dummyProfilingSet);

		/* Disable all attributes of the profiling set, except for the ones corresponding to 
		 * entries that are selected */
		Map attrs = dummyProfilingSet.getAttributes();
		for (Iterator attrIterator = attrs.values().iterator(); attrIterator.hasNext();)
		{
			ProfilingAttribute profilingAttribute = (ProfilingAttribute)attrIterator.next();
			profilingAttribute.setEnabled(false);
		}
		
		/* Add the options of the selected entries */
		for(int idx=0; idx<optionsList.size(); idx++)
		{
			ProfilingAttribute option = (ProfilingAttribute) optionsList.get(idx);
			attrs.put(option.getName(), option);
		}
		
		wc.setAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_SET, dummyProfilingSetID);		
		profilingSetManager.applyManagerChanges();		
		profilingSetManager.writeProfilingSetsToPreference();
	}
	
	
	/**
	 * Removes all dummy profiling sets that are not in use except for
	 * the one that is associated with the configuration name passed in.
	 * 
	 * @param configurationName A configuration name whose associated dummy
	 * profiling set will be preserved.
	 */
	private static void cleanProfilingSets(ProfilingSetsManagerCopy profilingSetManager, String configurationName)
	{
		/* Get all profiling sets */
		Map profilingSets = profilingSetManager.getProfilingSets();
		Object[] profilingSetIds = profilingSets.keySet().toArray();
		Hashtable launchConfigurationsMap = new Hashtable();
		int dummyPSLength = LauncherConstants.DUMMY_PROFILING_SET_PREFIX.length();
		
		/* Store the launch configurations */
		try
		{
			ILaunchConfiguration[] launchConfigurations = DebugPlugin.getDefault().getLaunchManager().getLaunchConfigurations();
			for (int i = 0; i < launchConfigurations.length; i++)
				launchConfigurationsMap.put(launchConfigurations[i].getName(), launchConfigurations[i]);
			
		} catch (CoreException e)
		{			
			return;
		}
		
		
		for (int i = 0; i < profilingSetIds.length; i++)
		{
			String profilingSetId = (String)profilingSetIds[i];
			
			
			if (profilingSetId.startsWith(LauncherConstants.DUMMY_PROFILING_SET_PREFIX) &&							/* Is this a dummy profiling set? */
				!profilingSetId.equals(LauncherConstants.DUMMY_PROFILING_SET_PREFIX + configurationName) &&			/* It doesn't correspond to the configuration name passed in */
				launchConfigurationsMap.get(profilingSetId.substring(dummyPSLength)) == null)		/* There is no corresponding launch configuration */
				
				profilingSets.remove(profilingSetId);			
		}		
	}
}
