/**********************************************************************
 * Copyright (c) 2005, 2006 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ProfilingSetsManager.java,v 1.13 2006/04/08 19:11:51 amehregani Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.launcher;

import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.hyades.internal.execution.local.common.Options;
import org.eclipse.hyades.internal.execution.local.control.AgentConfigurationEntry;
import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.core.TraceFilterManager;
import org.eclipse.hyades.trace.ui.internal.util.FilterSetElement;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSet;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSetType;
import org.eclipse.hyades.trace.ui.launcher.IProfilingType;
import org.eclipse.hyades.trace.ui.launcher.ProfilingAttribute;
import org.eclipse.hyades.ui.internal.util.XMLUtil;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.tptp.trace.ui.internal.launcher.core.AnalysisTypeManager;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherConstants;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * @author popescu
 */
public class ProfilingSetsManager {
	
	protected static ProfilingSetsManager instance;
	protected Map profilingSets;
	
	/** @deprecated */
	protected Map profilingTypesGroups;
	
	protected Map profilingTypes;
	
	protected IProfilingSet defaultSet = null;
	protected FilterSetElement defaultFilterSet = null;
	
	/** @deprecated */
	protected final String profilingTypesGroupsExtPoint = "profilingTypesGroups";
	/** @deprecated */
	protected final String profilingTypesExtPoint = "profilingTypes";
	/** @deprecated */
	protected final String profilingSetsExtPoint = "profilingSets";
	
	protected final String profilingSetsStore = "profilingTypeSets";
	protected final String defaultProfilingSet = "defaultProfilingSet";
	
	
	private boolean isFilterInitialized;
	protected ProfilingSetsManager()
	{		
	}
	
	public static ProfilingSetsManager instance()
	{
		if(instance == null)
			 instance = new ProfilingSetsManager();
		
		return instance;
	}
	
	/** @deprecated */
	public Map getProfilingTypesGroups()
	{
		if(profilingTypesGroups == null)
			initialize();
		
		return profilingTypesGroups;
	}

	public Map getProfilingTypes()
	{
		if(profilingTypes == null)
			initialize();
		
		return profilingTypes;
	}

	public Map getProfilingSets()
	{
		if(profilingSets == null)
			initialize();
		
		return profilingSets;
	}
	
	protected void initialize()
	{
		initializeProfilingTypesGroups();
		initializeProfilingTypes();
		initializeProfilingSets();
		initializeProfilingFilters();		
	}
	
	
	/** @deprecated */
	protected void initializeProfilingTypesGroups()
	{
		profilingTypesGroups = new HashMap();
		
		IExtensionPoint point = Platform.getExtensionRegistry().getExtensionPoint(UIPlugin.PLUGIN_ID, profilingTypesGroupsExtPoint);

		if (point != null) {
			IConfigurationElement[] elements = point.getConfigurationElements();
			
			for (int i = 0; i < elements.length; i++) {
				
				IConfigurationElement elem = elements[i];

				String name = elem.getAttribute("name");				
				String id = elem.getAttribute("id");
				String iconPath = elem.getAttribute("icon");
				String description = UIPlugin.getResourceString(TraceMessages.DPRFS_D);
				//check if description attribute exists
				String[] attributes = elem.getAttributeNames();
				for (int j = 0; j < attributes.length; j++)
				{
					if (attributes[j].equalsIgnoreCase("description"))
					{
						//if the description provided is not "", then populate the string
						String temp = elem.getAttribute("description");
						if (temp.length() > 0) description = temp;
					}
				}
				if(id == null || name == null)
					 continue;

				ImageDescriptor icon = null;
				
				if(iconPath != null)
				{
					try {
						URL pluginInstallUrl = Platform.getBundle(elem.getDeclaringExtension().getNamespace()).getEntry("/"); //$NON-NLS-1$			
						
						icon = ImageDescriptor.createFromURL(new URL(pluginInstallUrl + iconPath));
					} catch (Exception e) {
						e.printStackTrace();
					}					
				}
				
				profilingTypesGroups.put(id, new ProfilingSetTypeGroup(id, name, description, icon));
			}		
		}		
	}
	
	/** @deprecated */
	protected void initializeProfilingTypes()
	{
		profilingTypes = new HashMap();
		
		IExtensionPoint point = Platform.getExtensionRegistry().getExtensionPoint(UIPlugin.PLUGIN_ID, profilingTypesExtPoint);

		if (point != null) {
			IConfigurationElement[] elements = point.getConfigurationElements();
			
			for (int i = 0; i < elements.length; i++) {
				
				IConfigurationElement elem = elements[i];

				String id = elem.getAttribute("id");				
				String name = elem.getAttribute("name");				
				
				if(id == null || name == null)
					 continue;

				String iconPath = elem.getAttribute("icon");				
				ImageDescriptor icon = null;
				if(iconPath != null)
				{
					try {
						
						URL pluginInstallUrl = Platform.getBundle(elem.getDeclaringExtension().getNamespace()).getEntry("/"); //$NON-NLS-1$									
						icon = ImageDescriptor.createFromURL(new URL(pluginInstallUrl + iconPath));
						
					} catch (Exception e) {
						e.printStackTrace();
					}					
				}
				
				String description = elem.getAttribute("description");
				String group = elem.getAttribute("group");
				boolean useFilter = elem.getAttribute("useFilters") == null ? false : elem.getAttribute("useFilters").equalsIgnoreCase("true");				
				
				IProfilingType profileType = null;
				try
				{
					profileType = (IProfilingType)elem.createExecutableExtension("className");
				}
				catch (Exception e)
				{
					e.printStackTrace();
				}
				
				if(profileType != null)			
				{
					IProfilingSetType pTypes = new ProfilingSetType(id, name, description, icon
							  , useFilter, group, profileType);
					
					profilingTypes.put(id, pTypes);					
				}
			}
			
		}
		
		
		/* Ali M.: To be able to support backward compatibility with the new framework in place,
		 * we'll need to also treat analysis types as profiling types. */
		profilingTypes.putAll(AnalysisTypeManager.getInstance().getAnalysisTypes());		
	}

	protected void initializeProfilingSets()
	{
		profilingSets = new HashMap();

		if (!readSetsFromPreferences())
			readSetsFromRegistry();
		
		setDefaultSet(getInitialDefaultSet(null));
	}
		
	/**
	 * Add profiling sets defined by extension points
	 *
	 */
	protected void readSetsFromRegistry()
	{		
		IExtensionPoint point = Platform.getExtensionRegistry().getExtensionPoint(UIPlugin.PLUGIN_ID, profilingSetsExtPoint);

		if (point != null) {
			IConfigurationElement[] elements = point.getConfigurationElements();
			
			for (int i = 0; i < elements.length; i++) {
				
				IConfigurationElement elem = elements[i];
				String id = elem.getAttribute("id");				
				
				if(id == null)
					 continue;
				
				IProfilingSet pSets = new ProfilingSet(id, elem.getAttribute("name"), elem.getAttribute("description"));
				
				//read profiling types
				IConfigurationElement[] types = elem.getChildren("profilingTypes");
				for(int k=0; k<types.length; k++)
				{
					IConfigurationElement[] aelements = types[k].getChildren("profilingType");	
					for(int ak=0; ak<aelements.length; ak++)
					{
						IConfigurationElement aelem = aelements[ak];
						String aid = aelem.getAttribute("id");
						
						if(aid == null)
							continue;
						
						pSets.getProfilingTypes().add(aid);
					}
				}
				
				
				//read attributes
				IConfigurationElement[] attributes = elem.getChildren("attributes");
				List attributeList = getDefaultProfilingOptions();
				for(int k=0; k<attributes.length; k++)
				{
					IConfigurationElement[] aelements = attributes[k].getChildren("attribute");	
					for(int ak=0; ak<aelements.length; ak++)
					{
						IConfigurationElement aelem = aelements[ak];
						String type = aelem.getAttribute("type");
						String aname = aelem.getAttribute("name");
						String value = aelem.getAttribute("value");
						
						if(type == null || aname == null || value == null)
							continue;
						
						if ("ATTR".equals(type)) {
							aname = UIPlugin.getPluginId() + ".ATTR_" + aname;
						}
						else {
							aname = (new StringBuffer(type)).append("_").append(aname).toString();
						}

						ProfilingAttribute option = new ProfilingAttribute(aname, value);
						attributeList.add(option);
					}
				}

				// remove duplicates and create a map
				Map map = pSets.getAttributes();
				Iterator iter = filterDuplicateOptions(attributeList).iterator();
				while (iter.hasNext()) {
					ProfilingAttribute option = (ProfilingAttribute)iter.next();
					map.put(option.getName(), option);
				}
				
				profilingSets.put(id, pSets);				
			}
		}
		
	}

	/**
	 * Read profiling sets stored in preferences
	 *
	 */
	protected boolean readSetsFromPreferences()
	{
		IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();
		String profilingSetsStr = store.getString(profilingSetsStore);		
				
		if(profilingSetsStr == null || profilingSetsStr.length() == 0)
			 return false;
		
		return readSets(profilingSetsStr);
		
	}

	protected boolean readSets(String profilingSetsStr)
	{
		if(profilingSetsStr == null || profilingSetsStr.length() == 0)
			 return false;
		
		Element prefElement = XMLUtil.loadDom(profilingSetsStr, "profilingSets");
		
		NodeList profilingSetsNodeList = XMLUtil.getChildrenByName(prefElement, "profilingSet");
		Element element = null;
		for(int i=0; i < profilingSetsNodeList.getLength();i++)
		{
			if(profilingSetsNodeList.item(i) instanceof Element)
				element = (Element)profilingSetsNodeList.item(i);
				String name = element.hasAttribute("name") ? element.getAttribute("name") : null;
				String description = element.hasAttribute("description") ?  element.getAttribute("description") : null;
				String id=element.getAttribute("id");
				
				if(id == null)
				  continue;
				
				IProfilingSet set;				
				Object obj = profilingSets.get(id);
				if(obj == null)
				{
					//a profiling set not defined by registry
					//create a new one
					set = new ProfilingSet(id, name, description);					
					profilingSets.put(id, set);
				}
				else
				{
					set = (IProfilingSet)obj;
					set.setName(name);
					set.setDescription(description);
					
					set.getAttributes().clear();
					set.getProfilingTypes().clear();
				}
				
				//set attributes and profiling types
				NodeList types = XMLUtil.getChildrenByName(element, "profilingTypes");
				for(int j=0; j<types.getLength(); j++)
				{
					Element atype = (Element)types.item(j);
					NodeList atypes = XMLUtil.getChildrenByName(atype, "profilingType");					
					for(int k=0; k<atypes.getLength(); k++)
					{
						Element atypeElem = (Element) atypes.item(k);
						String atypeId = atypeElem.getAttribute("name");
						
						if(atypeId != null && atypeId.length() > 0)
							  set.getProfilingTypes().add(atypeId);
					}
				}
				
				List attributeList = getDefaultProfilingOptions();
				NodeList attributes = XMLUtil.getChildrenByName(element, "attributes");
				for(int j=0; j<attributes.getLength(); j++)
				{
					Element atype = (Element)attributes.item(j);
					NodeList atypes = XMLUtil.getChildrenByName(atype, "attribute");					
					for(int k=0; k<atypes.getLength(); k++)
					{
						Element atypeElem = (Element) atypes.item(k);
						String atypeId = atypeElem.getAttribute("type");						
						String aname = atypeElem.getAttribute("name");
						String value = atypeElem.getAttribute("value");
						
						if(atypeId.length() > 0)							
							aname = (new StringBuffer(atypeId)).append("_").append(aname).toString();
							
						ProfilingAttribute option = new ProfilingAttribute(aname, value);
						attributeList.add(option);
					}
				}
				
				Iterator iter = filterDuplicateOptions(attributeList).iterator();
				while (iter.hasNext()) {
					ProfilingAttribute option = (ProfilingAttribute)iter.next();
					set.getAttributes().put(option.getName(), option);
				}
		}
				
		return true;
	}
	
	public IProfilingSet getInitialDefaultSet(String toBeRemovedProfileSetId)
	{
		IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();
		String defaultSet = store.getString(defaultProfilingSet);
		
		boolean defaultSetToBeRemoved = toBeRemovedProfileSetId != null && toBeRemovedProfileSetId.equals(defaultSet);
 
		if(defaultSet != null && !defaultSetToBeRemoved)
		{
			Object set = getProfilingSets().get(defaultSet);
			if(set != null)
			{
				return (IProfilingSet)set;
			}
		}
		
		defaultSetToBeRemoved = toBeRemovedProfileSetId != null && toBeRemovedProfileSetId.equals(this.getDefaultProfilingSet());
		
		if(profilingSets.size() > 0 && !defaultSetToBeRemoved)
		{
			Object set = getProfilingSets().get(this.getDefaultProfilingSet());
			if(set != null)
			{
				return (IProfilingSet)set;
			}
		}		
		
		Object[] sets = profilingSets.values().toArray();
		IProfilingSet currentSet = null;
		for (int i = 0; i < sets.length; i++)
		{
			currentSet = (IProfilingSet) sets[i];
			defaultSetToBeRemoved = toBeRemovedProfileSetId != null && toBeRemovedProfileSetId.equals(currentSet.getId());				
			if (!defaultSetToBeRemoved)
				return currentSet;
		}

		return null; //should never happen
	}

	

	protected void readFilters(String profilingSetsStr)
	{
		
	}
	
	/**
	 * Write profiling sets to preferences
	 *
	 */
	public void writeSetsToPreferences()
	{				
		IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();				
		store.setValue(TraceConstants.FILTERS_SET_KEY, getProfilingFiltersString().toString());
		
		writeProfilingSetsToPreference();
	}
	
	public void writeProfilingSetsToPreference()
	{
		IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();		
		store.setValue(profilingSetsStore, getProfilingSetsString().toString());
				
		IProfilingSet defaultInstanceSet = ProfilingSetsManager.instance().getDefaultSet(); 
		if(defaultInstanceSet != null)
		{
			store.setValue(defaultProfilingSet, defaultInstanceSet.getId());
		}
	}
	
	protected StringBuffer getProfilingSetsString()
	{
		StringBuffer xmlString = new StringBuffer();
		
		xmlString.append("<?xml");
		xmlString.append(XMLUtil.createXMLAttribute("version","1.0", false)).append(XMLUtil.createXMLAttribute("encoding","UTF-8", false));
		xmlString.append("?>");

		xmlString.append("<profilingSets>");

		//append profilingSet element
		Iterator sets = getProfilingSets().keySet().iterator();
		while(sets.hasNext())
		{
			Object key = sets.next();
			IProfilingSet elem = (IProfilingSet) profilingSets.get(key);
			
			xmlString.append("<profilingSet");
			xmlString.append(XMLUtil.createXMLAttribute("id", elem.getId(), false));
			if (elem.getName() != null) {
				xmlString.append(XMLUtil.createXMLAttribute("name", elem.getName(), false));			
			}
			if (elem.getDescription() != null) {
				xmlString.append(XMLUtil.createXMLAttribute("description", elem.getDescription(), false));
			}
			xmlString.append(">");
			
			//add Profiling Types
			xmlString.append("<profilingTypes>");	
			List types = elem.getProfilingTypes();
			for (int i = 0; i < types.size(); i++)
			{
				xmlString.append("<profilingType");
				xmlString.append(XMLUtil.createXMLAttribute("name", types.get(i).toString(), false));
				xmlString.append("/>");
			}
			xmlString.append("</profilingTypes>");
			
			//store the attributes
			xmlString.append("<attributes>");	
			Iterator attributes = elem.getAttributes().keySet().iterator();
			while(attributes.hasNext())
			{
				Object akey = attributes.next();
				ProfilingAttribute option = (ProfilingAttribute)elem.getAttributes().get(akey);
				xmlString.append("<attribute");
				xmlString.append(XMLUtil.createXMLAttribute("name", option.getName(), false));
				xmlString.append(XMLUtil.createXMLAttribute("value", option.getValue(), false));				
				xmlString.append("/>");
				
			}
			xmlString.append("</attributes>");
			
			xmlString.append("</profilingSet>");			
		}
		
		xmlString.append("</profilingSets>");
		
		return xmlString;
		
	}

	protected StringBuffer getProfilingFiltersString()
	{
		return new StringBuffer(TraceFilterManager.getInstance().serializeFiltersToXML());		
	}
	
	public IProfilingSetType[] getProfilingTypesForGroup(String groupId)
	{
		if(groupId == null)
			 return new IProfilingSetType[0];
		
		ArrayList list = new ArrayList();
		Iterator profileType = getProfilingTypes().values().iterator();
		while(profileType.hasNext())
		{
			IProfilingSetType profType = (IProfilingSetType) profileType.next();
			if(profType.getGroup().equals(groupId))
				list.add(profType);
				
		}
		
		IProfilingSetType[] array = new IProfilingSetType[list.size()];
		return (IProfilingSetType[])list.toArray(array);
		
	}
	/**
	 * @return Returns the defaultSet.
	 */
	public IProfilingSet getDefaultSet() {
		
		if(profilingSets == null)
			initialize();
		
		return defaultSet;
	}
	/**
	 * @param defaultSet The defaultSet to set.
	 */
	public void setDefaultSet(IProfilingSet defaultSet) {
		this.defaultSet = defaultSet;
	}
	
	protected void initializeProfilingFilters()
	{
		isFilterInitialized = true;
		Map filterSets = TraceFilterManager.getInstance().getFilterSet();
				
		if(getDefaultFilterSet() == null && filterSets.size() > 0)
		{
			setDefaultFilterSet(TraceFilterManager.getInstance().getFilterSetElement(LauncherConstants.DEFAULT_FILTER_ID));
			if (getDefaultFilterSet() == null) 
			{
				Object[] sets = filterSets.values().toArray();
				setDefaultFilterSet((FilterSetElement)sets[0]);
			}
		}		
	}
	
	/**
	 * @return Returns the filterSets.
	 */
	public Map getFilterSets() {
		
		return TraceFilterManager.getInstance().getFilterSet();
	}
	/**
	 * @param filterSets The filterSets to set.
	 */
	public void setFilterSets(Map filterSets) {
		TraceFilterManager.getInstance().storeFilterSetList(new ArrayList(filterSets.values()));
	}
	/**
	 * @return Returns the defaultFilterSet.
	 */
	public FilterSetElement getDefaultFilterSet() {

		if (!isFilterInitialized)
			initializeProfilingFilters();
		
		return defaultFilterSet;
	}
	/**
	 * @param defaultFilterSet The defaultFilterSet to set.
	 */
	public void setDefaultFilterSet(FilterSetElement defaultFilterSet) {
		this.defaultFilterSet = defaultFilterSet;
	}
	
	public Vector getLimitOptions(ILaunchConfiguration conf) throws CoreException
	{
		IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();
		Vector options = new Vector();

		boolean limitInvocations = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_LIMIT_INVOCATIONS, store.getBoolean(TraceConstants.LIMIT_TRACE_INVOC_OPTION));
		int invocations = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_LIMIT_INVOCATIONS_NUM, store.getInt(TraceConstants.LIMIT_INVOC_NB));
		boolean limitTime = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_LIMIT_SECONDS, store.getBoolean(TraceConstants.LIMIT_TRACE_TIME_OPTION));
		int seconds = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_LIMIT_SECONDS_NUM, store.getInt(TraceConstants.LIMIT_TIME_NB));

		//time,invocations limited events
		if (limitInvocations)
		{
			AgentConfigurationEntry entry = new AgentConfigurationEntry();
			entry.setName(Options.OPTION_BURST_MODE);
			entry.setType("SETOPTION");

			if (limitTime)
			{
				entry.setValue(Options.OPTION_VALUE_SECONDSANDINVOCATIONS);

				AgentConfigurationEntry entry1 = new AgentConfigurationEntry();
				entry1.setName(Options.OPTION_BURST_SECONDS);
				entry1.setValue(String.valueOf(seconds));
				entry1.setType("SETOPTION");

				options.addElement(entry1);
			}
			else
			{
				entry.setValue(Options.OPTION_VALUE_INVOCATIONS);
			}

			options.addElement(entry);

			AgentConfigurationEntry entry1 = new AgentConfigurationEntry();
			entry1.setName(Options.OPTION_BURST_INVOCATIONS);
			entry1.setValue(String.valueOf(invocations));
			entry1.setType("SETOPTION");

			options.addElement(entry1);

		}
		else if (limitTime)
		{
			AgentConfigurationEntry entry = new AgentConfigurationEntry();
			entry.setName(Options.OPTION_BURST_MODE);
			entry.setType("SETOPTION");
			entry.setValue(Options.OPTION_VALUE_SECONDS);

			options.addElement(entry);

			entry = new AgentConfigurationEntry();
			entry.setName(Options.OPTION_BURST_SECONDS);
			entry.setValue(String.valueOf(seconds));
			entry.setType("SETOPTION");

			options.addElement(entry);
		}

		return options;
	}

	public Vector getOptions(ILaunchConfiguration conf) throws CoreException {
		Vector options = getLimitOptions(conf);
		String profilingSetID = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_SET, (String)null);
		IProfilingSet set = (IProfilingSet)getProfilingSets().get(profilingSetID);
		if (set != null) {
			options.addAll(getProfileOptions(set));
		}
		
		AgentConfigurationEntry entry = new AgentConfigurationEntry();
		entry.setName(IProfileLaunchConfigurationConstants.ATTR_FILTER_SET);
		entry.setValue(conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_FILTER_SET, (String)null));
		entry.setType("SETOPTION");
		options.addElement(entry);

		return options;
	}

	public Vector getProfileOptions(IProfilingSet set) throws CoreException	{
		Vector options = new Vector();
		Map attributes = set.getAttributes();
		Iterator attrNames = attributes.keySet().iterator();
		String prefixName = "SETOPTION_";
		int numRACoptions = 0;
		ProfilingAttribute currentProfileAttribute;
		while(attrNames.hasNext())
		{
			String name = attrNames.next().toString();
			currentProfileAttribute = (ProfilingAttribute)attributes.get(name);
			
			/* If in case the current profile attribute is disabled, then skip it */
			if (!currentProfileAttribute.isEnabled())
				continue;
			
			if(name.startsWith(prefixName))
			{
				//profiling option, pass it to the RAC
				AgentConfigurationEntry entry = new AgentConfigurationEntry();
				entry.setName(name.substring(prefixName.length()));
				entry.setValue(currentProfileAttribute.getValue());
				entry.setType("SETOPTION");
				options.addElement(entry);
				numRACoptions++;
			}
			else
			{
				if(!filterOption(name))
				{
					//profiling option, pass it to the RAC
					AgentConfigurationEntry entry = new AgentConfigurationEntry();
					entry.setName(name);
					entry.setValue(currentProfileAttribute.getValue());
					entry.setType("SETOPTION");
					options.addElement(entry);
				}				
			}
		}

		if (numRACoptions == 0)
		{
			//means that we haven't set any profiling options to the RAC
			//send the default RAC options
			
			IProfilingSet defaultSet = getDefaultSet();
   		 	Iterator defaultOptions = defaultSet.getAttributes().values().iterator();
 
			while (defaultOptions.hasNext())
			{
				ProfilingAttribute attr = (ProfilingAttribute)defaultOptions.next();
				if (attr.getName().startsWith(prefixName))
				{
					AgentConfigurationEntry entry = new AgentConfigurationEntry();
					entry.setName(attr.getName().substring(prefixName.length()));
					entry.setValue(attr.getValue());
					entry.setType("SETOPTION");
					options.addElement(entry);
				}

			}
		}
		
		// add profiling types for history
		List types = set.getProfilingTypes();
		if (types != null) {
			Collections.sort(types, new Comparator() {
				Map map = getProfilingTypes();
				public int compare(Object o1, Object o2) {
					return ((IProfilingSetType)map.get(o1)).getName().compareTo(((IProfilingSetType)map.get(o2)).getName());
				}
			});
			for (int i=0;i<types.size();++i) {
				AgentConfigurationEntry entry = new AgentConfigurationEntry();
				entry.setName(UIPlugin.getPluginId() + ".type" + i);
				entry.setValue((String)types.get(i));
				entry.setType("SETOPTION");
				options.addElement(entry);
			}
		}
		
		return options;
	}
	
	protected boolean filterOption(String name)
	{
		return (name.equals(IProfileLaunchConfigurationConstants.ATTR_PROFILING_SET)
			 || name.equals(IProfileLaunchConfigurationConstants.ATTR_FILTER_SET));
	}

	public boolean getAutoMonitoring(ILaunchConfiguration conf) throws CoreException
	{
		return conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_AUTO_MONITORING, true);
	}
	
	public ArrayList getFilters(ILaunchConfiguration conf) throws CoreException
	{
		String filterSetId = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_FILTER_SET, LauncherConstants.DEFAULT_FILTER_ID);
		FilterSetElement filters = (FilterSetElement)getFilterSets().get(filterSetId);

		if (filters == null) 
		{
			filters = findNextAvailableFilterSet();
			if (filters == null)
				return new ArrayList(0);
		}
		
		return filters.getChildren();
	}
	
	
	public FilterSetElement findNextAvailableFilterSet()
	{
		/* Look for the default filter set */
		FilterSetElement filters = (FilterSetElement)getFilterSets().get(LauncherConstants.DEFAULT_FILTER_ID);
		
		/* If the default filter set is no longer available, then return the first filter set */
		if (filters == null && getFilterSets().size() > 0)
			filters = (FilterSetElement)getFilterSets().values().iterator().next();
		
		return filters;
	}
	
	public String getDefaultProfilingSet()
	{
		return TraceConstants.DEFAULT_PROFILING_SET;
	}

	/*
	 * By default, everything is turned off.
	 */
	public static List getDefaultProfilingOptions() {
		List list = new ArrayList();
		String[][] options = Options.OPTIONS_DEFAULT;

		final String PREFIX = "SETOPTION_";
		for (int i=0;i<options.length;++i) {
			list.add(new ProfilingAttribute(PREFIX + options[i][0], options[i][1]));
		}
		return list;
	}
	
	/*
	 * Removes all duplicate option keys. The resulting value is
	 * the one with the highest precedence.
	 */
	public static List filterDuplicateOptions(List original) {
		List result = new ArrayList();
		Iterator iter = original.iterator();
		while (iter.hasNext()) {
			addProfilingOption(result, (ProfilingAttribute)iter.next());
		}
		return result;
	}
	
	/**
	 * Add a profiling option to the set of existing options. If the option already exists,
	 * update the existing option value if necessary
	 * @param options
	 * @param option
	 */
	public static void addProfilingOption(List options, ProfilingAttribute option)
	{
		final String PREFIX = "SETOPTION_";
		for(int idx=0; idx<options.size(); idx++)
		{
			ProfilingAttribute pOption = (ProfilingAttribute)options.get(idx);
			if(pOption.getName().equals(option.getName()))
			{
				if(pOption.getValue().equals(Options.OPTION_VALUE_FALSE))
				{
					//false value is overridden by any other value
					pOption.setValue(option.getValue());
				}
				else if(pOption.getValue().equals(Options.OPTION_VALUE_NONE))
				{
					//none value is overridden by any other value
					pOption.setValue(option.getValue());
				}
				else if(option.getValue().equals(Options.OPTION_VALUE_FALSE))
				{
					//false value does not override anything
					return;
				}
				else if(pOption.getValue().equals(Options.OPTION_VALUE_NONE))
				{
					//none value does not override anything
					return;
				}
				else if(pOption.getName().equals(PREFIX + Options.OPTION_STACK_INFORMATION))
				{
					//OPTION_VALUE_NORMAL take precedence to OPTION_VALUE_NONE
					//OPTION_VALUE_BOUNDARY take precedence to OPTION_VALUE_NORMAL
					if(pOption.getValue().equals(Options.OPTION_VALUE_BOUNDARY))
						 return; //higher precedence
					
					if(!option.getValue().equals(Options.OPTION_VALUE_NONE))
						 pOption.setValue(option.getValue());
				}
				else if(pOption.getName().equals(PREFIX + Options.OPTION_TRACE_MODE))
				{
					if(pOption.getValue().equals(Options.OPTION_VALUE_NOOBJECTCORRELATION))
						pOption.setValue(option.getValue()); // override no object correlation value with any other value				
				}
				
				return;
			}
		}
		
		options.add(option);
	}	
}


