/**********************************************************************
 * Copyright (c) 2005, 2006 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExecutionTimeUI.java,v 1.16 2006/04/09 17:43:29 amehregani Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.launcher.defaults;


import java.util.Vector;

import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.hyades.ui.util.GridUtil;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Spinner;
import org.eclipse.swt.widgets.Text;
import org.eclipse.tptp.trace.ui.internal.launcher.application.PITimeAnalysisConf.ExecutionTimeConfigurationPage;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherMessages;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;
import org.eclipse.tptp.trace.ui.provisional.launcher.IStatusListener;
import org.eclipse.ui.PlatformUI;

/*
 * CONTEXT_ID topp0002 for collect boundary classes checkbox
 */
/*
 * CONTEXT_ID topp0008 for boundary class depth text field
 */
/*
 * CONTEXT_ID topp0009 for show exec flow radio button set
 */
/*
 * CONTEXT_ID topp0011 for show instance lvl info checkbox
 */
public class ExecutionTimeUI implements SelectionListener{
		
	private Composite _detailsHeapGroup;
	private Button _showHeapInstance;
	private Button _collectCPUtime;

	private Button _boundary;
	private Label _traceDepthLabel;
	private Text _traceDepth;

	private Composite _detailsExecGroup;
	private Button _showExecStatistic;
	private Button _showExecFlow;
	
	private Label titleLabel;
	private Label optionsLabel;
	private Composite _heapOptionGrp;
	private Button _advancedOptions;

	/* Indicates whether the title and description should be displayed */
	private boolean displayTitle;
	
	/* The status listeners that will be invoked to indicate normal/error conditions after
	 * field modification.  Instances are of type IStatusListener */
	private Vector listeners;
	
	/* A listener used to notify mofication of fields */
	private ModificationNotifier modificationNotifier;
	
	/* The automatically calculate polling frequency option */
	private Button _autoFrequency;
	
	/* The manually set polling frequency option */
	private Button _manualFrequency;
	
	/* The value of the polling frequency */
	private Spinner _pollingFreq;
	
	public ExecutionTimeUI() 
	{
		this(true);
	}

	public ExecutionTimeUI(boolean displayTitle)
	{
		super();
		this.displayTitle = displayTitle;
		modificationNotifier = new ModificationNotifier();
	}

	public Composite createControl(Composite parent) {
		
		GridLayout layout;

		Composite result = new Composite(parent, SWT.NONE);

		layout = new GridLayout();
		layout.numColumns = 1;
		layout.verticalSpacing = 2;

		GridData data = GridUtil.createFill();
		result.setLayout(layout);
		result.setLayoutData(data);
		
		if (displayTitle)
		{
			titleLabel = new Label(result, SWT.NULL);
			titleLabel.setText(UIPlugin.getResourceString("executionGroup"));
			titleLabel.setLayoutData(GridUtil.createHorizontalFill());

			Font font = titleLabel.getFont();
			if (font != null) {
				FontData[] fonts = font.getFontData();
				if (fonts != null && fonts.length >= 1) {
					titleLabel.setFont(new Font(Display.getDefault(), fonts[0].getName(), fonts[0].getHeight() + 3, SWT.BOLD));
				}
			}
			
			new Label(result, SWT.NULL);
		}
		
		
		if (displayTitle)
		{
			optionsLabel = new Label(result, SWT.WRAP);
			optionsLabel.setText(TraceMessages.AN_EXTM);
			data = new GridData();
			data.horizontalIndent = 20;
			data.widthHint = 375;
			optionsLabel.setLayoutData(data);
	
			Composite grp = new Composite(result, SWT.NULL);
			data = GridUtil.createHorizontalFill();
			layout = new GridLayout();
			layout.numColumns = 1;
			grp.setLayout(layout);
			grp.setLayoutData(data);
	
			addExecutionOption(grp);
			addHeapOption(grp);
			_detailsExecGroup.setVisible(true);
		}
		else
		{
			addExecutionOption(result);
			addHeapOption(result);
		}
				
		_detailsHeapGroup.setVisible(true);
		_traceDepth.setEnabled(_boundary.getSelection() );


		return result;
	}

	public void enableButtons()
	{
		_traceDepth.setEnabled(_boundary.getSelection() );
		enableHeapInstance(_showExecFlow.getSelection());
	}
	
	private void addHeapOption(Composite parent) {
		GridLayout layout;
		GridData data;

		_detailsHeapGroup = new Composite(parent, SWT.NULL);
		data = GridUtil.createHorizontalFill();
		data.horizontalIndent = 20;
		data.widthHint = 350;				
		_detailsHeapGroup.setLayoutData(data);
		layout = new GridLayout();
		layout.numColumns = 1;
		_detailsHeapGroup.setLayout(layout);

		_advancedOptions = new Button(_detailsHeapGroup, SWT.PUSH);
		_advancedOptions.setText(TraceMessages.ADV_BTN);
		_advancedOptions.addSelectionListener(this);

		_heapOptionGrp = new Composite(_detailsHeapGroup, SWT.NULL);
		_heapOptionGrp.setLayout(new GridLayout());
		
		_showHeapInstance = new Button(_heapOptionGrp, SWT.CHECK);
		_showHeapInstance.setText(TraceMessages.SH_INSTLI);	
		_showHeapInstance.addListener(SWT.Selection, modificationNotifier);
		
		Label info = new Label(_heapOptionGrp, SWT.WRAP);
		data = new GridData();
		data.widthHint = 375;		
		info.setLayoutData(data);
		info.setForeground(Display.getCurrent().getSystemColor(SWT.COLOR_RED));
		info.setText(TraceMessages.INSTL_INFO);
		
		_heapOptionGrp.setVisible(false);
		
		PlatformUI.getWorkbench().getHelpSystem().setHelp(
				_showHeapInstance,
				UIPlugin.getPluginId() + ".topp0011");
	}

	
	private void addExecutionOption(Composite parent) {
		
		GridLayout layout = new GridLayout(2, false);
		GridData data;
		
		Group levelOfDetail;
		if (displayTitle)
		{
			_detailsExecGroup = new Composite(parent, SWT.NULL);
			data = GridUtil.createHorizontalFill();
			if (displayTitle)
				data.horizontalIndent = 20;
			_detailsExecGroup.setLayoutData(data);
			
			_detailsExecGroup.setLayout(layout);
	
			levelOfDetail = new Group (_detailsExecGroup, SWT.NONE);
		}
		else
			levelOfDetail = new Group (parent, SWT.NONE);
		
		
		levelOfDetail.setLayout(layout);
		levelOfDetail.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));
		levelOfDetail.setText(LauncherMessages.CONFIGURATION_EXEC_GROUP_LEVEL);
		
		_collectCPUtime = new Button(levelOfDetail, SWT.CHECK);
		_collectCPUtime.setText(TraceMessages.COL_CPU);
		_collectCPUtime.addListener(SWT.Selection, modificationNotifier);
		
		
		_boundary = new Button(levelOfDetail, SWT.CHECK);
		data = GridUtil.createHorizontalFill();
		data.horizontalSpan = 2;
		_boundary.setLayoutData(data);
		_boundary.setText(TraceMessages.AUI_BCLS);
		_boundary.addListener(SWT.Selection, modificationNotifier);

		_traceDepthLabel = new Label(levelOfDetail, SWT.NULL);
		_traceDepthLabel.setText(TraceMessages.COL_DPT);
		_traceDepth = new Text(levelOfDetail, SWT.BORDER);
		data = new GridData();
		data.widthHint = 70;
		_traceDepth.setLayoutData(data);
		_traceDepth.addListener(SWT.Modify, modificationNotifier);

		Group _showExecGroup;
		
		if (displayTitle)
			_showExecGroup = new Group(_detailsExecGroup, SWT.NONE);	
		else
			_showExecGroup = new Group(parent, SWT.NONE);
		
		data = GridUtil.createHorizontalFill();
		data.horizontalSpan = 2;
		
		if (!displayTitle)
			data.verticalIndent = 20;
		
		_showExecGroup.setLayoutData(data);
		layout = new GridLayout();
		layout.numColumns = 1;
		_showExecGroup.setLayout(layout);
		_showExecGroup.setText(LauncherMessages.CONFIGURATION_EXEC_GROUP_TYPE);

		_showExecStatistic = new Button(_showExecGroup, SWT.RADIO);
		_showExecStatistic.setText(TraceMessages.EXEC_STAT);
		_showExecStatistic.addListener(SWT.Selection, modificationNotifier);
		
		/* Create the controls for allowing users to set a polling frequency */
		Composite pollingFreqComp = new Composite (_showExecGroup, SWT.NONE);
		GridLayout pollingFreqCompGL =  new GridLayout(2, false);
		pollingFreqCompGL.marginLeft = 20;
		pollingFreqComp.setLayout(pollingFreqCompGL);
		pollingFreqComp.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));
		
		
		/* The auto calculate button */
		_autoFrequency = new Button (pollingFreqComp, SWT.RADIO);
		GridData autoFrequencyGD = new GridData(SWT.FILL, SWT.DEFAULT, true, false);
		autoFrequencyGD.horizontalSpan = 2;		
		_autoFrequency.setLayoutData(autoFrequencyGD);
		_autoFrequency.setText(TraceMessages.EXECUTION_ANALYSIS_AUTO_FREQ);
		_autoFrequency.addListener(SWT.Selection, modificationNotifier);
		_autoFrequency.setSelection(true);
		
		/* The manual set button */
		_manualFrequency = new Button (pollingFreqComp, SWT.RADIO);		
		_manualFrequency.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));
		_manualFrequency.setText(TraceMessages.EXECUTION_ANALYSIS_MANUAL_FREQ);
		_manualFrequency.addListener(SWT.Selection, modificationNotifier);
		_manualFrequency.setSelection(false);
		
		/* The spinner for the manual set button */
		_pollingFreq = new Spinner (pollingFreqComp, SWT.BORDER);
		_pollingFreq.setLayout(new GridLayout());
		_pollingFreq.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));
		_pollingFreq.setValues(ExecutionTimeConfigurationPage.DEFAULT_POLLING_PERIOD, 1, 10000, 0, 10, 100);
		_pollingFreq.setEnabled(false);
		_pollingFreq.addListener(SWT.Selection, modificationNotifier);		
		
		/* Enable the polling frequency spinner when the manual set option is selected */
		_manualFrequency.addSelectionListener(new SelectionAdapter()
		{
			public void widgetSelected(SelectionEvent e) 
			{
				_pollingFreq.setEnabled(((Button)e.widget).getSelection());
			}
		});
		
		_showExecFlow = new Button(_showExecGroup, SWT.RADIO);
		_showExecFlow.setText(TraceMessages.DETAILED_DATA);
		_showExecFlow.addListener(SWT.Selection, modificationNotifier);
		_showExecFlow.addSelectionListener(new SelectionAdapter()
		{
			public void widgetSelected(SelectionEvent e) 
			{
				boolean isShowExecFlowSelected = !((Button)e.widget).getSelection();
				_autoFrequency.setEnabled(isShowExecFlowSelected);
				_manualFrequency.setEnabled(isShowExecFlowSelected);
				_pollingFreq.setEnabled(isShowExecFlowSelected && _manualFrequency.getSelection());
			}
		});
		
		_boundary.addSelectionListener(this);
		_showExecStatistic.addSelectionListener(this);
		_showExecFlow.addSelectionListener(this);
		
		PlatformUI.getWorkbench().getHelpSystem().setHelp(
			_boundary,
			UIPlugin.getPluginId() + ".topp0002");
		PlatformUI.getWorkbench().getHelpSystem().setHelp(
			_traceDepth,
			UIPlugin.getPluginId() + ".topp0008");
		PlatformUI.getWorkbench().getHelpSystem().setHelp(
			_showExecGroup,
			UIPlugin.getPluginId() + ".topp0009");

	}

	public void enable(boolean enable) {
	
		enableExecutionButtons(true);
	}

	public void enableExecutionButtons(boolean enable) {
		_boundary.setEnabled(enable);

		_traceDepth.setEnabled(enable);
		_traceDepthLabel.setEnabled(enable);

		_showExecStatistic.setEnabled(enable);
		_showExecFlow.setEnabled(enable);
	}

	public void enableBtnGroup() {
		enableExecutionBtnGroup();
	}


	


	private void enableExecutionBtnGroup() {
		_boundary.setEnabled(true);

		_showExecFlow.setEnabled(true);
		_showExecStatistic.setEnabled(true);

		_traceDepth.setEnabled(_boundary.getSelection());
		_traceDepthLabel.setEnabled(_boundary.getSelection());
	}



	public boolean getExecutionStatistic() {
		return _showExecStatistic.getSelection();
	}
	
	public boolean getExecutionFlow() {
		return _showExecFlow.getSelection();
	}

	public boolean getExecutionBoundary() {
		return _boundary.getSelection();
	}
	
	public boolean getInstance()
	{
		return _showHeapInstance.getSelection();
	}
	
	public boolean getCPUtime()
	{
		return _collectCPUtime.getSelection();
	}
	
	public void setInstance(boolean selection)
	{
		_showHeapInstance.setSelection(selection);
	}
	
	public void setCPUtime(boolean selection)
	{
		_collectCPUtime.setSelection(selection);
	}
	
	
	/**
	 * Indicates whether the polling frequency mode is automated
	 * or manually set to a constant.
	 * 
	 * @return true if automated; false otherwise
	 */
	public boolean isPollingFreqModeAutomated()
	{
		return _autoFrequency.getSelection();
	}
	
	
	/**
	 * Returns the polling frequency value.
	 * 
	 * @return the polling frequency
	 */
	public int getPollingFrequency()
	{
		return _pollingFreq.getSelection();
	}
	
 
	/**
	 * Set the polling frequency mode.
	 * 
	 * @param isAutomated Should be set if the polling frequency is to be
	 * automatically calculated; otherwise it should false.
	 */
	public void setPollingFrequencyMode(boolean isAutomated)
	{
		_autoFrequency.setSelection(isAutomated);
		_manualFrequency.setSelection(!isAutomated);
		_pollingFreq.setEnabled(_showExecStatistic.getSelection() && !isAutomated);
	}
	
	
	/**
	 * Sets the polling frequency.  For the frequency to be taken into effect,
	 * the polling frequency mode must be set to manual.
	 * 
	 * @param pollingFrequency The polling frequency
	 */
	public void setPollingFrequency(int pollingFrequency)
	{
		_pollingFreq.setSelection(pollingFrequency);
	}
		
	
	public int getCollectionDepth() {
		String text = _traceDepth.getText().trim();

		try {
			int depth = Integer.parseInt(text);
			if (depth > 0)
				return depth;
			else
				return UIPlugin
					.getDefault()
					.getPreferenceStore()
					.getDefaultInt(
					TraceConstants.COLLECT_DEPTH);
		} catch (Exception exc) {
			return UIPlugin.getDefault().getPreferenceStore().getDefaultInt(
				TraceConstants.COLLECT_DEPTH);
		}
	}




	public void setExecutionStatistic(boolean select) {
		_showExecStatistic.setSelection(select);
		_showHeapInstance.setEnabled(!select);
	}

	public void setExecutionFlow(boolean select) {
		_showExecFlow.setSelection(select);
		Event event = new Event();
		event.widget = _showExecFlow;
		_showExecFlow.notifyListeners(SWT.Selection, event);
	}

	public void setExecutionBoundary(boolean setValue) {
		_boundary.setSelection(setValue);

		_traceDepth.setEnabled(_boundary.getSelection());
		_traceDepthLabel.setEnabled(_boundary.getSelection());

	}

	public void setCollectionDepth(String value) {
		_traceDepth.setText(value);
	}
	
	public void enableHeapInstance(boolean enable) {
		if (enable==false) _showHeapInstance.setSelection(enable);
		_showHeapInstance.setEnabled(enable);
		//_advancedOptions.setEnabled(enable);
	}
	
	

	public void widgetDefaultSelected(SelectionEvent event) {
	}

	public void widgetSelected(SelectionEvent event) {
		if (event.widget == _boundary) {
			_traceDepth.setEnabled(_boundary.getSelection());
			_traceDepthLabel.setEnabled(_boundary.getSelection());
		}
		else if (event.widget == _showExecStatistic) {
			enableHeapInstance(_showExecFlow.getSelection());
		}
		else if (event.widget == _showExecFlow) {
			enableHeapInstance(_showExecFlow.getSelection());
		}
		else if (event.widget == _advancedOptions) {
			_heapOptionGrp.setVisible(!_heapOptionGrp.getVisible());
		}
	}

	
	/**
	 * Sets a default setting for the available options
	 */
	public void setDefaultSettings()
	{
		setCPUtime(false);
		setExecutionBoundary(false);		
		setCollectionDepth("1");
		setExecutionFlow(false);
		setExecutionStatistic(true);
		setInstance(false);
	}

	
	/**
	 * Register a status listener.  The listener will be invoked to indicate
	 * any errors that may occur as a result of bad input.
	 * 
	 * @param statusListener The status listener
	 */
	public void addStatusListener (IStatusListener statusListener)
	{
		if (listeners == null)
			listeners = new Vector();
		
		listeners.add(statusListener);
	}
	
	
	/**
	 * A modification notifier that is invoked every time a modifiable field
	 * is changed by the user.
	 * 
	 * @author Ali Mehregani
	 */
	public class ModificationNotifier implements Listener
	{
		public void handleEvent(Event event)
		{
			if (listeners == null)
				return;
			
			String errorMessage = null;			
			if (_boundary.getSelection() && !LauncherUtility.isInputPositiveInteger(_traceDepth.getText()))
				errorMessage = LauncherMessages.ERROR_CONFIG_EXEC_DEPTH;
			
			for (int i = 0, listenerSize = listeners.size(); i < listenerSize; i++)
			{
				IStatusListener statusListener = (IStatusListener)listeners.get(i);
				if (errorMessage == null)
					statusListener.handleOKEvent();
				else
					statusListener.handleErrorEvent(errorMessage);
			}			
		}
		
	}
}
