/**********************************************************************
 * Copyright (c) 2005, 2006 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ProfileDetailNavigatorExtension.java,v 1.6 2006/05/11 15:05:05 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.navigator;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCConfiguration;
import org.eclipse.hyades.models.hierarchy.TRCOption;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfilingSetsManager;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSetType;
import org.eclipse.hyades.ui.extension.INavigatorContribution;
import org.eclipse.hyades.ui.extension.INavigatorItem;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.tptp.trace.ui.internal.launcher.core.AnalysisType;
import org.eclipse.tptp.trace.ui.internal.launcher.core.AnalysisTypeManager;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherConstants;

/**
 * Extends the Profiling Monitor, providing profile detail items under
 * every TRCAgentProxy object if it has profiling detail.
 */
public class ProfileDetailNavigatorExtension implements INavigatorContribution {

	/* Modified By: Ali M. - This class used to reload the profiling type extension points.
	 * To encourage consistency I've changed this so that profiling types are accessed via 
	 * the profiling set manager. */
	
    protected static final String getProfilingTypesExtPoint = "profilingTypes";
    private Map typeItems = new HashMap();
    
	public boolean hasChildren(Object item) {
		return (item instanceof TRCAgentProxy);
	}
	
	public List getChildren(Object item) {
		//get the profiling details of this agent proxy
		if (item instanceof TRCAgentProxy)
			return getProfileDetailItems((TRCAgentProxy)item);
		return  new ArrayList(0);
	}
	
	// non-persistent items
	public Collection getModifiedItems() {
		return new ArrayList(0);
	}

	public void fillContextMenu(IMenuManager menu, IStructuredSelection selection) {
	}
	
	/**
	 * @param agentProxy
	 * @return a list of ProfileDetailItems for this agent proxy
	 * one Profile Detail Item per profiling type in the configuration
	 */
	private List getProfileDetailItems(TRCAgentProxy agentProxy)
	{
		Vector details = new Vector();
		
		List configs = agentProxy.getConfigurations();
		for (int i = 0; i < configs.size(); i++)
		{
			//need to loop through the configurations to find the correct one
			TRCConfiguration config = (TRCConfiguration)configs.get(i);
			List options = config.getOptions();
			for (int opt = 0; opt < options.size(); opt++)
			{
				//need to loop through all the options to find ones 
				//relating to the profiling type
				TRCOption option = (TRCOption)options.get(opt);
				
				/* Ali M. -- Because of defect https://bugs.eclipse.org/bugs/show_bug.cgi?id=135954, I had to change the 
				 * prefix form UIPlugin.getPluginId() + ".type" to LauncherConstants#ANALYSIS_TYPE_ASSOCIATION.  The 
				 * association is done in PrimaryLaunchDelegate#addAnalysisTypes */
				if (option.getKey().startsWith(LauncherConstants.ANALYSIS_TYPE_ASSOCIATION))
				{
					if (option.getValue().length() >0)
					{
						/* 
						 * Found a profiling type, add it to the list
						 * if it is a known type. If not, ignore it.
						 */
						INavigatorItem profilingType = getProfileDetailItem(agentProxy, option.getValue()); 
						if (profilingType != null) {
							details.add(profilingType);
						}
					}
				}
			}
			//if details size > 0, we found the right config, and can break
			if (details.size() > 0) break;
		}
		return details;
	}
	
	/**
	 * 
	 * @param agentProxy
	 * @param id - the id of the profiling type read from the configuration
	 * @return an instance of a ProfileDetailItem associatied with the
	 * agent proxy sent in, with it's corresponding
	 *  name and icon read from the extension point
	 */
	private ProfileDetailItem createProfileDetailItem(TRCAgentProxy agentProxy, String id)
	{
		IProfilingSetType profilingSetType = (IProfilingSetType) ProfilingSetsManager.instance().getProfilingTypes().get(id);

		if (profilingSetType != null) 
		{	
			AnalysisType analysisType = AnalysisTypeManager.getInstance().getAnalysisType(id);
			return new ProfileDetailItem(agentProxy, new String[] {profilingSetType.getName(), profilingSetType.getId()}, profilingSetType.getImage(), analysisType);			
		}
		
		return null;
	}

	public void dispose() {
	}
	
	private INavigatorItem getProfileDetailItem(TRCAgentProxy agent, String typeID) {
		Object obj = typeItems.get(agent);
		if (obj == null) {
			obj = new HashMap(5);
			typeItems.put(agent, obj);
		}
		
		Map agentTypeItems = (Map)obj;
		Object typeItem = agentTypeItems.get(typeID);
		if (typeItem == null) {
			INavigatorItem item = createProfileDetailItem(agent, typeID);
			if (item != null) {
				agentTypeItems.put(typeID, item);
				return item;
			}
			return null;
		}
		return (INavigatorItem)typeItem;
	}
}
