/**********************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: PIProcessListener.java,v 1.25 2007/05/29 13:54:43 amehregani Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.piclient;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.internal.execution.local.common.CommandElement;
import org.eclipse.hyades.internal.execution.local.common.CustomCommand;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.AgentConfiguration;
import org.eclipse.hyades.internal.execution.local.control.AgentPeerListener;
import org.eclipse.hyades.internal.execution.local.control.InactiveAgentException;
import org.eclipse.hyades.internal.execution.local.control.InactiveProcessException;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.internal.execution.local.control.ProcessImpl;
import org.eclipse.hyades.internal.execution.local.control.ProcessListener;
import org.eclipse.hyades.internal.execution.local.control.Variable;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.models.hierarchy.HierarchyFactory;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCEnvironmentVariable;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.models.hierarchy.util.SaveUtil;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.ProfileUIManager;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.util.AttachConfigurationDetail;
import org.eclipse.hyades.trace.ui.internal.util.PDCoreUtil;
import org.eclipse.hyades.trace.ui.internal.util.TString;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.hyades.trace.ui.launcher.ILauncherHandler;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.tptp.platform.common.internal.CommonPlugin;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIConstants;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceConstants;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;
import org.eclipse.tptp.trace.ui.internal.launcher.deleg.application.StatGeneratorDataProcessor;


public class PIProcessListener implements ProcessListener, AgentPeerListener {
    //~ Instance fields ----------------------------------------------------------------------------

    protected ArrayList _monitoredAgents = new ArrayList();
    protected ArrayList _processVariableList = new ArrayList();
    protected TRCNode fNode;
    protected TRCProcessProxy fProcess;
    protected boolean _autoMonitoring = true;
    protected boolean _monitor = false;
    protected int _launchMode = 0;
	private ArrayList filteredAgentList;
    protected static final String getLauncherHandlersExtPoint = "launcherHandler";
    private static Map launcherHandlers;

    /** A data processor associated with this listener */
    private StatGeneratorDataProcessor fdataProcessor;
    
    //~ Constructors -------------------------------------------------------------------------------

    /**
     * PIProcessListener constructor comment.
     */
    public PIProcessListener(TRCNode node) {
        super();
        fNode = node;
    }

    public PIProcessListener(TRCNode node, TRCProcessProxy process) {
        this(node);
        fProcess = process;
    }

    
    //~ Methods ------------------------------------------------------------------------------------

    /**
     * @param b
     */
    public void setAutoMonitoring(boolean b) {
        _autoMonitoring = b;
    }

    /**
     * Insert the method's description here.
     * Creation date: (5/2/2001 5:37:22 PM)
     * @param mode int
     */
    public void setLaunchMode(int mode) {
        _launchMode = mode;
    }

    /**
     *
     * @return HashMap
     */
    public ArrayList getProcessVariableList() {
        return _processVariableList;
    }
    
    /**
     * Return the data processor associated with this listener.
     * This can be null if no processor has not yet been associated with this listener.
     * 
     * @return The data processor associated with this listener
     */
    public StatGeneratorDataProcessor getDataProcessor()
    {
    	return fdataProcessor;
    }
    
    

    public synchronized void agentActive(Agent ag) {
        if (fProcess == null) {
            return;
        }

        if (_monitoredAgents.contains(ag.getName())) {
            _monitor = false;
        } else {
            _monitoredAgents.add(ag.getName());
        }

        if (!_monitor) {
            return;
        }

        TRCAgentProxy agentProxy = PDCoreUtil.getCorrespondingAgent(fProcess,ag,false);
		
        if (agentProxy == null) {
            return;
        }
        
        agentProxy.setRuntimeId(ag.getUUID());
        LoadersUtils.registerAgentInstance(agentProxy, ag);

        agentProxy.setAttached(true);
        agentProxy.setProfileFile(ag.getProfileFile());

        giveControl(agentProxy, true, true);
        
        if (agentProxy.isActive()) {
            final TRCAgentProxy agentTemp = agentProxy;

            Display d = Display.getDefault();

            d.asyncExec(new Runnable() {
                    public void run() {
                        //update ui
                        ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

                        event.setSource(agentTemp);
                        event.setType(ProfileEvent.START_MONITOR);
                        UIPlugin.getDefault().notifyProfileEventListener(event);
                    }
                });
            return; //this is an attach scenario
        }

        agentProxy.setActive(true);

        try {
            ag.publishConfiguration();

            CustomCommand applyFilterCommand = new CustomCommand();
            CustomCommand resumeCommand = new CustomCommand();

            applyFilterCommand.setData("APPLYFILTERS");
            ag.invokeCustomCommand(applyFilterCommand);

                           
          	fdataProcessor = (StatGeneratorDataProcessor) LoadersUtils.locateDataProcessor(agentProxy);
          	if (fdataProcessor == null) 
            {
          		fdataProcessor = new StatGeneratorDataProcessor(agentProxy);
                LoadersUtils.registerDataProcessor(agentProxy, fdataProcessor);                        
            }
           

            if (agentProxy.isToProfileFile()) {
            	fdataProcessor.setProfileFileName(agentProxy.getProfileFile());
            	fdataProcessor.createWriter();

                if (fdataProcessor.isNewProfileFile()) {
                	fdataProcessor.writeXMLVersion(fdataProcessor.getWriter());
                	fdataProcessor.startTrace(fdataProcessor.getWriter());
                }
            }

            agentProxy.setAttached(true);

            if (_autoMonitoring) {
                ag.startMonitoring(fdataProcessor);
                agentProxy.setMonitored(true);
            }

            agentProxy.setCollectionData(true);

            resumeCommand.setData("RESUME");
            ag.invokeCustomCommand(resumeCommand);
            
            giveControl(agentProxy, false, true);
        } catch (InactiveAgentException exc) {
            exc.printStackTrace();
        }

        final TRCAgentProxy agentTemp = agentProxy;

        Display d = Display.getDefault();

        d.asyncExec(new Runnable() {
                public void run() {
                    //update ui
                    ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

                    event.setSource(agentTemp);
                    event.setType(ProfileEvent.START_MONITOR);
                    UIPlugin.getDefault().notifyProfileEventListener(event);
                }
            });
    }
    


    public synchronized void agentInactive(Agent ag) {
        if (fProcess == null) {
            return;
        }

        TRCAgentProxy agentProxy = PDCoreUtil.getCorrespondingAgent(fProcess,ag,true);
        if (agentProxy == null) {
            return;
        }
  
        
        agentProxy.setActive(false);
        agentProxy.setAttached(false);
        agentProxy.setMonitored(false);
        fProcess.setActive(false);

        giveControl(agentProxy, true, false);
        
        final TRCAgentProxy agentTemp = agentProxy;
        

        Display d = Display.getDefault();

        d.asyncExec(new Runnable() {
                public void run() {
                    //update ui
                    ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

                    event.setSource(agentTemp);
                    event.setType(ProfileEvent.STOP_MONITOR);
                    UIPlugin.getDefault().notifyProfileEventListener(event);
                }
            });
    }
    /**
     * called during an agentActive or agentInactive scenario
     * @param agentProxy
     * @param pre
     * @param active
     */
    
    protected void giveControl(TRCAgentProxy agentProxy, boolean pre, boolean active)
    {
    	//if the map has not been populated yet, populate it
    	if (launcherHandlers == null) initializeLauncherHandlers();
    	
    	Object[] launcherHandlersArray = launcherHandlers.values().toArray();
    	
    	for (int num = 0; num < launcherHandlersArray.length; num++)
    	{
    		ILauncherHandler type = (ILauncherHandler)launcherHandlersArray[num];
    		
    		try
			{
    			if (active)
    			{
    				if (pre)type.handlePreAgentActive(agentProxy);
    				else type.handlePostAgentActive(agentProxy);
    			}
    			else
    			{
    				//inactive
    				type.handleAgentInactive(agentProxy);
    			}
    			
			}
    		catch (Exception e)
			{
    			if(e.getMessage() != null)
    			{
    				IStatus status = new Status(IStatus.ERROR,UIPlugin.getPluginId(),IStatus.ERROR,e.getMessage(),e);
        			UIPlugin.getDefault().getLog().log(status);    				
    			}
    			//quietly catch any exceptions
    			//and ignore them
			}

    	}
    }
    
	/**
	 * called during a process Lauhced or process Exited scenario
	 * @param processProxy
	 * @param launched
	 */
    protected void giveControl(TRCProcessProxy processProxy, boolean launched)
    {

    	//if the map has not been populated yet, populate it
    	if (launcherHandlers == null) initializeLauncherHandlers();
    	
    	Object[] launcherHandlersArray = launcherHandlers.values().toArray();
    	
    	for (int num = 0; num < launcherHandlersArray.length; num++)
    	{
    		ILauncherHandler type = (ILauncherHandler)launcherHandlersArray[num];
    		
    		try
			{
        		if (launched)
        		{
        			type.handleProcessLaunched(processProxy);
        		}
        		else
        		{
        			type.handleProcessExited(processProxy);
        		}
    			
			}
    		catch (Exception e)
			{
    			if(e.getMessage() != null)
    			{
    				IStatus status = new Status(IStatus.ERROR,UIPlugin.getPluginId(),IStatus.ERROR,e.getMessage(),e);
        			UIPlugin.getDefault().getLog().log(status);    				
    			}
    			//quietly catch any exceptions
    			//and ignore them
			}

    	}
    }
    
    private void initializeLauncherHandlers()
    {
    	//reads the extension points, and creates instances of the GetControlType objects
    	//specified by the class attribute
    	launcherHandlers = new HashMap();
		
		IExtensionPoint point = Platform.getExtensionRegistry().getExtensionPoint(UIPlugin.PLUGIN_ID, getLauncherHandlersExtPoint);

		if (point != null) {
			IConfigurationElement[] elements = point.getConfigurationElements();
			for (int i = 0; i < elements.length; i++) {
				
				IConfigurationElement elem = elements[i];

				String id = elem.getAttribute("id");								

				ILauncherHandler getControlType = null;
				try
				{
					getControlType = (ILauncherHandler)elem.createExecutableExtension("class");
				}
				catch (Exception e)
				{
	    			if(e.getMessage() != null)
	    			{
	    				IStatus status = new Status(IStatus.ERROR,UIPlugin.getPluginId(),IStatus.ERROR,e.getMessage(),e);
	        			UIPlugin.getDefault().getLog().log(status);    				
	    			}
					
					e.printStackTrace();
				}
				
				if(getControlType != null)			
				{
					//put the instace of the class in a map
					launcherHandlers.put(id, getControlType);					
				}
			}
		}
    }

    public void dispose() {
        _monitoredAgents.clear();
        fNode = null;
        fProcess = null;
        _processVariableList.clear();
    }

    /**
     * Invoked when an error is recieved from the agent.
     */
    public void error(Agent agent, String errorId, String errStr) {
        final String errorMsg = errStr;

        Display d = Display.getDefault();

        d.asyncExec(new Runnable() {
                public void run() {
                    Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.INTERNAL_ERROR, errorMsg, null);

                    ErrorDialog.openError(new Shell(), TraceMessages.LJP_ERROR_
                    		, TraceMessages.PB_AC_WARN_, // no special message
                                          err);
                }
            });
    }

    public void handleCommand(Agent agent, CommandElement command) {
    }

    /*
     * return false if the agent should ignore the agentActive event
     */
    public void monitor(boolean monitor) {
        _monitor = monitor;
        _monitoredAgents.clear();
    }

    /**
     * Invoked when an agent requests to be monitored because this client
     * is currently monitoring another agent.
     */
    public synchronized void peerWaiting(Agent agent, Agent peer) {
        /*
         * Check if we're already attached to this agent. If so,
         * don't try to attach again.
         */
        if (!peer.isAttached() && fProcess != null && fNode != null) {
	        /*
	         * Find the TRCAgentProxy associated with the agent, in
	         * order to match the settings on the new TRCAgentProxy.
	         */
	        TRCAgentProxy trcAgent = null;
	        Iterator iter = fProcess.getAgentProxies().iterator();
	        while (iter.hasNext()) {
	            TRCAgentProxy agentProxy = (TRCAgentProxy)iter.next();
	            Object obj = LoadersUtils.locateAgentInstance(agentProxy);
	            if (obj == agent) {
	                trcAgent = agentProxy;
	                break;
	            }
	        }
	        
	        /*
	         * Set the default project, monitor, and profile file.
	         */
			IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();
			String projectName = store.getString(CommonUITraceConstants.TRACE_PROJECT_NAME);
			String monitorName = store.getString(CommonUITraceConstants.TRACE_MONITOR_NAME);
			
			// bug 95689: if agent is currently profile to file, the peer agent will also be profile to file.
			String profileFile = (agent!=null? agent.getProfileFile():null);
			String peerProfileFile = null;
			if (profileFile!=null) {
				
				StringBuffer suffix = new StringBuffer().append("_").
					append(trcAgent.getProcessProxy().getNode().getName()).append("_").
					append(trcAgent.getProcessProxy().getPid()).append("_").
					append(trcAgent.getRuntimeId()).append(".trcxml"); //$NON-NLS-1$
				
				int ext = profileFile.lastIndexOf(".trcxml"); //$NON-NLS-1$
				if (ext>0) {
					peerProfileFile = new StringBuffer(profileFile.substring(0,ext)).append(suffix).toString();
				}
				else {
					peerProfileFile =  new StringBuffer(profileFile).append(suffix).toString();
				}
			}
	
	        /*
	         * If the peer agent can be found, match the values on the
	         * newly discovered agent.
	         */
	        if (trcAgent != null) {
	            TRCProcessProxy trcProcess = trcAgent.getProcessProxy();
	            if (trcProcess != null) {
	                TRCNode trcNode = trcProcess.getNode();
	                if (trcNode != null) {
	                    TRCMonitor trcMonitor = trcNode.getMonitor();
	                    if (trcMonitor != null) {
//	                        IResource folder = HyadesUtil.getSelectionFolder(trcMonitor);
	                    	URI projectPath = trcMonitor.eResource().getURI();
	                        String fullProjectPath =  projectPath.trimSegments(1).path();
	                        fullProjectPath = fullProjectPath.substring(fullProjectPath.indexOf('/', 1));
	                        fullProjectPath = fullProjectPath.replaceAll("%20", " ");
	                        if (fullProjectPath != null) {
	                            monitorName = trcMonitor.getName();
	                            projectName = fullProjectPath;
	                        }
	                    }
	                }
	            }
	        }
	        
	        try {
	            Node node = peer.getProcess().getNode();
	            final String hostname = node.getName();
	            String port = String.valueOf(node.getConnection().getPort());
	            Vector agents = new Vector();
	            agents.add(peer);
	            
	            /*
	             * Copy the agent options to the newly discovered
	             * agent.
	             */
	            Vector options = new Vector();
	            AgentConfiguration conf = agent.getConfiguration();
	            for (int i=0;i<conf.size();++i) {
	                options.add(conf.getEntryAt(i));
	            }
	            
	            /*
	             * Force the connection to be established before attaching.
	             * This will avoid the delay in re-establishing when start
	             * monitoring, which freezes the UI.
	             */
	            PDCoreUtil.profileConnect(hostname, port);
	            
	            /*
	             * Attach to the newly discovered agent.
	             */
	            AttachConfigurationDetail attachConfigurationDetail = new AttachConfigurationDetail();
	            attachConfigurationDetail.setHostName(hostname);
	            attachConfigurationDetail.setAgents(agents);
	            attachConfigurationDetail.setOptions(options);
	            attachConfigurationDetail.setProject(projectName);
	            attachConfigurationDetail.setMonitor(monitorName);
	            attachConfigurationDetail.setPortNumber(port);
	            attachConfigurationDetail.setProfileFile(peerProfileFile);
	            attachConfigurationDetail.setSwitchPerspective(false);
	            
	            List trcAgents = PDCoreUtil.attach(attachConfigurationDetail);
	            
	            if (trcAgents != null)
	            {
		            for (int i = 0, agentsCount = trcAgents.size(); i < agentsCount; i++)
					{
		            	TRCAgentProxy currentAgentProxy = (TRCAgentProxy)trcAgents.get(i);
		            	Object currentAgent = LoadersUtils.locateAgentInstance(currentAgentProxy);
		            	if (currentAgent instanceof Agent)
		            	{
		            		PDCoreUtil.doAttach((Agent)currentAgent, currentAgentProxy);
		            	}
					}
		            
		            if (trcAgents.size() == 1) {
		                final Object obj = trcAgents.get(0);
		                if (obj instanceof TRCAgentProxy) {
		                    Display.getDefault().syncExec(new Runnable() {
		                        public void run() {
		    			            ProfileUIManager.getInstance().startMonitoring((TRCAgentProxy)obj);
		                        }
		                    });
		                }
		            }	            	
	            }
	        }
	        catch (Exception e) {
	        	CommonPlugin.logError(e);
	        }
        }
    }

    public synchronized void processExited(Process result) {
        if (fProcess != null) {
            fProcess.setActive(false);
            giveControl(fProcess, false);
            
        }
    }

    public synchronized void processLaunched(Process result) {
    	try {
            if ((result.getProcessId() == null) || result.getProcessId().equals("-1")) {
                return;
            }

            /* Create the process only if it's not already intialized */
            if (fProcess == null)
            	fProcess = createProcess(result);
            

             if (fProcess == null) {
                return;
            }

            //create agents
            Enumeration agents = result.listAgents();

            while (agents.hasMoreElements()) 
            {
            	Agent currentAgent = (Agent)agents.nextElement();
            	if (filteredAgentList == null || isAgentIncluded(currentAgent))
            		createAgent(currentAgent, fProcess);
            	
            }
            
            giveControl(fProcess, true);
            
        } catch (InactiveProcessException ex) {
            String text = TraceMessages.RAC_CERR;
            final String msg = NLS.bind(TraceMessages.LNCH_ERR, result.getName());
            final Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.INTERNAL_ERROR, text, null);

            Display.getDefault().syncExec(new Runnable() {
				public void run() {
		            ErrorDialog.openError(UIPlugin.getDefault().getViewer().getShell(), TraceMessages.TRC_MSGT, msg, err);
				}
            });
            return;
        } catch (Exception exc) {
            exc.printStackTrace();
        }
    }

     private boolean isAgentIncluded(Agent currentAgent)
	{
    	String agentName = currentAgent.getName();
    	String agentType = currentAgent.getType();
		for (int i = 0, agentListSize = filteredAgentList.size(); i < agentListSize; i++)
		{
			Agent agent = (Agent)filteredAgentList.get(i);
			if (agent.getName().equals(agentName) && agent.getType().equals(agentType))
				return true;
		}
		return false;
	}

	public Agent sendConfigurationToAgent(TRCAgentProxy agent, String host, String processId) {
        try {
            Node node = PDCoreUtil.profileConnect(host, String.valueOf(agent.getProcessProxy().getNode().getPort()));

            if (node == null) {
                return null;
            }

            Process p = node.getProcess(processId);

            if (p != null) {
                Agent a = p.getAgent(agent.getName());

                PDCoreUtil.setAgentConfiguration(agent, a);

                if ((a != null) && a.isActive()) {
                    a.publishConfiguration();

                    //			  a.invokeCustomCommand(new ApplyFiltersCommand());
                    CustomCommand command = new CustomCommand();

                    command.setData("APPLYFILTERS");
                    a.invokeCustomCommand(command);
                }

                return a;
            }
        } catch (Exception exc) {
            exc.printStackTrace();
        }

        return null;
    }



    private void addEnvironment(TRCProcessProxy process) {
        for (int idx = 0; idx < _processVariableList.size(); idx++) {
            Variable var = (Variable) _processVariableList.get(idx);

            HierarchyFactory factory = UIPlugin.getDefault().getPerftraceFactory();

            if (!var.getName().equals("UICLASSPATH")) {
                TRCEnvironmentVariable env = factory.createTRCEnvironmentVariable();

                env.setName(var.getName());
                env.setValue(var.getValue());
                process.getEnvironmentVariables().add(env);
            } else {
                process.setClasspath(var.getValue());
            }
        }
    }

    private TRCAgentProxy createAgent(Agent a, TRCProcessProxy process) {
        if (process == null) {
            return null;
        }

        TRCAgentProxy agent = PDCoreUtil.createAgent(process, a);

        //    agent.setAgentListener(this);
        LoadersUtils.registerAgentListener(agent, this);

        return agent;
    }

    private TRCProcessProxy createProcess(Process p, TRCNode pNode) {
        try {
            final TRCNode node = pNode;
            TRCMonitor monitor = node.getMonitor();


            String processId = p.getProcessId();
            String[] processAtts = LauncherUtility.resolveProcessAttributes(p);
            String pName = processAtts[0];
            String params = processAtts[1];
            String vmparam = processAtts[2];
            
            String rId = ((ProcessImpl) p).getUUID();

            if (rId == null) {
                rId = "";
            }

            String fileName = new StringBuffer(monitor.getName()).append("_").append(node.getName()).append("_").append(TString.change(pName, " ", "")).append(processId).append("_").append(rId).append(".").append(CommonUIConstants.PROCESS_EXT).toString();

            String folderPath = node.eResource().getURI().toString();
            IPath path = new Path(folderPath);

            if (path.segmentCount() > 1) {
                folderPath = path.removeLastSegments(1).toString();
            }

            IPath filePath = new Path(folderPath).append(fileName);

//            URI uri = SaveUtil.createURI(filePath.toString());
            URI uri = SaveUtil.createURI(filePath.toString()+"#").trimFragment();
            
            HierarchyFactory factory = UIPlugin.getDefault().getPerftraceFactory();
            Resource pDoc = Resource.Factory.Registry.INSTANCE.getFactory(uri).createResource(uri);
            pDoc.setModified(true);
            EList pExt = pDoc.getContents();

            UIPlugin.getDefault().getResourceSet().getResources().add(pDoc); // prevents reloading later

//            //*** adding support for multiple files
//            SaveUtil.addDocument(pDoc);

            final TRCProcessProxy process = factory.createTRCProcessProxy();

            process.setPid(Integer.parseInt(processId));
            process.setRuntimeId(rId);
            process.setName(pName);
            process.setLocation(((ProcessImpl) p).getlocation());
            process.setNode(node);

            addEnvironment(process);

            process.setParameters(params.trim());
            process.setVmArguments(vmparam.trim());
            process.setLaunchMode(_launchMode);
            process.setActive(true);
            pExt.add(process);

            node.getProcessProxies().add(process);

            Display d = Display.getDefault();

            d.asyncExec(new Runnable() {
                    public void run() {
                        ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

                        event.setSource(null);
                        event.setType(ProfileEvent.UNSPECIFIED);
                        UIPlugin.getDefault().notifyProfileEventListener(event);
                    }
                });

            _processVariableList.clear();

            return process;
        } catch (Exception exc) {
            exc.printStackTrace();
        }

        return null;
    }

    private TRCProcessProxy createProcess(Process p) {
        return createProcess(p, fNode);
    }

    /**
     * If this method is invoked to add an agent, then only agents
     * that are explicitly added will be loaded as part of the model; otherwise
     * all agents associated with the process will be loaded.
     * 
     * @param agent The agent to be added
     */
    public void addAgent(Agent agent)
    {
    	if (filteredAgentList == null)
    		filteredAgentList = new ArrayList();
    	
    	filteredAgentList.add(agent);
    }
}
