/********************************************************************** 
 * Copyright (c) 2005, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: ContextMappingPreferencePage.java,v 1.8 2008/07/17 20:54:50 ewchan Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.preferences;

import java.util.Arrays;
import java.util.Comparator;

import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.internal.context.ContextMappingPreference;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.hyades.ui.provisional.context.ContextManager;
import org.eclipse.hyades.ui.provisional.context.ContextOpenSourceHandlerManager;
import org.eclipse.hyades.ui.provisional.context.IContextHandler;
import org.eclipse.hyades.ui.provisional.context.IContextLanguage;
import org.eclipse.hyades.ui.provisional.context.IContextOpenSourceHandler;
import org.eclipse.hyades.ui.util.GridUtil;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIPlugin;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;
import org.eclipse.ui.PlatformUI;

public class ContextMappingPreferencePage extends PreferencePage implements IWorkbenchPreferencePage, SelectionListener {

	private Table contextLanguagesTable;
	private Table contextProvidersTable;
	private Table contextOpenSourceProvidersTable;
	
	private ContextMappingPreference contextMappingPreference;

	
	public ContextMappingPreferencePage()
	{
		noDefaultAndApplyButton();
		contextMappingPreference = new ContextMappingPreference(CommonUIPlugin.getDefault().getPreferenceStore());
	}
	
	protected Control createContents(Composite parent) {
		
		contextMappingPreference.populateMappingByContext();
		PlatformUI.getWorkbench().getHelpSystem().setHelp(
				parent,
				UIPlugin.getPluginId()+".tbpp0007");
		
		Composite pageComponent = new Composite(parent, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		gridLayout.marginWidth = 0;
		gridLayout.marginHeight = 0;
		pageComponent.setLayout(gridLayout);
		pageComponent.setLayoutData(GridDataUtil.createFill());
		
		contextLanguagesTable = createSelectionTable(pageComponent, TraceMessages._10, 8);

		Group group = createGroup(pageComponent);
		contextProvidersTable = createSelectionTable(group, TraceMessages._11, 5);
		Label spacer = new Label(group, SWT.LEFT);
		GridData data = new GridData();
		data.heightHint = 1;
		spacer.setLayoutData(data);
		contextOpenSourceProvidersTable = createSelectionTable(group, TraceMessages._12, 5);
		
		populateLanguagesTable();
		populateProviderTables();
		
		Dialog.applyDialogFont( pageComponent );
		
		return pageComponent;
	}
	
	private Table createSelectionTable(Composite parent, String labelText, int rows)
	{
		Label label = new Label(parent, SWT.LEFT);
		label.setText(labelText);
		Table selectionTable = new Table(parent, SWT.SINGLE | SWT.BORDER | SWT.FULL_SELECTION);
		GridData gridData = new GridData(GridData.FILL_HORIZONTAL);
		gridData.heightHint = (parent.getFont().getFontData())[0].getHeight() * rows;
		selectionTable.setLayoutData(gridData);		
		selectionTable.addSelectionListener(this);

		return selectionTable;
	}
	
	private Group createGroup(Composite parent)
	{
		Group group = new Group(parent, SWT.SHADOW_ETCHED_IN);
		group.setText(TraceMessages._13); //$NON-NLS-1$
	
		GridLayout layout= new GridLayout();
		layout.horizontalSpacing = 0;
		group.setLayout(layout);
		group.setLayoutData(GridUtil.createFill());

		return group;
	}
	
	/**
	 * @see org.eclipse.jface.preference.IPreferencePage#performOk()
	 */
	public boolean performOk()
	{
		boolean result = contextMappingPreference.writeMappingToPreferenceStore();

		//if the preference was written to the preference store okay, and the context mapping preference
		//  changed, we should send an event that there has been a context handler selection change,
		//  so that the views are notified and can update any UI if need be
		if (result && contextMappingPreference.changed())
		{
			ProfileEvent event = UIPlugin.getDefault().getContextHandlerSelectionChangedEvent(this);
			UIPlugin.getDefault().notifyProfileEventListener(event);
		}
		
		return result;
	}
	
    protected IPreferenceStore doGetPreferenceStore() {
        return HyadesUIPlugin.getInstance().getPreferenceStore();
    }
    
    private void populateLanguagesTable()
    {
		IContextLanguage[] languages = ContextManager.getContextLanguages();
    	
		// sort alphabetically
		Arrays.sort(languages, new Comparator() {
			public int compare(Object o1, Object o2) {
				if (o1 != null && o2 != null) {
					String name1 = ((IContextLanguage)o1).name();
					String name2 = ((IContextLanguage)o2).name();
					if (name1 != null && name2 != null) {
						return name1.compareTo(name2);
					}
					return o1.hashCode() - o2.hashCode();
				}
				return 0;
			}
		});
		
		contextLanguagesTable.removeAll();
		
		for(int i = 0, maxi = languages.length; i < maxi; i++)
		{
			TableItem item = new TableItem(contextLanguagesTable, SWT.NONE);
			item.setData(languages[i]);
			
			ImageDescriptor imageDescriptor = languages[i].icon();
			if(imageDescriptor != null)
				item.setImage(imageDescriptor.createImage());
	
			item.setText(languages[i].name());
		}
		
		contextLanguagesTable.select(0);
    }
    
    private void populateContextProviderTable()
    {
    	IContextLanguage language = (IContextLanguage)contextLanguagesTable.getSelection()[0].getData();
    	
    	IContextHandler[] chs = ContextManager.getContextHandlers(language.contextKey());
    	
		// sort alphabetically
		Arrays.sort(chs, new Comparator() {
			public int compare(Object o1, Object o2) {
				if (o1 != null && o2 != null) {
					String name1 = ((IContextHandler)o1).getContextProvider().getName();
					String name2 = ((IContextHandler)o2).getContextProvider().getName();
					if (name1 != null && name2 != null) {
						return name1.compareTo(name2);
					}
					return o1.hashCode() - o2.hashCode();
				}
				return 0;
			}
		});

    	
		contextProvidersTable.removeAll();
		
		String selectedContextProviderId = contextMappingPreference.getContextProviderId(language.contextKey());
		int selectedIndex = -1;
		
		for(int i = 0, maxi = chs.length; i < maxi; i++)
		{
			TableItem item = new TableItem(contextProvidersTable, SWT.NONE);
			item.setData(chs[i]);
			
			if (chs[i].getContextProvider().getName() != null)
				item.setText(chs[i].getContextProvider().getName());
			else
				item.setText(chs[i].id());
			
			if (chs[i].id().equals(selectedContextProviderId))
				selectedIndex = i;
		}
		
		if (selectedIndex >= 0)
			contextProvidersTable.select(selectedIndex);
    }

    private void populateContextOpenSourceProviderTable()
    {
    	IContextLanguage language = (IContextLanguage)contextLanguagesTable.getSelection()[0].getData();
    	
    	IContextOpenSourceHandler[] coshs = ContextOpenSourceHandlerManager.getContextOpenSourceHandlers(language.contextKey());
    	
		// sort alphabetically
		Arrays.sort(coshs, new Comparator() {
			public int compare(Object o1, Object o2) {
				if (o1 != null && o2 != null) {
					String name1 = ((IContextOpenSourceHandler)o1).getContextOpenSourceProvider().getName();
					String name2 = ((IContextOpenSourceHandler)o2).getContextOpenSourceProvider().getName();
					if (name1 != null && name2 != null) {
						return name1.compareTo(name2);
					}
					return o1.hashCode() - o2.hashCode();
				}
				return 0;
			}
		});

    	
		contextOpenSourceProvidersTable.removeAll();
		
		String selectedContextOpenSourceProviderId = contextMappingPreference.getContextOpenSourceProviderId(language.contextKey());
		int selectedIndex = -1;
		
		for(int i = 0, maxi = coshs.length; i < maxi; i++)
		{
			TableItem item = new TableItem(contextOpenSourceProvidersTable, SWT.NONE);
			item.setData(coshs[i]);

			if (coshs[i].getContextOpenSourceProvider().getName() != null)
				item.setText(coshs[i].getContextOpenSourceProvider().getName());
			else
				item.setText(coshs[i].id());
			
			if (coshs[i].id().equals(selectedContextOpenSourceProviderId))
				selectedIndex = i;
		}
		
		if (selectedIndex >= 0)
			contextOpenSourceProvidersTable.select(selectedIndex);
    }

    private void populateProviderTables()
    {
    	populateContextProviderTable();
    	populateContextOpenSourceProviderTable();
    }

	public void widgetSelected(SelectionEvent e) {
		if(e.widget == contextLanguagesTable)
		{
			if (contextLanguagesTable.getSelection().length > 0)
				populateProviderTables();
		}
		else if(e.widget == contextProvidersTable)
		{
			if (contextProvidersTable.getSelection().length > 0)
				contextMappingPreference.setContextProviderId(
						((IContextLanguage)contextLanguagesTable.getSelection()[0].getData()).contextKey(),
						((IContextHandler)contextProvidersTable.getSelection()[0].getData()).id());
		}
		else if(e.widget == contextOpenSourceProvidersTable)
		{
			if (contextOpenSourceProvidersTable.getSelection().length > 0)
				contextMappingPreference.setContextOpenSourceProviderId(
						((IContextLanguage)contextLanguagesTable.getSelection()[0].getData()).contextKey(),
						((IContextOpenSourceHandler)contextOpenSourceProvidersTable.getSelection()[0].getData()).id());
		}
		
	}

	public void widgetDefaultSelected(SelectionEvent e) {
		widgetSelected(e);
	}

	public void init(IWorkbench workbench) {
		
	}
	
	/**
	 * @see org.eclipse.jface.dialogs.IDialogPage#dispose()
	 */
	public void dispose()
	{
		super.dispose();
	}
}
