/* ***********************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: CSVTraceReportWizard.java,v 1.6 2010/11/29 15:12:11 mreid Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/
package org.eclipse.hyades.trace.ui.internal.reports;


import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.trace.TRCClass;
import org.eclipse.hyades.models.trace.TRCMethod;
import org.eclipse.hyades.models.trace.TRCPackage;
import org.eclipse.hyades.trace.internal.ui.PDLabelProvider;
import org.eclipse.hyades.trace.ui.internal.util.PerftraceUtil;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;

import com.ibm.icu.util.ULocale;

/**
 *
 */
public class CSVTraceReportWizard extends HTMLTraceReportWizard {
	
	public CSVTraceReportWizard() {
		super();
        setWindowTitle(TraceMessages._18); //$NON-NLS-1$
        
        FILE_EXTENSION = "csv";
	}

	/**
     * 
     * @param sel
     * @param fileName
     * @return
     */
    protected boolean export(Object selection, String fileName) {
		
		Writer writer = null;
		try {
			File file = new File(fileName);
			OutputStream outputStream = new FileOutputStream(file.toString(), false);
			ULocale locale = ULocale.getDefault();
			String encoding = "UTF8";
			
			if (locale.getLanguage() == ULocale.JAPANESE.getLanguage()) {
				encoding = "MS932";
			}
			else if (locale.getLanguage() == ULocale.KOREAN.getLanguage()) {
				encoding = "MS949";
			}
			else if (locale.getLanguage() == ULocale.CHINESE.getLanguage()) {
				if (locale.getCountry() == ULocale.TRADITIONAL_CHINESE.getCountry()) {
						encoding = "MS950_HKSCS";	
				}
				else if (locale.getCountry() == ULocale.SIMPLIFIED_CHINESE.getCountry()) {
					encoding = "MS936";
				}
			}
			
			//only windows machines is affected
			if (!System.getProperty("os.name").startsWith("Windows") //$NON-NLS-1$ //$NON-NLS-2$          
					&& !encoding.equals("UTF8"))
			{
				encoding = "UTF8";
			}

			writer = new OutputStreamWriter(outputStream, encoding);
		} catch (IOException ioException) {
			ioException.printStackTrace();			
			return false;
		}
		
		StringBuffer buffer = new StringBuffer();        		
		List agentList = getSelectionAgents(selection);
		generateAgentStatistics(agentList, buffer);
		
		try {
			writer.write(buffer.toString());
			writer.flush();
			writer.close();
		} catch (Exception exception) {
			exception.printStackTrace();
		}
		
		return true;
	}

	protected void generateAgentStatistics(List agentList, StringBuffer buffer) {
		
		PDLabelProvider label = new PDLabelProvider();
		for(int idx=0; idx<agentList.size(); idx++)
		{
			TRCAgentProxy agent = (TRCAgentProxy)agentList.get(idx);
			
     	     buffer.append(newLine);
			
			buffer.append(getCSVFormat(label.getText(agent)));
						
			if(PerftraceUtil.getTotalSize(agent) > 0)
			{//has memory statistics
				generateMemoryStatistics(agent, buffer);				
			}
			
			if (PerftraceUtil.getMaximumCumulativeTime(agent) > 0)
			{//has execution statistics
				generateExecutionStatistics(agent, buffer);				
			}			
		}
		
	}
    
	protected void generateExecutionStatistics(TRCAgentProxy agent, StringBuffer buffer) {
		Object[] packages = PerftraceUtil.getAllPackages(agent);
		HashMap<Double,TRCPackage> map = new HashMap<Double,TRCPackage>();
		
		for(int idx=0;idx<packages.length; idx++)
		{
			TRCPackage pack = (TRCPackage)packages[idx];
			if(pack.getCalls() == 0)
				continue;
			
			map.put(pack.getBaseTime(), pack);			
		}
		
		Object[] keys = map.keySet().toArray();
		double[] time = new double[keys.length];
		for(int idx=0; idx<keys.length; idx++)
			time[idx]=(Double)keys[idx];
		Arrays.sort(time);
		
		buffer.append(newLine).append(getCSVFormat(TraceReportMessages._11));
		
		buffer.append(newLine).append(getCSVFormat(TraceReportMessages._1))
		.append(",").append(">").append(TraceReportMessages._4)
		.append(",").append(TraceReportMessages._5)		
		.append(",").append(TraceReportMessages._7);
		
		for(int idx=time.length-1;idx>=0; idx--)
		{
			double timeVal = time[idx];
			TRCPackage pack = (TRCPackage)map.get(timeVal);
			
			buffer.append(newLine).append(getCSVFormat(PerftraceUtil.getPackageName(pack, agent)));
			buffer.append(",").append(getCSVFormat(PerftraceUtil.formatTimeValue(timeVal)));
			buffer.append(",").append(getCSVFormat(PerftraceUtil.formatTimeValue(pack.getCumulativeTime())));
			buffer.append(",").append(pack.getCalls());
			
			generateClassExecutionStatistics(pack, buffer);
		}
		
	}
	

	protected void generateClassExecutionStatistics(TRCPackage pack, StringBuffer buffer) {
		
		Iterator classes = pack.getClasses().iterator();
		HashMap<Double,TRCClass> map = new HashMap<Double,TRCClass>();
				
		while(classes.hasNext())
		{
			TRCClass cls = (TRCClass)classes.next();
			if(cls.getCalls() == 0)
				continue;
			
			map.put(cls.getBaseTime(), cls);			
		}
		
		Object[] keys = map.keySet().toArray();
		double[] time = new double[keys.length];
		for(int idx=0; idx<keys.length; idx++)
			time[idx]=(Double)keys[idx];
		Arrays.sort(time);
		
		for(int idx=time.length-1;idx>=0; idx--)
		{
			double timeVal = time[idx];
			TRCClass cls = (TRCClass)map.get(timeVal);
			
			buffer.append(newLine).append("    ").append(getCSVFormat(cls.getName()));
			buffer.append(",").append(getCSVFormat(PerftraceUtil.formatTimeValue(timeVal)));
			buffer.append(",").append(getCSVFormat(PerftraceUtil.formatTimeValue(cls.getCumulativeTime())));
			buffer.append(",").append(cls.getCalls());
			
			generateMethodExecutionStatistics(cls, buffer);
		}
		
	}

	protected void generateMethodExecutionStatistics(TRCClass cls, StringBuffer buffer) {
		
		Iterator methods = cls.getMethods().iterator();
		HashMap<Double,TRCMethod> map = new HashMap<Double,TRCMethod>();
				
		while(methods.hasNext())
		{
			TRCMethod meth = (TRCMethod)methods.next();
			if(meth.getCalls() == 0)
				continue;
			
			map.put(meth.getBaseTime(), meth);			
		}
		
		Object[] keys = map.keySet().toArray();
		double[] time = new double[keys.length];
		for(int idx=0; idx<keys.length; idx++)
			time[idx]=(Double)keys[idx];
		Arrays.sort(time);
		
		for(int idx=time.length-1;idx>=0; idx--)
		{
			double timeVal = time[idx];
			TRCMethod meth = (TRCMethod)map.get(timeVal);
			
			buffer.append(newLine)
			  .append("         ").append(getCSVFormat(meth.getName()+meth.getSignature()));
			buffer.append(",").append(getCSVFormat(PerftraceUtil.formatTimeValue(timeVal)));
			buffer.append(",").append(getCSVFormat(PerftraceUtil.formatTimeValue(meth.getCumulativeTime())));
			buffer.append(",").append(meth.getCalls());
		}
	}

	/**
	 * 
	 * @param agent
	 */
	protected void generateMemoryStatistics(TRCAgentProxy agent, StringBuffer buffer) {
		
		Object[] packages = PerftraceUtil.getAllPackages(agent);
		HashMap map = new HashMap();
		
		for(int idx=0;idx<packages.length; idx++)
		{
			TRCPackage pack = (TRCPackage)packages[idx];
			if(pack.getTotalInstances() == 0)
				continue;
			
			map.put(String.valueOf(pack.getTotalSize()), pack);			
		}
		
		Object[] keys = map.keySet().toArray();
		int[] size = new int[keys.length];
		for(int idx=0; idx<keys.length; idx++)
			size[idx]=Integer.parseInt(keys[idx].toString());
		Arrays.sort(size);
		
		buffer.append(newLine).append(getCSVFormat(TraceReportMessages._8)).append(newLine);
		
		buffer.append(TraceReportMessages._1);
		buffer.append(",").append(TraceReportMessages._2);
		buffer.append(",").append(TraceReportMessages._9);		
		buffer.append(",").append(">").append(TraceReportMessages._6);
		buffer.append(",").append(TraceReportMessages._10);
		buffer.append(",").append(TraceReportMessages._3);
		
		for(int idx=size.length-1;idx>=0; idx--)
		{
			TRCPackage pack = (TRCPackage)map.get(String.valueOf(size[idx]));
			
			buffer.append(newLine)
					.append(getCSVFormat(PerftraceUtil.getPackageName(pack, agent)));
			buffer.append(",").append(pack.getTotalInstances());
			buffer.append(",").append(pack.getTotalInstances()-pack.getCollectedInstances());			
			buffer.append(",").append(size[idx]);
			buffer.append(",").append(pack.getTotalSize()-pack.getCollectedSize());
			buffer.append(",").append(pack.getCollectedInstances());
			
			generateClassMemoryStatistics(pack, buffer);
			
		}
	}    
	
	
	protected void generateClassMemoryStatistics(TRCPackage pack, StringBuffer buffer) {
		
		Iterator classes = pack.getClasses().iterator();
		HashMap map = new HashMap();
		
		while(classes.hasNext())
		{
			TRCClass clas = (TRCClass)classes.next();
			if(clas.getTotalInstances() == 0)
				continue;
			
			map.put(String.valueOf(clas.getTotalSize()), clas);			
		}
		
		Object[] keys = map.keySet().toArray();
		int[] size = new int[keys.length];
		for(int idx=0; idx<keys.length; idx++)
			size[idx]=Integer.parseInt(keys[idx].toString());
		Arrays.sort(size);
		
		for(int idx=size.length-1;idx>=0; idx--)
		{
			TRCClass clas = (TRCClass)map.get(String.valueOf(size[idx]));
			
			buffer.append(newLine)
					.append("    ").append(getCSVFormat(clas.getName()));
			buffer.append(",").append(clas.getTotalInstances());
			buffer.append(",").append(clas.getTotalInstances()-clas.getCollectedInstances());			
			buffer.append(",").append(size[idx]);
			buffer.append(",").append(clas.getTotalSize()-clas.getCollectedSize());
			buffer.append(",").append(clas.getCollectedInstances());
		}
		
	}

}
