/* ***********************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: HTMLTraceReportWizard.java,v 1.6 2010/11/29 15:12:11 mreid Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 ************************************************************/
package org.eclipse.hyades.trace.ui.internal.reports;


import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.models.trace.TRCClass;
import org.eclipse.hyades.models.trace.TRCMethod;
import org.eclipse.hyades.models.trace.TRCPackage;
import org.eclipse.hyades.trace.internal.ui.PDLabelProvider;
import org.eclipse.hyades.trace.ui.internal.util.PerftraceUtil;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.hyades.ui.extension.INavigatorItem;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIConstants;

/**
 *
 */
public class HTMLTraceReportWizard extends HTMLTraceStatReportWizard {

	
	public HTMLTraceReportWizard() {
		super();
        setWindowTitle(TraceMessages._16); //$NON-NLS-1$
	}

	public boolean isAvailable(ISelection selection) {
		
		if(selection == null || selection.isEmpty())
			return false;
		
		Object object = ((StructuredSelection)selection).getFirstElement();
		if(object == null)
			return false;
		
		if(object instanceof TRCMonitor)
		{
			TRCMonitor mon = (TRCMonitor) object;
			
			Iterator nodes = mon.getNodes().iterator();
			while(nodes.hasNext())
			{
				TRCNode node = (TRCNode)nodes.next();
				
				Iterator processes = node.getProcessProxies().iterator();
				while(processes.hasNext())
				{				
					TRCProcessProxy proces = (TRCProcessProxy)processes.next();
					Iterator agents = proces.getAgentProxies().iterator();
					while(agents.hasNext())
					{
						TRCAgentProxy ag = (TRCAgentProxy)agents.next();
						if(ag.getType() != null && ag.getType().equals(CommonUIConstants.PROFILE_AGENT_TYPE))
							return true;
					}
				}
			}
		}				
		else if(object instanceof TRCNode)
		{
			TRCNode node = (TRCNode)object;
			
			Iterator processes = node.getProcessProxies().iterator();
			while(processes.hasNext())
			{				
				TRCProcessProxy proces = (TRCProcessProxy)processes.next();
				Iterator agents = proces.getAgentProxies().iterator();
				while(agents.hasNext())
				{
					TRCAgentProxy ag = (TRCAgentProxy)agents.next();
					if(ag.getType() != null && ag.getType().equals(CommonUIConstants.PROFILE_AGENT_TYPE))
						return true;
				}
			}
		}		
		else if(object instanceof TRCProcessProxy)
		{
			TRCProcessProxy proces = (TRCProcessProxy)object;
			Iterator agents = proces.getAgentProxies().iterator();
			while(agents.hasNext())
			{
				TRCAgentProxy ag = (TRCAgentProxy)agents.next();
				if(ag.getType() != null && ag.getType().equals(CommonUIConstants.PROFILE_AGENT_TYPE))
					return true;
			}
		}
		else if(object instanceof TRCAgentProxy && ((TRCAgentProxy)object).getType() != null 
				&& ((TRCAgentProxy)object).getType().equals(CommonUIConstants.PROFILE_AGENT_TYPE))
		{
			return true;
		}
		else if(object instanceof INavigatorItem)
		{
			Object parent = ((INavigatorItem)object).getParent();
			if(parent != null && parent instanceof TRCAgentProxy && ((TRCAgentProxy)parent).getType() != null 
					&& ((TRCAgentProxy)parent).getType().equals(CommonUIConstants.PROFILE_AGENT_TYPE))
				return true;
		}
		
				
		return false;
	}
	
    /**
     * @see org.eclipse.hyades.ui.report.ReportGeneratorWizard#generate(org.eclipse.core.runtime.IProgressMonitor)
     */
    public IFile generate(IProgressMonitor monitor) throws Exception {
    	
    	Object sel = getSelection().getFirstElement();
      
        monitor.beginTask("", 4); //$NON-NLS-1$             
    	
        IFile reportFile = getReportFile();        
        monitor.worked(1);
        export(sel, reportFile.getLocation().toOSString());
        monitor.worked(1);

        reportFile.getProject().refreshLocal(IResource.DEPTH_INFINITE, null);
        monitor.done();
           
        return reportFile;
    }
	
    /**
     * 
     * @param sel
     * @param fileName
     * @return
     */
    protected boolean export(Object selection, String fileName) {
		
		Writer writer = null;
		try {
			File file = new File(fileName);
			OutputStream outputStream = new FileOutputStream(file.toString(), false);
			writer = new OutputStreamWriter(outputStream, "UTF8");
		} catch (IOException ioException) {
			ioException.printStackTrace();			
			return false;
		}
		
		StringBuffer buffer = new StringBuffer();        
		buffer.append(printHeader());
		buffer.append(printTitle((new PDLabelProvider()).getText(selection)));
		
		List agentList = getSelectionAgents(selection);
		generateAgentStatistics(agentList, buffer);
		buffer.append(printFooter());
		
		try {
			writer.write(buffer.toString());
			writer.flush();
			writer.close();
		} catch (Exception exception) {
			exception.printStackTrace();
		}
		
		return true;
	}

	protected void generateAgentStatistics(List agentList, StringBuffer buffer) {
		
		PDLabelProvider label = new PDLabelProvider();
		for(int idx=0; idx<agentList.size(); idx++)
		{
			TRCAgentProxy agent = (TRCAgentProxy)agentList.get(idx);
			printTitle(label.getText(agent));
						
			if(PerftraceUtil.getTotalSize(agent) > 0)
			{//has memory statistics
				generateMemoryStatistics(agent, buffer);				
			}
			
			if (PerftraceUtil.getMaximumCumulativeTime(agent)> 0)
			{//has execution statistics
				generateExecutionStatistics(agent, buffer);				
			}			
		}
		
	}

	protected void generateExecutionStatistics(TRCAgentProxy agent, StringBuffer buffer) {
		Object[] packages = PerftraceUtil.getAllPackages(agent);
		HashMap<Double,TRCPackage> map = new HashMap<Double,TRCPackage>();
		
		for(int idx=0;idx<packages.length; idx++)
		{
			TRCPackage pack = (TRCPackage)packages[idx];
			if(pack.getCalls() == 0)
				continue;
			
			map.put(pack.getBaseTime(), pack);			
		}
		
		Object[] keys = map.keySet().toArray();
		double[] time = new double[keys.length];
		for(int idx=0; idx<keys.length; idx++)
			time[idx]=(Double)keys[idx];
		Arrays.sort(time);
		
		buffer.append("<p>").append(printTitle(TraceReportMessages._11)).append("<p>");
		
		buffer.append("<table border=0 cellspacing=2 cellpadding=2 WIDTH=\"100%\">").append(newLine);
		buffer.append("<tr width=\"10\">").append(newLine);

		buffer.append("<th align=").append(getAlign()).append(">").append(TraceReportMessages._1).append("</th>").append(newLine);
		buffer.append("<th align=").append(getAlign()).append(">").append("&lt;").append(TraceReportMessages._4).append("</th>").append(newLine);
		buffer.append("<th align=").append(getAlign()).append(">").append(TraceReportMessages._5).append("</th>").append(newLine);		
		buffer.append("<th align=").append(getAlign()).append(">").append(TraceReportMessages._7).append("</th>").append(newLine);
		
		buffer.append("</tr>").append(newLine);
		boolean color = true;

		for(int idx=time.length-1;idx>=0; idx--)
		{
			double timeVal = time[idx];
			TRCPackage pack = (TRCPackage)map.get(timeVal);
			
			if(color)  
			    buffer.append("<tr BGCOLOR=\"#ffffbb\">").append(newLine);
			else   
				buffer.append("<tr>" + newLine);
			
			buffer.append("<td align=").append(getAlign()).append(">")
					.append(PerftraceUtil.getPackageName(pack, agent))
					.append("</td>");
			buffer.append("<td align=").append(getAlign()).append(">")
				.append(PerftraceUtil.formatTimeValue(timeVal))
				.append("</td>");
			buffer.append("<td align=").append(getAlign()).append(">")
				.append(PerftraceUtil.formatTimeValue(pack.getCumulativeTime()))
				.append("</td>");			
			buffer.append("<td align=").append(getAlign()).append(">")
				.append(pack.getCalls())
				.append("</td>");			
			
			buffer.append(newLine);			
			color = !color;
			
			buffer.append("</tr>").append(newLine);		
			
			generateClassExecutionStatistics(pack, buffer, color);
		}
		
		buffer.append("</table>").append(newLine);
	}

	protected void generateClassExecutionStatistics(TRCPackage pack, StringBuffer buffer, boolean color) {
		
		Iterator classes = pack.getClasses().iterator();
		HashMap<Double,TRCClass> map = new HashMap<Double,TRCClass>();
		
		while(classes.hasNext())
		{
			TRCClass cls = (TRCClass)classes.next();
			if(cls.getCalls() == 0)
				continue;
			
			map.put(cls.getBaseTime(), cls);
		}
		
		Object[] keys = map.keySet().toArray();
		double[] time = new double[keys.length];
		for(int idx=0; idx<keys.length; idx++)
			time[idx]=(Double)keys[idx];
		Arrays.sort(time);
		
		for(int idx=time.length-1;idx>=0; idx--)
		{
			Double timeVal = time[idx];
			TRCClass cls = (TRCClass)map.get(timeVal);
			
			if(color)  
			    buffer.append("<tr BGCOLOR=\"#ffffbb\">").append(newLine);
			else   
				buffer.append("<tr>" + newLine);
			
			buffer.append("<td align=").append(getAlign()).append(">")
					.append("&nbsp;&nbsp;&nbsp;&nbsp;").append(cls.getName())
					.append("</td>");
			buffer.append("<td align=").append(getAlign()).append(">")
				.append("&nbsp;&nbsp;&nbsp;&nbsp;").append(PerftraceUtil.formatTimeValue(timeVal))
				.append("</td>");
			buffer.append("<td align=").append(getAlign()).append(">")
				.append("&nbsp;&nbsp;&nbsp;&nbsp;").append(PerftraceUtil.formatTimeValue(cls.getCumulativeTime()))
				.append("</td>");			
			buffer.append("<td align=").append(getAlign()).append(">")
				.append("&nbsp;&nbsp;&nbsp;&nbsp;").append(cls.getCalls())
				.append("</td>");			
			
			buffer.append(newLine);			
			color = !color;
			
			buffer.append("</tr>").append(newLine);		
			
			generateMethodExecutionStatistics(cls, buffer, color);
			
		}
	}

	protected void generateMethodExecutionStatistics(TRCClass cls, StringBuffer buffer, boolean color) {
		Iterator methods = cls.getMethods().iterator();
		HashMap<Double,TRCMethod> map = new HashMap<Double,TRCMethod>();
				
		while(methods.hasNext())
		{
			TRCMethod meth = (TRCMethod)methods.next();
			if(meth.getCalls() == 0)
				continue;
			
			map.put(meth.getBaseTime(), meth);			
		}
		
		Object[] keys = map.keySet().toArray();
		double[] time = new double[keys.length];
		for(int idx=0; idx<keys.length; idx++)
			time[idx]=(Double)keys[idx];
		Arrays.sort(time);

		for(int idx=time.length-1;idx>=0; idx--)
		{
			double timeVal = time[idx];
			TRCMethod meth = (TRCMethod)map.get(timeVal);

			if(color)  
			    buffer.append("<tr BGCOLOR=\"#ffffbb\">").append(newLine);
			else   
				buffer.append("<tr>" + newLine);
			
			buffer.append("<td align=").append(getAlign()).append(">")
					.append("&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;").append(meth.getName()).append(meth.getSignature())
					.append("</td>");
			buffer.append("<td align=").append(getAlign()).append(">")
				.append("&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;").append(PerftraceUtil.formatTimeValue(timeVal))
				.append("</td>");
			buffer.append("<td align=").append(getAlign()).append(">")
				.append("&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;").append(PerftraceUtil.formatTimeValue(meth.getCumulativeTime()))
				.append("</td>");			
			buffer.append("<td align=").append(getAlign()).append(">")
				.append("&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;").append(meth.getCalls())
				.append("</td>");			
			
			buffer.append(newLine);			
			color = !color;
			
			buffer.append("</tr>").append(newLine);		
		}

	}

	/**
	 * 
	 * @param object
	 * @return
	 */
	protected List getSelectionAgents(Object object) {
		
		List agentList = new ArrayList();
		if(object instanceof TRCMonitor)
		{
			TRCMonitor mon = (TRCMonitor) object;
			
			Iterator nodes = mon.getNodes().iterator();
			while(nodes.hasNext())
			{
				TRCNode node = (TRCNode)nodes.next();
				
				Iterator processes = node.getProcessProxies().iterator();
				while(processes.hasNext())
				{				
					TRCProcessProxy proces = (TRCProcessProxy)processes.next();
					Iterator agents = proces.getAgentProxies().iterator();
					while(agents.hasNext())
					{
						TRCAgentProxy ag = (TRCAgentProxy)agents.next();
						if(ag.getType() != null && ag.getType().equals(CommonUIConstants.PROFILE_AGENT_TYPE))
							agentList.add(ag);
					}
				}
			}
		}				
		else if(object instanceof TRCNode)
		{
			TRCNode node = (TRCNode)object;
			
			Iterator processes = node.getProcessProxies().iterator();
			while(processes.hasNext())
			{				
				TRCProcessProxy proces = (TRCProcessProxy)processes.next();
				Iterator agents = proces.getAgentProxies().iterator();
				while(agents.hasNext())
				{
					TRCAgentProxy ag = (TRCAgentProxy)agents.next();
					if(ag.getType() != null && ag.getType().equals(CommonUIConstants.PROFILE_AGENT_TYPE))
						agentList.add(ag);
				}
			}
		}		
		else if(object instanceof TRCProcessProxy)
		{
			TRCProcessProxy proces = (TRCProcessProxy)object;
			Iterator agents = proces.getAgentProxies().iterator();
			while(agents.hasNext())
			{
				TRCAgentProxy ag = (TRCAgentProxy)agents.next();
				if(ag.getType() != null && ag.getType().equals(CommonUIConstants.PROFILE_AGENT_TYPE))
					agentList.add(ag);
			}
		}
		else if(object instanceof TRCAgentProxy && ((TRCAgentProxy)object).getType() != null 
				&& ((TRCAgentProxy)object).getType().equals(CommonUIConstants.PROFILE_AGENT_TYPE))
		{
			agentList.add(object);
		}
		else if(object instanceof INavigatorItem)
		{
			Object parent = ((INavigatorItem)object).getParent();
			if(parent != null && parent instanceof TRCAgentProxy && ((TRCAgentProxy)parent).getType() != null 
					&& ((TRCAgentProxy)parent).getType().equals(CommonUIConstants.PROFILE_AGENT_TYPE))
				agentList.add(parent);
		}
		
		return agentList;
	}
    
	/**
	 * 
	 * @param agent
	 */
	protected void generateMemoryStatistics(TRCAgentProxy agent, StringBuffer buffer) {
		
		Object[] packages = PerftraceUtil.getAllPackages(agent);
		HashMap map = new HashMap();
		
		for(int idx=0;idx<packages.length; idx++)
		{
			TRCPackage pack = (TRCPackage)packages[idx];
			if(pack.getTotalInstances() == 0)
				continue;
			
			map.put(String.valueOf(pack.getTotalSize()), pack);			
		}
		
		Object[] keys = map.keySet().toArray();
		int[] size = new int[keys.length];
		for(int idx=0; idx<keys.length; idx++)
			size[idx]=Integer.parseInt(keys[idx].toString());
		Arrays.sort(size);
		
		buffer.append("<p>").append(printTitle(TraceReportMessages._8)).append("<p>");
		
		buffer.append("<table border=0 cellspacing=2 cellpadding=2 WIDTH=\"100%\">").append(newLine);
		buffer.append("<tr width=\"10\">").append(newLine);

		buffer.append("<th align=").append(getAlign()).append(">").append(TraceReportMessages._1).append("</th>").append(newLine);
		buffer.append("<th align=").append(getAlign()).append(">").append(TraceReportMessages._2).append("</th>").append(newLine);
		buffer.append("<th align=").append(getAlign()).append(">").append(TraceReportMessages._9).append("</th>").append(newLine);		
		buffer.append("<th align=").append(getAlign()).append(">").append("&lt;").append(TraceReportMessages._6).append("</th>").append(newLine);
		buffer.append("<th align=").append(getAlign()).append(">").append(TraceReportMessages._10).append("</th>").append(newLine);
		buffer.append("<th align=").append(getAlign()).append(">").append(TraceReportMessages._3).append("</th>").append(newLine);
		
		buffer.append("</tr>").append(newLine);
		boolean color = true;

		for(int idx=size.length-1;idx>=0; idx--)
		{
			TRCPackage pack = (TRCPackage)map.get(String.valueOf(size[idx]));
			
			if(color)  
			    buffer.append("<tr BGCOLOR=\"#ffffbb\">").append(newLine);
			else   
				buffer.append("<tr>" + newLine);
			
			buffer.append("<td align=").append(getAlign()).append(">")
					.append(PerftraceUtil.getPackageName(pack, agent))
					.append("</td>");
			buffer.append("<td align=").append(getAlign()).append(">")
				.append(pack.getTotalInstances())
				.append("</td>");
			buffer.append("<td align=").append(getAlign()).append(">")
				.append(pack.getTotalInstances()-pack.getCollectedInstances())
				.append("</td>");			
			buffer.append("<td align=").append(getAlign()).append(">")
				.append(size[idx])
				.append("</td>");
			buffer.append("<td align=").append(getAlign()).append(">")
				.append(pack.getTotalSize()-pack.getCollectedSize())
				.append("</td>");
			buffer.append("<td align=").append(getAlign()).append(">")
				.append(pack.getCollectedInstances())
				.append("</td>");
			
			
			buffer.append(newLine);			
			color = !color;
			
			buffer.append("</tr>").append(newLine);		
			
			generateClassMemoryStatitics(pack, buffer, color);
			
		}
		
		buffer.append("</table>").append(newLine);		
	}

	protected void generateClassMemoryStatitics(TRCPackage pack, StringBuffer buffer, boolean color) {

		Iterator classes = pack.getClasses().iterator();
		HashMap map = new HashMap();
		
		while(classes.hasNext())
		{
			TRCClass clas = (TRCClass)classes.next();
			if(clas.getTotalInstances() == 0)
				continue;
			
			map.put(String.valueOf(clas.getTotalSize()), clas);			
		}
		
		Object[] keys = map.keySet().toArray();
		int[] size = new int[keys.length];
		for(int idx=0; idx<keys.length; idx++)
			size[idx]=Integer.parseInt(keys[idx].toString());
		Arrays.sort(size);

		for(int idx=size.length-1;idx>=0; idx--)
		{
			TRCClass clas = (TRCClass)map.get(String.valueOf(size[idx]));
						
			if(color)  
			    buffer.append("<tr BGCOLOR=\"#ffffbb\">").append(newLine);
			else   
				buffer.append("<tr>" + newLine);
			
			buffer.append("<td align=").append(getAlign()).append(">")
					.append("&nbsp;&nbsp;&nbsp;&nbsp;").append(clas.getName())
					.append("</td>");
			buffer.append("<td align=").append(getAlign()).append(">")
				.append("&nbsp;&nbsp;&nbsp;&nbsp;").append(clas.getTotalInstances())
				.append("</td>");
			buffer.append("<td align=").append(getAlign()).append(">")
				.append("&nbsp;&nbsp;&nbsp;&nbsp;").append(clas.getTotalInstances()-clas.getCollectedInstances())
				.append("</td>");			
			buffer.append("<td align=").append(getAlign()).append(">")
				.append("&nbsp;&nbsp;&nbsp;&nbsp;").append(size[idx])
				.append("</td>");
			buffer.append("<td align=").append(getAlign()).append(">")
				.append("&nbsp;&nbsp;&nbsp;&nbsp;").append(clas.getTotalSize()-clas.getCollectedSize())
				.append("</td>");
			buffer.append("<td align=").append(getAlign()).append(">")
				.append("&nbsp;&nbsp;&nbsp;&nbsp;").append(clas.getCollectedInstances())
				.append("</td>");
			
			
			buffer.append(newLine);			
			color = !color;
			
			buffer.append("</tr>").append(newLine);		
			
		}
		
	}    
	
}
