/*******************************************************************************
 * Copyright (c) 2011 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.trace.ui.internal.util;

import java.io.File;

import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.internal.execution.local.control.Application;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.security.internal.util.BaseConnectUtil;
import org.eclipse.hyades.security.internal.util.NullConnectUtilUI;
import org.eclipse.tptp.platform.execution.client.core.ACPathInfo;
import org.eclipse.tptp.platform.execution.client.core.IAgentController;
import org.eclipse.tptp.platform.execution.client.core.internal.AgentController;
import org.eclipse.tptp.platform.execution.util.internal.AgentControllerPool;
import org.osgi.framework.Bundle;

public class AgentControllerProbe implements Application {

	public enum tribool {
	
		UNKNOWN,
		FALSE,
		TRUE;
		
		public String yesno() {
			switch( this ) {
			case TRUE:
				return "Yes"; //$NON-NLS-1$
			case FALSE:
				return "No"; //$NON-NLS-1$
			case UNKNOWN:
				return "Unknown"; //$NON-NLS-1$
			}
			
			throw new IllegalArgumentException("Incomplete switch statement"); //$NON-NLS-1$
		}
	}
	
	public static class Result {
	
		public final String  host;
		public final int     port;
		
		public final boolean running;
		public final tribool secure;
		public final boolean legacyOnly;
		public final tribool isIac;
		public final String  acHome;
	
		public Result( String host, int port, boolean running, tribool secure, boolean legacyOnly, tribool isIac, String acHome ) {
			this.host = host;
			this.port = port;
			this.running = running;
			this.secure = secure;
			this.legacyOnly = legacyOnly;
			this.isIac = isIac;
			this.acHome = acHome;
		}

		public String toString() {
			
			return "running="+running + ", secure=" + secure + ", legacyOnly=" + legacyOnly + ", isIac=" + isIac + ", acHome=" + acHome;  //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
			
		}
		
	}
	
	private String userid;
	private Application app;
	
	public AgentControllerProbe(String userid, Application app) {
		
		this.userid = userid;
		this.app = (null == app) ? this : app;
		
	}
	
	public Result runProbe( String host, int port ) {

		Node legacyNode = connectLegacy(host, port);
		boolean legacy = (null != legacyNode);
		tribool secured = tribool.UNKNOWN;
		
		if( legacy )
			secured = (null != legacyNode.getSecurityParameters()) ? tribool.TRUE : tribool.FALSE;
		
		AgentController ac = (AgentController)connectNew(host, port);
		if( null != ac ) {

			try {
				if( !ac.isAgentPathSupported() )
					return new Result( host, port, true, secured, false, tribool.UNKNOWN, null );
					
				ACPathInfo acPath = ac.queryACPath();
				String acHome = acPath.getAcPath();

				return new Result( host, port, true, secured, false, isIAC(acHome) ? tribool.TRUE : tribool.FALSE, acHome );
									
			} catch( Exception e ) {
				
				// Exception thrown during AC path bits; tell them what we know
				return new Result( host, port, true, secured, false, tribool.UNKNOWN, null );
				
			} finally {
				ac.disconnect();
			}
			
		} else {
			
			// Failed to make a new connection; either we are legacy only (in which
			// case we know its not the IAC) or we failed entirely
			return new Result( host, port, legacy, secured, legacy, legacy ? tribool.FALSE : tribool.UNKNOWN, null );
			
		}
		
	}

	private Node connectLegacy( String hostName, int port ) {
	
		BaseConnectUtil hyadesUtil = new BaseConnectUtil(hostName, "" + port, this.userid, this.app); //$NON-NLS-1$
		
		hyadesUtil.setConnectionUI( new NullConnectUtilUI() );
		
		int result = hyadesUtil.connect("password", false); //$NON-NLS-1$
		if( !(BaseConnectUtil.CONNECTION_SUCCESS == result 
			  || BaseConnectUtil.CONNECTION_AUTHENTICATION_FAILED == result
			  || BaseConnectUtil.CONNECTION_AUTHENTICATION_SUCCESS == result) ) {
			return null;
		}
		
		return hyadesUtil.getNode();
		
	}
	
	private IAgentController connectNew( String hostName, int port ) {
		
		try {

			return AgentControllerPool.getInstance().getConnection(hostName, port, true);
			
		} catch(Exception e) {
			return null;
		}

	}

	private boolean isIAC( String home ) {
		
		Bundle iacAdmin = Platform.getBundle("org.eclipse.tptp.platform.iac.administrator"); //$NON-NLS-1$
		String iacHome = Platform.getStateLocation(iacAdmin).toString()
			.replace('/', File.separatorChar); // IPath#toString always uses '/', but `home` will use the native file separator
		
		if( home.equals( iacHome ) )
			return true;
		else
			return false;
		
	}

	public String getName() {
		return this.getClass().getName();
	}
	
}
