/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: FilterSetElement.java,v 1.4 2010/08/22 22:10:13 jwest Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.util;

import java.util.*;

import org.eclipse.hyades.trace.ui.UIPlugin;

/**
 * A filter set containing one or more filter criteria. Each filter set has an ID, which
 * is either provided as a default, or a dynamic one is generated for sets added by
 * the user. In order to translate properly, the default sets contain a key which points
 * to the resource bundle property with the name.
 */
public class FilterSetElement
{
	private String fId;
    private String fName;
	private String fKey;
	private ArrayList fChildren;
	
	/** Some filters may be auto-populated at a later stage in the launch process (Bug 323330) */
	private boolean isFilterAutoPopulated = false; 

	/**
	 * Constructs a new FilterSetElement with the given unique identifier.
	 * 
	 * @param id the unique language-neutral identifier.
	 */
	public FilterSetElement(String id) {
		fId = id;
	}
	
	/**
	 * Returns the unique language-neutral identifier of this filter set.
	 * 
	 * @return the filter set's ID.
	 */
	public String getId() {
		return fId;
	}
	
	/**
	 * Returns the string containing the name to be displayed for this filter
	 * set. The difference with getName() is that this one will lookup the key
	 * for the default sets. Default filter sets must store the key to the property
	 * string, because they can be translated (unless the user renames them).
	 *  
	 * @return the name to be displayed for this filter set.
	 */
	public String getDisplayName() {
		if (fName != null) {
			return fName;
		}
		else if (fKey != null) {
			return UIPlugin.getResourceString(fKey);
		}
		return null;
	}
	
	/**
	 * Returns the resource bundle key for this filter set's name. This
	 * is used for default sets only. They must be properly translated, even
	 * when the language is changed, so it is necessary to store the key. If
	 * the key is present, the name should not be used, but looked up using this
	 * key.
	 * 
	 * @return the resource bundle key for this filter's name.
	 */
	public String getKey() {
		return fKey;
	}
	
	/**
	 * Returns the name field for this filter set. This does not lookup the translated
	 * name for the default sets. To get the display name, use getDisplayName() instead.
	 * 
	 * @return the name field for this filter set.
	 */
	public String getName() {
		return fName;
	}
	
	/**
	 * Sets this filter set's unique language-neutral identifier.
	 * 
	 * @param id the identifier to use; any string.
	 */
	public void setId(String id) {
		fId = id;
	}
	
	/**
	 * Set the resource bundle key for this filter set's name. This should
	 * only be used if there is no name field set.
	 * 
	 * @param key the resource bundle key for this filter set's name.
	 */
	public void setKey(String key) {
		fKey = key;
	}
	
	/**
	 * Sets the filter set's name. This will override the default translated name for
	 * default sets. New names assigned by the user cannot be translated.
	 * 
	 * @param name the new name for this filter set.
	 */
	public void setName(String name) {
		fName= name;
	}
	
	/**
	 * Returns the filter elements contained by this filter set. This comprises the
	 * class/method patterns, and also the directive (include/exclude).
	 * 
	 * @return the filter set's elements, or children.
	 */
	public ArrayList getChildren() {
		return fChildren;		
	}
	
	/**
	 * Assigns the given patterns and directives to this filter set.
	 * 
	 * @param children the filter set elements to assign.
	 */
	public void setChildren(ArrayList children)	{
		fChildren = children;
	}
	
	public void setFilterAutoPopulated(boolean isFilterAutopopulated) {
		this.isFilterAutoPopulated = isFilterAutopopulated;
	}
	
	public boolean isFilterAutoPopulated() {
		return isFilterAutoPopulated;
	}
}
