/**********************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: PDCoreUtil.java,v 1.53 2009/05/06 12:00:47 paules Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.trace.ui.internal.util;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Map;
import java.util.Vector;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.model.IProcess;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.internal.execution.local.common.CustomCommand;
import org.eclipse.hyades.internal.execution.local.common.Options;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.AgentConfiguration;
import org.eclipse.hyades.internal.execution.local.control.AgentConfigurationEntry;
import org.eclipse.hyades.internal.execution.local.control.AgentFactory;
import org.eclipse.hyades.internal.execution.local.control.AgentListener;
import org.eclipse.hyades.internal.execution.local.control.InactiveAgentException;
import org.eclipse.hyades.internal.execution.local.control.InactiveProcessException;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.NodeFactory;
import org.eclipse.hyades.internal.execution.local.control.NotConnectedException;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.internal.execution.local.control.ProcessFactory;
import org.eclipse.hyades.internal.execution.local.control.ProcessListener;
import org.eclipse.hyades.internal.execution.local.control.Variable;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.models.hierarchy.HierarchyFactory;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCCollectionMode;
import org.eclipse.hyades.models.hierarchy.TRCConfiguration;
import org.eclipse.hyades.models.hierarchy.TRCFilter;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCOption;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.models.hierarchy.util.SaveUtil;
import org.eclipse.hyades.models.util.ModelDebugger;
import org.eclipse.hyades.security.internal.util.ConnectUtil;
import org.eclipse.hyades.trace.internal.ui.PDPerspective;
import org.eclipse.hyades.trace.internal.ui.PDProjectExplorer;
import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.ProfileUIManager;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.launcher.IProfileLaunchConfigurationConstants;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfilingSetsManager;
import org.eclipse.hyades.trace.ui.internal.launcher.TraceArguments;
import org.eclipse.hyades.trace.ui.internal.piclient.PIAttachListener;
import org.eclipse.hyades.trace.ui.internal.piclient.PIProcessListener;
import org.eclipse.hyades.trace.ui.internal.piclient.XMLTraceDataProcessor;
import org.eclipse.hyades.trace.ui.internal.wizard.TraceWizardMessages;
import org.eclipse.hyades.ui.HyadesUI;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Display;
import org.eclipse.tptp.platform.common.internal.CommonPlugin;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIConstants;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIPlugin;
import org.eclipse.tptp.platform.execution.client.agent.IAgent;
import org.eclipse.tptp.platform.execution.client.core.internal.IProcessListener;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherConstants;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;
import org.eclipse.tptp.trace.ui.internal.launcher.deleg.application.PIDelegateHelper.PeriodicPoll;
import org.eclipse.tptp.trace.ui.provisional.launcher.AgentSelection;
import org.eclipse.tptp.trace.ui.provisional.launcher.PrimaryLaunchDelegate;
import org.eclipse.tptp.trace.ui.provisional.launcher.ProcessParameters;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;

import com.ibm.icu.text.DateFormat;


public class PDCoreUtil extends org.eclipse.tptp.platform.common.ui.trace.internal.util.PDCoreUtil 
{
	
	/* The modes that the PI agent can run in */
    public static final byte PI_MODE_CONTROLLED = 0x01;
    public static final byte PI_MODE_APPLICATION = 0x02;
    public static final byte PI_MODE_STANDALONE = 0x03;
    public static final byte PI_MODE_ENABLED = 0x04;
	
    
    public static final Vector attach(AttachConfigurationDetail configurationDetail)
    {
       	if (configurationDetail.isSwitchPerspective())
    		switchPerspective();

       	String hostName = configurationDetail.getHostName();
        try {
            Node fnode = NodeFactory.createNode(hostName);

            hostName = fnode.getInetAddress().getHostName();
        } catch (UnknownHostException exc) {
        }

        //create the monitor
        IContainer container = createContainer(new Path(configurationDetail.getProject()));
        TRCMonitor monitor = createMonitor(container, configurationDetail.getMonitor());

        //create the node   
        TRCNode node = createNode(monitor, hostName, configurationDetail.getPortNumber());       
        Vector trcAgentList = new Vector();

        Vector agents = configurationDetail.getAgents();
               
        /* For each agent */
        for (int idx = 0; idx < agents.size(); idx++) 
        {
        	Agent oldAgent = null;
        	IAgent newAgent = null;
        	
        	if (agents.elementAt(idx) instanceof Agent)
        		oldAgent = (Agent)agents.elementAt(idx);        	
        	else if (agents.elementAt(idx) instanceof IAgent)
        		newAgent = (IAgent)agents.elementAt(idx);        	
        	else
        		continue;
        	            
            
        	
            final IProcess adapter = new ProcessAdapter(oldAgent == null ? (Object)newAgent.getProcess() : oldAgent.getProcess(), configurationDetail.getLaunch(), true, 0);
    		UIPlugin.getDefault().registerLaunchProcess(adapter);
            
    		if (oldAgent == null)
    		{
	    		newAgent.getProcess().addProcessListener(new IProcessListener() 
	            {
					public void processLaunched(org.eclipse.tptp.platform.execution.client.core.IProcess arg0)
					{					
					}
					
					public void processExited(org.eclipse.tptp.platform.execution.client.core.IProcess arg0)
					{
						notifyProcessExit (adapter);						
					}
	            });
    		}
    		else
    		{
    			oldAgent.getProcess().addProcessListener(new ProcessListener()
    			{
    				public void processLaunched(Process arg0)
					{											
					}
    				
					public void processExited(Process arg0)
					{
						notifyProcessExit (adapter);
					}    				
    			});
    		}
    		
    		/* Configure the agent */
    		TRCProcessProxy process = null;
    		if (oldAgent != null)
    			configureAgent (configurationDetail, oldAgent);
    		
    		process = createProcess(node, oldAgent == null ? (Object)newAgent.getProcess() : oldAgent.getProcess());
    		LauncherUtility.addLaunchConfigId(process, LauncherConstants.CONFIG_TYPE_IMAGE_ATTACH);
    		process.eAdapters().add(new ProcessProxyAdapter());
	
	        /* add the mapping between TRCProcessProxy -> IProcess */
	        ProcessMap.put(process, adapter);
	         
	        try
	        {
	        	TRCAgentProxy agentProxy = oldAgent == null ? 
	         			attachToAgent (process, newAgent,  configurationDetail.getOptions(), true) :
	         			attachToAgent (process, oldAgent,  configurationDetail.getOptions(), true);
	         	trcAgentList.add(agentProxy);
	         }
	         catch (Exception e)
	         {
	         	UIPlugin.getDefault().log(e);
	         }	
        }

        return trcAgentList;
    }
       

	private static void configureAgent(AttachConfigurationDetail configurationDetail, Agent agent)
	{
        AgentConfiguration ac = agent.getConfiguration();
        if (ac == null) 
        {
            agent.setConfiguration(new AgentConfiguration());
        }

        if (configurationDetail.isForceAgentConfClear())
        	agent.getConfiguration().clear();
        
        String profileFile = configurationDetail.getProfileFile();
        agent.setProfileFile(profileFile == null || profileFile.trim().length() <= 0 ? "" : profileFile);
                
        if (configurationDetail.getFilters() != null)
        	addDefaultProfileData(agent, configurationDetail.getFilters());
        
        Vector options = configurationDetail.getOptions();
        if (options == null)
        	options = new Vector();
        
        addDefaultOptionsData(agent, options);		
	}
	
	private static void notifyProcessExit(IProcess adapter)
    {
		UIPlugin.getDefault().deregisterLaunchProcess(adapter);
		DebugPlugin.getDefault().fireDebugEventSet(new DebugEvent[] 
		{
   			new DebugEvent(adapter, DebugEvent.TERMINATE)
		});
    }
    
    /**
     * Equivalent to attachToAgent (process, agent, null, false)
     * 
     */
    public static TRCAgentProxy attachToAgent (TRCProcessProxy process, Agent agent) throws InactiveAgentException, InactiveProcessException
    {
    	return attachToAgent (process, agent, null, false);
    }
    
    
    /**
     * Attach to the agent passed in.
     * 
     * @param process The model entity representing the process
     * @param agent The agent to attach to
     * @param options The options set for this attach session (can be null if 'mayNeedPolling' is set to false)
     * @param mayNeedPolling Indicates whether polling may be needed.  Regular polling 
     * is required for the aggregated JVMPI agent.
     * @return The model entity representing the agent
     */
    public static TRCAgentProxy attachToAgent (TRCProcessProxy process, Object agent, Vector options, boolean mayNeedPolling) throws InactiveAgentException, InactiveProcessException
    {
    	TRCAgentProxy agentProxy = createUniqueAgent(process, agent);

        activate(agentProxy);        
        final ProfileEvent event = UIPlugin.getDefault().getProfileEvent();
        event.setSource(agentProxy);
        event.setType(ProfileEvent.ATTACH);

        Display.getDefault().syncExec(new Runnable() {
			public void run() {
                UIPlugin.getDefault().notifyProfileEventListener(event);
			}
        });
        	        	
        return agentProxy;
    }
     
	public static final void detachAgent(TRCAgentProxy agent) throws InactiveAgentException, InactiveProcessException {
		detachAgent(agent, true);
	}

	public static final void detachAgent(TRCAgentProxy agent,boolean deregistrate) throws InactiveAgentException, InactiveProcessException {
        TRCProcessProxy process = agent.getProcessProxy();
        String hostName = process.getNode().getName();
        Node node = PDCoreUtil.profileConnect(hostName, String.valueOf(process.getNode().getPort()));

        if (node != null) {
            Object instance = LoadersUtils.locateAgentInstance(agent);

            if ((instance != null) && instance instanceof Agent) {
                Agent a = (Agent) instance;
				AgentListener al = (AgentListener)LoadersUtils.locateAgentListener(agent);
				a.removeAgentListener(al);

                if (a.isAttached()) {
                    a.detach();
                }
            }
        }

        agent.setMonitored(false);
        agent.setAttached(false);

        LoadersUtils.deregisterDataProcessor(agent);
//		LoadersUtils.deregisterAgentInstance(agent);
//		LoadersUtils.deregisterAgentListener(agent);

        if (deregistrate) {
         LoadersUtils.deregisterAgentProxyContext(agent);
        }
        
        if (agent.isToProfileFile()) {
            XMLTraceDataProcessor processor = (XMLTraceDataProcessor) LoadersUtils.locateDataProcessor(agent);

            if (processor != null) {
                processor.endTrace(processor.getWriter());
                processor.setProfileFileName(null);

                //					  processor.setProfilingFile(null);
                processor.setWriter(null);
            }
        }
    }    




    /**
     * Equivalent to doAttach (null, agentProxy, agent, null)
     */
    public static void doAttach(Agent agent, TRCAgentProxy agentProxy) throws InactiveAgentException, InactiveProcessException 
    {
    	doAttach (null, agentProxy, agent, null, null);
    }
    	
    
    
    /**
     * Attaches to the execution agent (i.e. 'agent') that is passed in.
     * 
     * @param trcProcessProxy The process proxy (Can be null if polling is not required)
     * @param agentProxy The agent proxy
     * @param agent The execution agent
     * @param options The options for this attach session (Can be null if polling is not required)
     * 
     * @throws InactiveAgentException In case of an inactive agent
     * @throws InactiveProcessException In case of an inactive process
     */
	public static void doAttach(TRCProcessProxy trcProcessProxy, TRCAgentProxy agentProxy, Agent agent, Vector options, ILaunchConfiguration configuration) throws InactiveAgentException, InactiveProcessException
	{
        PIProcessListener processListener = (PIProcessListener) LoadersUtils.locateAgentListener(agentProxy);

        if (processListener == null) {
            processListener = new PIAttachListener(agentProxy.getProcessProxy().getNode(), agentProxy.getProcessProxy());
            LoadersUtils.registerAgentListener(agentProxy, processListener);
        }
        
        /* Determine if we need to do regular polling */
        if (trcProcessProxy != null && (options != null || configuration != null))
        {
        	String[] pollingOptions = new String[3];
        	if (options != null)
        	{
        		AgentConfigurationEntry[] confEntries = PeriodicPoll.getOption(options, new String[] {
        	    		IProfileLaunchConfigurationConstants.ATTR_SHOW_EXECUTION_FLOW,
        	    		IProfileLaunchConfigurationConstants.ATTR_EXEC_POLLING_FREQ_MODE,
        	    		IProfileLaunchConfigurationConstants.ATTR_EXEC_POLLING_FREQ});        	    
        	    for (int i = 0; i < confEntries.length; i++)
        		{
        	    	pollingOptions[i] = confEntries[i] == null ? null : confEntries[i].getValue();
        		}
        	}
        	else if (configuration != null)
        	{
        		try
				{
					pollingOptions[0] = String.valueOf(configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_SHOW_EXECUTION_FLOW, false));
					pollingOptions[1] = String.valueOf(configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_EXEC_POLLING_FREQ_MODE, true));
	        		int pollingFrequency = configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_EXEC_POLLING_FREQ, -1);
	        		pollingOptions[2] = pollingFrequency < 0 ? null : String.valueOf(pollingFrequency);
	        		
				} 
        		catch (CoreException e)
				{
					UIPlugin.getDefault().log(e);
				}
        		
        	}
        	
	        PeriodicPoll periodicPolling = PeriodicPoll.createInstance(pollingOptions, trcProcessProxy, processListener, agent);
	        if (periodicPolling != null)
	        	periodicPolling.start();
        }
        
        /* process agent active command for this listener */ 
        processListener.monitor(true);
        processListener.setLaunchMode(0); //attach
        agent.addAgentListener(processListener);

        agent.attach();
	}
	
	
	public static void addProcessParameters(Process lprocess, ProcessParameters parameters, boolean switchPerspective) throws CoreException
	{
		if (switchPerspective)
			switchPerspective();
		
		try
		{			  	  	        
			String paramStr = parameters.getApplicationName();
				  
			if (paramStr == null)
				paramStr = "";
			
			/* Add the executable arguments and the program arguments */
			String execParam = parameters.getExecutableParametersStr();
			if (isValidString(execParam)) 
				paramStr = execParam + " " + paramStr;
			
			
			String appParam = parameters.getApplicationParametersStr();
			if (isValidString(appParam))  
				paramStr += " " + appParam;
			
												
			lprocess.setParameters(paramStr);	 
		}
		catch (Exception e)
		{
			throw LauncherUtility.createCoreException(IStatus.ERROR, e.getMessage(), e);
		}
	}
	
	private static boolean isValidString (String str)
	{
		return str != null && str.length() > 0;
	}
	
	
	public static Agent createAgent(Process lprocess, ArrayList filters, Vector options, TRCProcessProxy trcProcessProxy, String agentName, String agentType)
	{
        Agent a = AgentFactory.createAgent(lprocess, agentName, agentType);

        a.setAutoAttach(true);

        AgentConfiguration ac = a.getConfiguration();

        if (ac == null) {
            a.setConfiguration(new AgentConfiguration());
        }

        // Make sure agent configuraiton is clear.	
        a.getConfiguration().clear();


        addDefaultProfileData(a, filters);
        addDefaultOptionsData(a, options);
        
        return a;
	}
	
	public static Agent createAgent(Process lprocess, ArrayList filters, Vector options, TRCProcessProxy trcProcessProxy)
	{
		return createAgent (lprocess, filters, options, trcProcessProxy, CommonUIConstants.PROFILE_AGENT_NAME, CommonUIConstants.PROFILE_AGENT_TYPE);
	}

	
	/**
	 * @deprecated Contributors should avoid using this method.  {@link PrimaryLaunchDelegate} provides
	 * a launch method that uses TPTP's standard for starting one or more profiling/monitoring session.
	 */
	public static final boolean launchTrace(TraceArguments trace, ArrayList filters, Vector options, String selectedProject, String selectedMonitor, ILaunch launch) {
		return launchTrace( trace,  filters,  options,  selectedProject,  selectedMonitor,  launch, true);	
	}

	/**
	 * @deprecated Contributors should avoid using this method.  {@link PrimaryLaunchDelegate} provides
	 * a launch method that uses TPTP's standard for starting one or more profiling/monitoring session.
	 */
    public static final boolean launchTrace(TraceArguments trace, ArrayList filters, Vector options, String selectedProject, String selectedMonitor, ILaunch launch, boolean switchPerspective) {    	
    	IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();        
        
    	if (switchPerspective)
        	switchPerspective();

        try {
        	   
        	String nodeName = trace.getHostName();
                 
            Node lhost = profileConnect(nodeName, String.valueOf(trace.getPortNumber()));

            if (lhost == null) {
                return false;
            }

            nodeName = lhost.getName();
			
            
            //create the container
            IContainer container = createContainer(new Path(selectedProject));

            //create the monitor
            TRCMonitor monitor = createMonitor(container, selectedMonitor);

            //create the node
            TRCNode node = createNode(monitor, nodeName, String.valueOf(trace.getPortNumber()));


            PIProcessListener listener = new PIProcessListener(node);
            listener.monitor(true);

            /* Create the execution process object and add its required fields */
            Process lprocess = ProcessFactory.createProcess(lhost);

            
            //add process location
            if (ConnectUtil.isLocalHost(nodeName))
				lprocess.setLocation(trace.getLocation());
            else 
				lprocess.setLocation(null);
            
			lprocess.addProcessListener(listener);
            listener.setLaunchMode(1);
            listener.setAutoMonitoring(trace.isAutoMonitoring());

            boolean hasProfAgents = false;

            /* Add The profile option and filter settings to an execution agent */          
            if (store.getBoolean(TraceConstants.PROF_OPTION_KEY)) {
            	hasProfAgents = true;

                Agent a = AgentFactory.createAgent(lprocess, CommonUIConstants.PROFILE_AGENT_NAME, CommonUIConstants.PROFILE_AGENT_TYPE);

                a.addAgentListener(listener);

                a.setAutoAttach(true);

                AgentConfiguration ac = a.getConfiguration();

                if (ac == null) {
                    a.setConfiguration(new AgentConfiguration());
                }

                // EC: make sure agent configuraiton is clear.	
                a.getConfiguration().clear();

                if (trace.isToProfileFile()) {
                    a.setProfileFile(trace.getProfileFile());
                }

                addDefaultProfileData(a, filters);
                addDefaultOptionsData(a, options);
            }

            String exec = trace.getClassName();

            
            // Add the VM arguments and the program arguments 
            if ((trace.getVMArguments() != null) && !trace.getVMArguments().equals("")) {
                exec = trace.getVMArguments() + " " + exec;
            }

            if ((trace.getParameters() != null) && !trace.getParameters().equals("")) {
                exec += (" " + trace.getParameters());
            }

            String param = "";

            if (!hasProfAgents) {
                param = exec;
            }

            // Add the executable and the JVMPI specific flag 
            if (hasProfAgents) {
                lprocess.setExecutable("java.exe");
                param = "-XrunpiAgent:server=controlled " + exec;
            } else {
                lprocess.setExecutable("java.exe");
                param = exec;
            }

            lprocess.setParameters(param);
            lprocess.addEnvironmentVariable(new Variable("CLASSPATH", TString.stripNewLineChar(trace.getClassPath())));
            lprocess.addEnvironmentVariable(new Variable("UICLASSPATH", trace.getClassPath()));
            listener.getProcessVariableList().add(new Variable("UICLASSPATH", trace.getClassPath()));

            setEnvironmentVariable(lprocess, trace.getEnvironmentVariable(), listener);
            

            try {
               lprocess.launch();
               
               final IProcess adapter = new ProcessAdapter(lprocess, launch, true, 1);
               UIPlugin.getDefault().registerLaunchProcess(adapter);
               lprocess.addProcessListener(new ProcessListener() {
                  public void processLaunched(Process p) {
               	  }
               	  public void processExited(Process p) {
                     UIPlugin.getDefault().deregisterLaunchProcess(adapter);
               		 DebugPlugin.getDefault().fireDebugEventSet(new DebugEvent[] {
                        new DebugEvent(adapter, DebugEvent.TERMINATE)
                     });
                  }
               });
               
               // add the mapping between TRCProcessProxy -> IProcess
               int pID = Integer.parseInt(lprocess.getProcessId());
               Iterator iter = node.getProcessProxies().iterator();
               while (iter.hasNext()) {
                  TRCProcessProxy proxy = (TRCProcessProxy)iter.next();
                  if (pID == proxy.getPid() && (proxy.isActive()==true)) { /** bug 83130 **/	
                     ProcessMap.put(proxy, adapter);
                     break;
                  }
               }
            } catch (Exception e) {
                String text = TraceMessages.PLERR_INFO_;
                final String msg = NLS.bind(TraceMessages.PL_ERROR_, trace.getClassName());
                final Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.INTERNAL_ERROR, text, null);

                HyadesUIPlugin.logError(e);
                
                Display.getDefault().syncExec(new Runnable() {
					public void run() {
		                ErrorDialog.openError(UIPlugin.getActiveWorkbenchShell(), TraceMessages.TRC_MSGT, msg, err);
					}
                });
                
                return false;
            }
        } catch (Exception exc) {
            exc.printStackTrace();
            return false;
        }

        return true;
    }
    

    public static Enumeration getNodeConfiguration(String nodeName, int portNumber, boolean showErrorMsgs) {
        try {
            Node node = profileConnect(nodeName, String.valueOf(portNumber), showErrorMsgs);

            if (node == null) {
                return null;
            }

            return node.listProcesses();
        } catch (NotConnectedException exc2) {
            exc2.printStackTrace();
        }

        return null;
    }
    
    public static Agent getAgentInstance(String hostName, int portNumber, TRCAgentProxy trcAgent) {
    	Enumeration processes = PDCoreUtil.getNodeConfiguration(hostName, portNumber ,false);
		if (processes!=null) {
			while (processes.hasMoreElements()) {
				Object p = processes.nextElement();
				if (p!= null && p instanceof Process) {
					Enumeration agents = ((Process)p).listAgents();
					while (agents.hasMoreElements()) {
						Object a = agents.nextElement();
						if (a!=null && a instanceof Agent && ((Agent)a).getUUID().equals(trcAgent.getRuntimeId()))
							return (Agent)a;
					}
				}
			}
		}
		return null;
    }
    
    
    public static Enumeration getNodeConfiguration(String nodeName, int portNumber) {
    	return getNodeConfiguration(nodeName, portNumber, true);
    }    

    protected static TRCAgentProxy createAgentNoCheck(TRCProcessProxy process, Agent agent, String sessionId) 
    {
    	return createAgentNoCheck (process, (Object)agent, sessionId);
    }
    
    protected static TRCAgentProxy createAgentNoCheck(TRCProcessProxy process, IAgent agent, String sessionId) 
    {
    	return createAgentNoCheck (process, (Object)agent, sessionId);
    }

    protected static TRCAgentProxy createAgentNoCheck(TRCProcessProxy process, Object agent, String sessionId) 
    {
    	Agent oldAgent = null;
    	IAgent newAgent = null;
    	
    	if (agent instanceof Agent)
    		oldAgent = (Agent)agent;
    	else if (agent instanceof IAgent)
    		newAgent = (IAgent)agent;
    	else
    		return null;
    	    	
    	
        TRCAgentProxy agentProxy;
        String agentName = oldAgent == null ? newAgent.getName() : oldAgent.getName();
        String agentType = oldAgent == null ? null : oldAgent.getType();
        String profileFile = oldAgent == null ? null : oldAgent.getProfileFile(); //bugzilla 187705, IAgent case is covered in org.eclipse.tptp.platform.jvmti.client.internal.launcher.TIDelegateHelper.configureDataProcessor(ILaunchConfiguration, TRCAgentProxy, ICollector)
        String timestamp = DateFormat.getDateTimeInstance(DateFormat.FULL, DateFormat.FULL).format(new Date());
        String rID = timestamp;

        rID = TString.change(rID, " ", "");
        rID = TString.change(rID, ":", "");

        String qualifiedAgentName = new StringBuffer(process.getNode().getMonitor().getName()).append("_").append(process.getNode().getName()).append("_").append(process.getPid()).append("_").append(rID).append("_").append(sessionId).append("_").append(agentName).toString();


        String pPath = process.eResource().getURI().toString();
        IPath path = new Path(pPath);

        if (path.segmentCount() > 1) {
            pPath = path.removeLastSegments(1).toString();
        }

        String fileName;
        if(ModelDebugger.INSTANCE.debugDatabaseResourcePostfix!=null && ModelDebugger.INSTANCE.debugDatabaseResourcePostfix.length()>1)
        {
        	fileName = qualifiedAgentName + ModelDebugger.INSTANCE.debugDatabaseResourcePostfix;
        }
        else
        {
            fileName = qualifiedAgentName + "." + CommonUIConstants.AGENT_EXT;
        }

        IPath filePath = new Path(pPath).append(fileName);

        URI uri = SaveUtil.createURI(filePath.toString()+"#").trimFragment();

        
        Resource agDoc = Resource.Factory.Registry.INSTANCE.getFactory(uri).createResource(uri);
        EList agExt = agDoc.getContents();

        UIPlugin.getDefault().getResourceSet().getResources().add(agDoc); // prevents reloading later

        HierarchyFactory factory = UIPlugin.getDefault().getPerftraceFactory();

        agentProxy = factory.createTRCAgentProxy();
        agentProxy.setName(agentName);

        if (agentType != null)
        {
            agentProxy.setType(agentType);
        }

        agentProxy.setProcessProxy(process);
        agentProxy.setProfileFile(profileFile);

        TRCAgent iAgent = factory.createTRCAgent();

        iAgent.setAgentProxy(agentProxy);
        agExt.add(iAgent);

		markModifiedResources(iAgent,true);


		/* Add analysis type to the configuration of the trc agent proxy */
		if (oldAgent != null)
			addAnalysisType(factory, oldAgent, agentProxy);
		
		
		/* Ali M. -- It's bad that we're hard coding types and names here */
        // create configuration
        if (oldAgent != null && agentProxy.getType().equals(CommonUIConstants.PROFILE_AGENT_TYPE)) 
        {
        	computeCollectionMode(factory, oldAgent, agentProxy);
        }

        if (agentProxy.getAgent() != null) 
        {
            agentProxy.getAgent().setCollectionMode(agentProxy.getCollectionMode());
        }

        iAgent.setName(agentProxy.getName());
        iAgent.setRuntimeId(agentProxy.getRuntimeId());
        iAgent.setType(agentProxy.getType());
        iAgent.setCollectionMode(agentProxy.getCollectionMode());

        return agentProxy;
    }



    private static void computeCollectionMode(HierarchyFactory factory, Agent agent, TRCAgentProxy agentProxy)
	{
    	if (agentProxy.getName().equals(CommonUIConstants.J2EE_AGENT_NAME)) 
    	{
            agentProxy.setCollectionMode(TRCCollectionMode.HEAP_AND_EXECUTION_FULL_LITERAL);
            return;
        } 
    	
        TRCConfiguration config = factory.createTRCConfiguration();

        config.setName(getFilterSetName(agent));   
        config.setActive(true);
        config.setAgentProxy(agentProxy);
        agentProxy.getConfigurations().add(config);

        AgentConfiguration aconfig = agent.getConfiguration();
        boolean collectHeapInfo = false;
        boolean collectHeapInstanceLevelInfo = false;
        boolean collectExecInfo = false;
        boolean collectExecInstanceLevelInfo = false;
        boolean showExecutionFlowFull = false;

        if (aconfig != null) {
            for (int idx = 0; idx < aconfig.size(); idx++) {
                AgentConfigurationEntry entry = aconfig.getEntryAt(idx);

                if(entry.getName().equals(IProfileLaunchConfigurationConstants.ATTR_SHOW_EXECUTION_FLOW))
                {
                    TRCOption opt = factory.createTRCOption();
                    opt.setKey(entry.getName());
                    opt.setValue(entry.getValue());
                    opt.setConfiguration(config);
                	showExecutionFlowFull = entry.getValue().equalsIgnoreCase("true");
                }
                else if(entry.getName().equals(IProfileLaunchConfigurationConstants.ATTR_SHOW_EXEC_INSTANCES))
                {
                	collectExecInstanceLevelInfo = entry.getValue().equalsIgnoreCase("true");
                }
                else if(entry.getName().equals(IProfileLaunchConfigurationConstants.ATTR_SHOW_HEAP_INSTANCES))
                {
                	collectHeapInstanceLevelInfo = entry.getValue().equalsIgnoreCase("true");
                }
                else if (entry.getType().equals("SETFILTER")) {
                    TRCFilter filter = factory.createTRCFilter();

                    filter.setMode(entry.getValue());
                    filter.setOperation("");
                    filter.setPattern(entry.getName());
                    filter.setType(entry.getValue());
                    filter.setActive(new Boolean(entry.isEnabled()));

                    filter.setConfiguration(config);
                } else if (entry.getType().equals("SETMETHODFILTER")) {
                    String criteria = entry.getName();
                    int offset = criteria.indexOf(' ');

                    if (offset > 0) {
                        TRCFilter filter = factory.createTRCFilter();

                        filter.setMode(entry.getValue());
                        filter.setPattern(criteria.substring(0, offset));
                        filter.setOperation(criteria.substring(offset + 1, criteria.length()));
                        filter.setType(entry.getValue());
                        filter.setActive(new Boolean(entry.isEnabled()));
                        filter.setConfiguration(config);
                    }
                } 
                else if (entry.getType().equals(LauncherConstants.SET_OPTION_PREFIX)) 
                {
                    TRCOption opt = factory.createTRCOption();
                    opt.setKey(entry.getName());
                    opt.setValue(entry.getValue());
                    opt.setConfiguration(config);
                	
                    if (entry.getName().equals(Options.OPTION_TRACE_MODE)) {
                        collectHeapInfo = !entry.getValue().equals(Options.OPTION_VALUE_NONE) && !entry.getValue().equals(Options.OPTION_VALUE_NOOBJECTCORRELATION);
                    }

                    if (entry.getName().equals(Options.OPTION_STACK_INFORMATION)) {
                        collectExecInfo = !entry.getValue().equals(Options.OPTION_VALUE_NONE);
                    }
                }
            }
        }

        
        /**
         * 
         * UI selections:
         * a = collect heap info
         * b = instance info in heap
         * c = collect execution info
         * d = exection full
         * e = instance info in exection
         * 
         * ---------------------------
         * |     | a | b | c | d | e |
         * |-------------------------|   
         * |HS   | 1 | 0 | 0 | - | - | HS = HEAP_STATISTICS_ONLY
         * |-------------------------|   
         * |HF   | 1 | 1 | 0 | - | - | HF = HEAP_FULL
         * |-------------------------|   
         * |HES  | 1 | 0 | 1 | 0 | - | HES = HEAP_AND_EXECUTION_STATISTICS_ONLY
         * |-------------------------|   
         * |HSEN | 1 | 0 | 1 | 1 | 0 | HEF = HEAP_STATISTICS_AND_EXECUTION_NO_INSTANCE
         * |-------------------------|   
         * |HEF  | 1 | 0 | 1 | 1 | 1 | HEF = HEAP_AND_EXECUTION_FULL
         * |-------------------------|   
         * |HFES | 1 | 1 | 1 | 0 | - | HFES = HEAP_FULL_AND_EXECUTION_STATISTICS_ONLY
         * |-------------------------|   
         * |HEF  | 1 | 1 | 1 | 1 | 0 | HEF = HEAP_AND_EXECUTION_FULL
         * |-------------------------|   
         * |HEF  | 1 | 1 | 1 | 1 | 1 | HEF = HEAP_AND_EXECUTION_FULL
         * |-------------------------|   
         * |ES   | 0 | - | 1 | 0 | - | ES = EXECUTION_STATISTICS_ONLY
         * |-------------------------|   
         * |ENI  | 0 | - | 1 | 1 | 0 | ENI = EXECUTION_NO_INSTANCES
         * |-------------------------|   
         * |EF   | 0 | - | 1 | 1 | 1 | EF = EXECUTION_FULL
         * |-------------------------| 
         * 
         */
        if (collectExecInfo) {
            if (collectHeapInfo) // heap + execution
            {
            	if (collectHeapInstanceLevelInfo) // collect heap instance
            	{
            		if (showExecutionFlowFull) // exec full, no need to worry about instance level up to this level
            			agentProxy.setCollectionMode(TRCCollectionMode.HEAP_AND_EXECUTION_FULL_LITERAL);
            		else // exec compressed, no instance collection available
            			agentProxy.setCollectionMode(TRCCollectionMode.HEAP_FULL_AND_EXECUTION_STATISTICS_ONLY_LITERAL);
            	}
            	else // no heap instance
            	{
            		if (showExecutionFlowFull) { // exectution full
            			if (collectExecInstanceLevelInfo) // execution full and execution instance
            				agentProxy.setCollectionMode(TRCCollectionMode.HEAP_AND_EXECUTION_FULL_LITERAL);
            			else  // execution full but no execution instance
            				agentProxy.setCollectionMode(TRCCollectionMode.HEAP_STATISTICS_AND_EXECUTION_NO_INSTANCES_LITERAL);
            		}
            		else {// execution compress, no instance collection available
            			agentProxy.setCollectionMode(TRCCollectionMode.HEAP_AND_EXECUTION_STATISTICS_ONLY_LITERAL);
            		}
            	}
            } 
            else // execution only
            {
                if (!showExecutionFlowFull) // exec compressed
                {
                    agentProxy.setCollectionMode(TRCCollectionMode.EXECUTION_STATISTICS_ONLY_LITERAL);
                } 
                else 
                { // exec full
                	if (collectExecInstanceLevelInfo) // collect exec instance
                		agentProxy.setCollectionMode(TRCCollectionMode.EXECUTION_FULL_LITERAL);
                	else // no exec instance
                		agentProxy.setCollectionMode(TRCCollectionMode.EXECUTION_NO_INSTANCES_LITERAL);
                }
            }
        } 
        else // heap only
        {                	
        	if(collectHeapInstanceLevelInfo)// collect heap instance                        	                	
                agentProxy.setCollectionMode(TRCCollectionMode.HEAP_FULL_LITERAL);
        	else // no heap instance
                agentProxy.setCollectionMode(TRCCollectionMode.HEAP_STATISTICS_ONLY_LITERAL);
        }
        
	}


	private static void addAnalysisType(HierarchyFactory factory, Agent agent, TRCAgentProxy agentProxy)
	{
    	/* Ali M. -- 135954 We need to read in the associated analysis types of the agent and 
		 * include it as part of the agent proxy */
		AgentConfiguration agentConfiguration = agent.getConfiguration();
		int agentConfigurationCount = agentConfiguration == null ? 0 : agentConfiguration.size();
		
		/* For each configuration */
		TRCConfiguration agentProxyConfiguration = null;
		for (int i = 0; i < agentConfigurationCount; i++)
		{
			AgentConfigurationEntry agentConfigurationEntry = agentConfiguration.getEntryAt(i);
			String configuraitonEntryName = agentConfigurationEntry.getName();
			/* Is this an analysis type option */
			if (configuraitonEntryName != null && configuraitonEntryName.startsWith(LauncherConstants.ANALYSIS_TYPE_ASSOCIATION) &&
				LauncherConstants.SET_OPTION_PREFIX.equals(agentConfigurationEntry.getType()))
			{
				
				TRCOption agentProxyConfigurationEntry = factory.createTRCOption();
                agentProxyConfigurationEntry.setKey(configuraitonEntryName);
                agentProxyConfigurationEntry.setValue(agentConfigurationEntry.getValue());
                
                /* Create the agent proxy configuration */
                if (agentProxyConfiguration == null)
                {
    				agentProxyConfiguration = factory.createTRCConfiguration();

    				agentProxyConfiguration.setName(LauncherConstants.SET_OPTION_PREFIX);   
    				agentProxyConfiguration.setActive(true);
    				agentProxyConfiguration.setAgentProxy(agentProxy);
                    agentProxy.getConfigurations().add(agentProxyConfiguration);
                }
                agentProxyConfigurationEntry.setConfiguration(agentProxyConfiguration);              
			}
		}
	}


	private static void addDefaultOptionsData(Agent a, Vector optionsList) {
        AgentConfiguration config = a.getConfiguration();

        for (int idx = 0; idx < optionsList.size(); idx++) {
            config.addEntry((AgentConfigurationEntry) optionsList.elementAt(idx));
        }
    }

    private static void addDefaultProfileData(Agent a, ArrayList filtersList) {
        AgentConfiguration config = a.getConfiguration();

        for (int idx = 0; idx < filtersList.size(); idx++) {
            FilterTableElement elem = (FilterTableElement) filtersList.get(idx);

            AgentConfigurationEntry entry = new AgentConfigurationEntry();

            /* RKD:  If there is no method filtercriteria then just send the class info */
            if (elem.getMethod().equals("")) {
                entry.setName(elem.getText());
                entry.setType("SETFILTER");
            } else {
                /* RKD:  The method filter criteria is specified by placing a space between the two strings */
                entry.setName(elem.getText() + ' ' + elem.getMethod());
                entry.setType("SETMETHODFILTER");
            }

            entry.setValue(elem.getVisibility().equals(TraceMessages.INCLUDE)?"INCLUDE":"EXCLUDE");
            entry.setEnabled(true);

            config.addEntry(entry);
        }
    }

    public static void setEnvironmentVariable(Process process, String[] envVariables, PIProcessListener listener) {
        if (envVariables != null && process != null) 
        {
	        for (int i = 0; i < envVariables.length; i++) 
	        {
	            try 
	            {
	            	/* Ali M.: Defect https://bugs.eclipse.org/bugs/show_bug.cgi?id=152196 */
	            	int equalInx = envVariables[i] == null ? -1 : envVariables[i].indexOf('=');
	            	if (equalInx == -1)
	            		continue;
	            	
	            	String name = envVariables[i].substring(0, equalInx);
	            	String value = envVariables[i].substring(equalInx + 1);
	                process.addEnvironmentVariable(new Variable(name, value));
	                listener.getProcessVariableList().add(new Variable(name, value));
	            }
	            catch (Exception e) 
	            {
	            	CommonPlugin.logError(e);
	            }
	        }
        }
    }

	public static void copyAgentProperties(TRCAgentProxy fAgentOld, TRCAgentProxy fAgent) {
		fAgent.setCollectionMode(fAgentOld.getCollectionMode());
		fAgent.setProfileFile(fAgentOld.getProfileFile());
		fAgent.getConfigurations().clear();
		for (Iterator iter = fAgentOld.getConfigurations().iterator(); iter.hasNext();) {
			TRCConfiguration configuration = (TRCConfiguration) iter.next();
			fAgent.getConfigurations().add(((TRCConfiguration)(EcoreUtil.copy(configuration))));
			
		};
		TRCProcessProxy process = fAgent.getProcessProxy();
		TRCNode node = process.getNode();
		sendConfigurationToAgent(fAgent, node.getName(), process.getPid());			
				
	}
	private static void sendConfigurationToAgent(TRCAgentProxy agent, String host, int processId)
	{
		Node node = PDCoreUtil.profileConnect(host, String.valueOf(agent.getProcessProxy().getNode().getPort()));
		if(node == null)
			return;

	
		Process p = node.getProcess(String.valueOf(processId));

		if(p != null)
		{
			try {
				Agent a = p.getAgent(agent.getName());
				a.setProfileFile(agent.getProfileFile());
				PDCoreUtil.setAgentConfiguration(agent, a);
				if(a != null && a.isActive()) {
				  a.publishConfiguration();
			  
				  /* Apply the filters */
				  CustomCommand command = new CustomCommand();
				  command.setData("APPLYFILTERS");
				  a.invokeCustomCommand(command);			  
				}			
			}
			catch(InactiveAgentException aex)
			{
			}
		}
	}
	
	

	private static void switchPerspective() {
        Display.getDefault().syncExec(new Runnable() {
        	public void run() {
        		try{
    		 		IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
    		 		boolean shouldSwitch = CommonUIPlugin.getDefault().openPerspectiveSwitchDialog(
    						window.getShell(), PDPerspective.ID_TRACE_PERSPECTIVE,
    						CommonUIPlugin.getDefault().getPreferenceStore(), 
    						HyadesUI.SWITCH_TO_PERSPECTIVE_KEY, 
    						TraceWizardMessages.CONFIRM_PSWITCH);
    			  	if(shouldSwitch) {
    			  		UIPlugin.getActivePage().showView(PDPerspective.ID_PD_NAVIGATOR_VIEW);
    			  	}
    		    	if (PDPerspective.ID_TRACE_PERSPECTIVE.equals(window.getActivePage().getPerspective().getId())) {
   						PDProjectExplorer nav = (PDProjectExplorer)UIPlugin.getActivePage().showView(PDPerspective.ID_PD_NAVIGATOR_VIEW);
   						nav.getViewer().refresh();
    		        }
        		} catch (Exception exc) {
        			CommonPlugin.logError(exc);
        		}
        	}
        });

	}
	
	/**
	 * Retrieves the filter set name for the given agent.
	 */
	private static String getFilterSetName(Agent a) {
        AgentConfiguration conf = a.getConfiguration();
        for (int i=0;i<conf.size();++i) {
        	AgentConfigurationEntry e = conf.getEntryAt(i);
        	if (IProfileLaunchConfigurationConstants.ATTR_FILTER_SET.equals(e.getName())) {
				Map filterSets = ProfilingSetsManager.instance().getFilterSets();
				Object obj = e.getValue() == null ? null : filterSets.get(e.getValue());
				if (obj != null) {
					FilterSetElement elem = (FilterSetElement)obj;
					return elem.getDisplayName();
				}
        	}
        }
        return UIPlugin.getResourceString(TraceConstants.DEFAULT_FILTER_SET);
	}
	
	
    public static TRCAgentProxy createAgent(TRCProcessProxy process, Agent a) 
    {
        TRCAgentProxy agentProxy = createAgent(process, a, "");

        //	setSessionStartTime(a.get, agentProxy);
        return agentProxy;
    }
    
    public static TRCAgentProxy createAgent(TRCProcessProxy processProxy, Agent agent, String sessionId) 
    {
        TRCAgentProxy agentProxy = getCorrespondingAgent(processProxy,agent,false);
        if(agentProxy!=null)
        	return agentProxy;

        return createAgentNoCheck(processProxy, agent, sessionId);
    }
        
    public static TRCAgentProxy createUniqueAgent(TRCProcessProxy processProxy, Object agent) 
    {
        long timeStamp = (new Date()).getTime();
        double time = ((double) timeStamp) / 1000;
        TRCAgentProxy agentProxy = createAgentNoCheck(processProxy, agent, timeStamp + "");
		LoadersUtils.registerAgentInstance(agentProxy,agent);
        LoadersUtils.setSessionStartTime(time, agentProxy);

        return agentProxy;
    }
    
    public static void monitorAgents(AgentSelection[] selection, ILaunchConfiguration configuration, boolean monitor)
    {
    	if (selection == null)
			return;
    	
		for (int i = 0; i < selection.length; i++)
		{
			TRCAgentProxy agentProxy = selection[i].getAgentProxy();
			try
			{
				PDCoreUtil.doAttach (agentProxy.getProcessProxy(), agentProxy, (Agent)selection[i].getAgent(), null, configuration);
			} 
			catch (InactiveAgentException e)
			{
				UIPlugin.getDefault().log(e);
				continue;
			} 
			catch (InactiveProcessException e)
			{
				UIPlugin.getDefault().log(e);
				continue;
			}		
			
			if (monitor)
				ProfileUIManager.getInstance().startMonitoring(agentProxy);			
		}
    }

    public static boolean isTBFFile(byte[] magic) {
    	if (magic.length < 4) {
    		return false;
    	}
        if ((magic[1] == 'T') && (magic[2] == 'B') && (magic[3] == 'F')) {
            return true;
        }
        return false;
    }
    
    public static boolean isTBFFile(InputStream readStream) throws IOException {
        byte[] magic = new byte[4];

        if (readStream.read(magic) > 1) {
        	return isTBFFile(magic);
        }

        return false;
    }

    public static boolean isTBFFile(String fileName) throws IOException {
        FileInputStream readStream = new FileInputStream(fileName);
        boolean ret = isTBFFile(readStream);

        readStream.close();
        readStream = null;

        return ret;
    }

    public static boolean isXMLFile(byte[] magic) {
    	if (magic.length < 5) {
    		return false;
    	}
        if ((magic[0] == '<') && 
                (magic[1] == '?') &&
                (magic[2] == 'x') &&
                (magic[3] == 'm') &&
                (magic[4] == 'l')) {
                return true;
            }
        return false;
    }    
    
    public static boolean isXMLFile(InputStream readStream) throws IOException {
        byte[] magic = new byte[5];

        if (readStream.read(magic) > 1) {
        	return isXMLFile(magic);
        }

        return false;
    }

    public static boolean isXMLFile(String fileName) throws IOException {
        FileInputStream readStream = new FileInputStream(fileName);
        boolean ret = isXMLFile(readStream);

        readStream.close();
        readStream = null;

        return ret;
    }
}
