/**********************************************************************
 * Copyright (c) 2006, 2010 Intel Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: WorkspaceFiltersExtractor.java,v 1.5 2010/08/22 22:10:13 jwest Exp $
 * 
 **********************************************************************/
package org.eclipse.hyades.trace.ui.internal.util;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.jdt.core.IClassFile;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.ui.IWorkingSet;

public class WorkspaceFiltersExtractor {

	public static ArrayList<FilterTableElement> extractFilters(final IProject project, final IWorkingSet[] workingSets) throws CoreException {
		Set<String> included = new HashSet<String>();
		
//		System.out.println("extractFilters: "+project.getName());
		
		addProjectFilters(project, included, false);

		IProject[] dependencies = project.getReferencedProjects();
		if (dependencies != null && dependencies.length > 0) {
			Set<String> dependenciesScope = getDependenciesScope(workingSets);
			Set<String> dependentIncluded = new HashSet<String>();
			
			for (int j = 0; j < dependencies.length; j++) {
				addProjectFilters(dependencies[j], dependentIncluded, false);
			}
			dependentIncluded = excludeDependenciesFilters(dependentIncluded, dependenciesScope);
			included.addAll(dependentIncluded);
		}
		return createFilters(included);
	}


	private static ArrayList<FilterTableElement> createFilters(final Set<String> methodNameFilters) {
		ArrayList<FilterTableElement> result = new ArrayList<FilterTableElement>();
		Iterator<String> iterator = methodNameFilters.iterator();
		while (iterator.hasNext()) {
			String methodNameFilter = iterator.next();
			result.add(new FilterTableElement(methodNameFilter + "*", "*", TraceMessages.INCLUDE));
		}
		return result;
	}


	
	private static void addProjectFilters(final IProject project, final Set<String> filters, final boolean isVerbose) throws CoreException {
		
		if(!project.isAccessible()) return;
		
		IResource[] members = project.members();
		for (int i = 0; i < members.length; i++) {
			addResourceFilters(members[i], filters, isVerbose);
		}
	}
	
	private static void addResourceFilters(final IResource resourse, final Set<String> filters, final boolean isVerbose) throws CoreException {
		if (resourse.isDerived()) {
			return;
		}
		if (resourse.getType() == IResource.FILE) {
			addFileFilters((IFile) resourse, filters, isVerbose);
		} else if (resourse.getType() == IResource.FOLDER) {
			addFolderFilters((IFolder) resourse, filters, isVerbose);
		}
	}

	private static void addFolderFilters(final IFolder folder, final Set<String> filters, final boolean isVerbose) throws CoreException {
		IResource[] members = folder.members();
		for (int j = 0; j < members.length; j++) {
			addResourceFilters(members[j], filters, isVerbose);
		}
	}

	private static void addFileFilters(final IFile file, final Set<String> filters, final boolean isVerbose) throws CoreException {
		IJavaElement element = JavaCore.create(file);
		if (element != null) {
			addJavaElementFilter(element, filters, isVerbose);
		}
	}
	
	private static void addJavaElementFilter(final IJavaElement element, final Set<String> filters, final boolean isVerbose) throws CoreException {
		switch (element.getElementType()) {
		
		case IJavaElement.JAVA_PROJECT:
			addProjectFilters(((IJavaProject) element).getProject(), filters, isVerbose);
			break;
			
		case IJavaElement.PACKAGE_FRAGMENT_ROOT:
			addPackageFragmentRootFilters((IPackageFragmentRoot) element, filters, isVerbose);
			break;
			
		case IJavaElement.PACKAGE_FRAGMENT:
			addPackageFragmentFilters((IPackageFragment) element, filters, isVerbose);
			break;
			
		case IJavaElement.COMPILATION_UNIT:
			addCompilationUnitFilters((ICompilationUnit) element, filters, isVerbose);
			break;
			
		case IJavaElement.CLASS_FILE:
			addClassFileFilters((IClassFile) element, filters, isVerbose);
			break;
			
		case IJavaElement.TYPE:
			addTypeFilters((IType) element, filters, isVerbose);
			break;
		default:
			break;
		}
	}

	private static void addPackageFragmentRootFilters(final IPackageFragmentRoot root, final Set<String> filters, final boolean isVerbose) throws CoreException {

// 		System.out.println(" * "+root.getElementName());
		
		IJavaElement[] children = root.getChildren();
		for (int i = 0; i < children.length; i++) {
			addJavaElementFilter(children[i], filters, isVerbose);
		}
	}

	private static void addClassFileFilters(final IClassFile file, final Set<String> filters, final boolean isVerbose) throws JavaModelException {
		
		addTypeFilters(file.getType(), filters, isVerbose);
	}

	private static void addTypeFilters(final IType type, final Set<String> filters, final boolean isVerbose) {
		IPackageFragment pack = type.getPackageFragment();
		if (!pack.isDefaultPackage()) {
			if (isVerbose) {
				filters.add(pack.getElementName() + "." + type.getElementName());
			} else {
				filters.add(pack.getElementName());
			}
		} else {
			filters.add(type.getTypeQualifiedName());
		}
	}
	
	private static void addCompilationUnitFilters(final ICompilationUnit unit, final Set<String> filters, final boolean isVerbose) throws JavaModelException 
	{
		try
		{
			IType[] elements = unit.getAllTypes();
			for (int i = 0; i < elements.length; i++) {
				addTypeFilters(elements[i], filters, isVerbose);
			}
		}
		catch (JavaModelException jme)
		{
			/* Ignore */
		}
	}

	private static void addPackageFragmentFilters(final IPackageFragment fragment, final Set<String> filters, final boolean isVerbose) throws JavaModelException {
		
		// System.out.println(" * "+fragment.getElementName());
		
		if (!fragment.hasSubpackages() && !isVerbose) {
			filters.add(fragment.getElementName());
		}
		ICompilationUnit[] children = fragment.getCompilationUnits();
		for (int i = 0; i < children.length; i++) {
			addCompilationUnitFilters(children[i], filters, isVerbose);
		}
	}
	
	private static Set<String> getDependenciesScope(final IWorkingSet[] workingSets) throws CoreException {
		final Set<String> result = new HashSet<String>();
		if (workingSets == null || workingSets.length == 0) {
			return result;
		}
		
		for (int i = 0; i < workingSets.length; i++) {
			IAdaptable[] elements = workingSets[i].getElements();
			for (int j = 0; j < elements.length; j++) {
				IJavaElement javaElement = (IJavaElement) elements[j].getAdapter(IJavaElement.class);
				
				if (javaElement != null) { 
					addJavaElementFilter(javaElement, result, true);
				} else {
					IResource resource = (IResource)elements[j].getAdapter(IResource.class);
					if (resource != null) {
						addResourceFilters(resource, result, true);
					}
				}
			}
		}
		return result;
	}

	private static Set<String> excludeDependenciesFilters(final Set<String> included, final Set<String> scope) {
		if(scope.isEmpty()) {
			return included;
		}
		Set<String> result = new HashSet<String>();
		
		Iterator<String> it = included.iterator();
		while (it.hasNext()) {
			String element = (String) it.next();
			if (scope.contains(element)) {
				result.add(element);
			} else {
				result.addAll(getContainingSubstring(scope, element));
			}
		}
		return result;
	}

	private static List<String> getContainingSubstring(final Set<String> scope, final String substring) {
		List<String> result = new ArrayList<String>();
		Iterator<String> it = scope.iterator();
		
		while (it.hasNext()) {
			String element = (String) it.next();
			if (element.startsWith(substring)) {
				result.add(element);
			}
		}		
		return result;
	}
}
