/**********************************************************************
 * Copyright (c) 2010 IBM Corporation
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.util;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaModel;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.pde.launching.IPDELauncherConstants;

public class WorkspacePluginAppExtractor {
	
	/**
	 * Parses a list that looks like this: 
	 * 	org.eclipse.tptp.trace.jvmti.client@default:default, org.eclipse.tptp.platform.integration.pde@default:default
	 * e.g. for an app with two plugins
	 * @param s
	 * @return List containing only the project names
	 */
	private static List<String> parsePDEProjectList(String s) {
		List<String> result = new ArrayList<String>();
		
		String[] rawProjectArray = s.split(",");  //$NON-NLS-1$
		
		for(int x = 0; x < rawProjectArray.length; x++) {
			int c = rawProjectArray[x].indexOf("@");  //$NON-NLS-1$
			if(c != -1) {
				String str = rawProjectArray[x].substring(0, c);
				if(str.trim().length() != 0) {
					result.add(str);
				}
			}
		}
		
		return result;
	}
	
	private static List<IJavaProject> convertStringListToJavaProjectList(List<String> l) {
		List<IJavaProject> result = new ArrayList<IJavaProject>();
		
		for(String name : l) {
			IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(name);
			IJavaElement element = JavaCore.create(project);
			if(element instanceof IJavaProject) {
				result.add((IJavaProject)element);
			}
		}
		
		return result;
	}
	
	/** 
	 * This method is used to grab the plugin list from an Eclipse applicaiton launch configuration
	 * (or from other plugin-based projects) 
	 * 
	 * If you call this, ensure you check for the empty list case, is this will be indicative of a failure of 
	 * the plugin detection algorithm. 
	 * @param restrictResultsToWorkspacePlugins Whether or not the results should only contain plugins that are in the workspace (e.g. their source)  
	 * */
	public static List<IJavaProject> getJavaProjectList(ILaunchConfiguration configuration, boolean restrictResultsToWorkspacePlugins) {
		String allOfEm, selectedFromPDE, deSelectedFromPDE;
		boolean useDefault, autoAdd;

		List<IJavaProject> result = new ArrayList<IJavaProject>();
		
		try {
//			autoAdd = configuration.getAttribute(IPDELauncherConstants.AUTOMATIC_ADD, true);
			useDefault = configuration.getAttribute(IPDELauncherConstants.USE_DEFAULT, true);
//			useDefault |= configuration.getAttribute(IPDELauncherConstants.USEFEATURES, false);

			if (useDefault) {
				// Get all plugins in the workspace
				final IJavaModel model= JavaCore.create(ResourcesPlugin.getWorkspace().getRoot());
				if (model != null) {
					final IJavaProject[] projects = model.getJavaProjects();
					
					// Copy the projects to the result array
					for(IJavaProject p : projects)
						result.add(p);
					
				}

			} else {
				
				if(restrictResultsToWorkspacePlugins) {
					selectedFromPDE = configuration.getAttribute(IPDELauncherConstants.SELECTED_WORKSPACE_PLUGINS, ""); //$NON-NLS-1$					
				} else {
					selectedFromPDE = configuration.getAttribute(IPDELauncherConstants.SELECTED_TARGET_PLUGINS, ""); //$NON-NLS-1$
				}
				
				List<String> selProjList = parsePDEProjectList(selectedFromPDE);
				
				result.addAll(convertStringListToJavaProjectList(selProjList));
			}
			
		} catch (Exception e) {
			e.printStackTrace();
		}
		
		return result;
	}

}
