/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation, Intel Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ImportTracePage1.java,v 1.67 2010/08/18 20:37:20 jcayne Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 * 
 **********************************************************************/
package org.eclipse.hyades.trace.ui.internal.wizard;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.Enumeration;
import java.util.Timer;
import java.util.TimerTask;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.hyades.execution.local.CommunicationDebug;
import org.eclipse.hyades.loaders.util.BinaryLoader;
import org.eclipse.hyades.loaders.util.InvalidXMLException;
import org.eclipse.hyades.loaders.util.XMLLoader;
import org.eclipse.hyades.models.hierarchy.TRCCollectionMode;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.util.MonitoredInputStream;
import org.eclipse.hyades.models.util.ModelDebugger;
import org.eclipse.hyades.models.util.provisional.ITraceModelInteraction;
import org.eclipse.hyades.models.util.provisional.TraceModelInteraction;
import org.eclipse.hyades.trace.internal.ui.PDPerspective;
import org.eclipse.hyades.trace.internal.ui.PDPluginImages;
import org.eclipse.hyades.trace.internal.ui.PDProjectExplorer;
import org.eclipse.hyades.trace.internal.ui.PDProjectViewer;
import org.eclipse.hyades.trace.ui.HyadesUtil;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.core.TraceLocationUI;
import org.eclipse.hyades.trace.ui.internal.util.PDCoreUtil;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.hyades.ui.HyadesUI;
import org.eclipse.hyades.ui.filters.IFilterScopes;
import org.eclipse.hyades.ui.internal.navigator.INavigator;
import org.eclipse.hyades.ui.util.GridUtil;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.TabFolder;
import org.eclipse.swt.widgets.TabItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.tptp.platform.common.internal.CommonPlugin;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIPlugin;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.WizardDataTransferPage;

import com.ibm.icu.text.MessageFormat;

/*
 * CONTEXT_ID xmli0000 for XML Trace Import wizard page
 */

/**
 *  Page 1 of the base resource import-from-zip Wizard.
 *
 *  Note that importing from .jar is identical to importing from .zip, so
 *  all references to .zip in this class are equally applicable to .jar
 *  references.
 */
public class ImportTracePage1 extends WizardDataTransferPage implements Listener, ModifyListener, SelectionListener, IRunnableWithProgress {
	//~ Instance fields ----------------------------------------------------------------------------

	protected Button sourceBrowseButton;

	protected String fError;

	protected String fInputLog = "";

	protected TRCMonitor fMonitor;

	// widgets
	protected Text sourceNameField;

	protected TraceLocationUI _location;

	protected double fStartPercentage;

	protected double fStopPercentage;

	private Shell shell;

	private long logLength;

	private int collectionMode = TRCCollectionMode.HEAP_STATISTICS_ONLY;

	protected TabFolder details;

	protected TabItem rangeTab;

	protected ImportTraceRangeUI rangeUI;

	protected TabItem filterTab;

	protected ImportTraceFilterUI filterUI;

	private boolean rootElementRequired;
	
	final private Object importJobLock = new Object(); 

	//~ Constructors -------------------------------------------------------------------------------

	/**
	 *  Creates an instance of this class
	 * @param aWorkbench IWorkbench
	 * @param selection IStructuredSelection
	 */
	public ImportTracePage1(IWorkbench aWorkbench, IStructuredSelection selection) {
		super("xmlTraceFileImportPage1");
		setTitle(TraceMessages.IMP_XMLPT);
		setDescription(TraceMessages.IMP_XMLPD);
		setImageDescriptor(PDPluginImages.getImageDescriptor(PDPluginImages.IMG_UI_WZ_IMPORT_PROF));

		shell = aWorkbench.getActiveWorkbenchWindow().getShell();
	}

	//~ Methods ------------------------------------------------------------------------------------

	/** (non-Javadoc)
	 * Method declared on IDialogPage.
	 */
	public void createControl(Composite parent) {
		Composite composite = new Composite(parent, SWT.NULL);

		composite.setLayout(new GridLayout());
		composite.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_FILL | GridData.HORIZONTAL_ALIGN_FILL));
		//        composite.setSize(composite.computeSize(SWT.DEFAULT, SWT.DEFAULT));

		createSourceGroup(composite);

		createSpacer(composite);

		createBoldLabel(composite, TraceMessages.IMP_XML_SRC);

		_location = new TraceLocationUI();
		_location.createControl(composite);
		_location.getLocationLabel().setText(TraceMessages.IMP_FLD);

		createSpacer(composite);

		details = new TabFolder(composite, SWT.NONE);
		GridData data = GridUtil.createFill();
		details.setLayoutData(data);
		details.addSelectionListener(this);

		rangeTab = new TabItem(details, SWT.NONE);
		rangeTab.setText(TraceMessages.IM_TRRAN);
		rangeUI = new ImportTraceRangeUI(this);
		rangeTab.setControl(rangeUI.createControl(details));

		filterTab = new TabItem(details, SWT.NONE);
		filterTab.setText(TraceMessages.IM_TRFIL);
		filterUI = new ImportTraceFilterUI(getFilterScope());
		filterTab.setControl(filterUI.createControl(details));

		restoreWidgetValues();
		updateWidgetEnablements();
		setPageComplete(determinePageCompletion());

		setControl(composite);

		PDProjectExplorer projectExplorer = null;
		PDProjectViewer projectViewer = null;

		IWorkbenchPage page = UIPlugin.getActiveWorkbenchWindow().getActivePage();

		if ((page != null) && page.getPerspective().getId().equals(PDPerspective.ID_TRACE_PERSPECTIVE)) {
			projectExplorer = UIPlugin.getDefault().getViewer();
		}

		IProject selectionProject = null;
		TRCMonitor selectionMonitor = null;

		if (projectExplorer != null) {
			projectViewer = (PDProjectViewer) projectExplorer.getViewer();
		}

		if (projectViewer != null) {
			selectionProject = projectViewer.getSelectionProject();
			selectionMonitor = projectViewer.getSelectionMonitor();
		}

		_location.getLocation().addModifyListener(this);
		_location.getMonitor().addModifyListener(this);
		sourceNameField.addModifyListener(this);

		if (selectionProject != null) {
			_location.setLocation(selectionProject.getName());
		}

		if (selectionMonitor != null) {
			_location.setMonitor(selectionMonitor.getName());
		}

		rangeUI.getshowHeapStatistic().addSelectionListener(this);
		rangeUI.getshowExecStatistic().addSelectionListener(this);
		rangeUI.getshowExecFlow().addSelectionListener(this);
		rangeUI.getWholeFileBtn().addSelectionListener(this);
		rangeUI.getInRangeBtn().addSelectionListener(this);
		rangeUI.getstartText().addModifyListener(this);
		rangeUI.getstopText().addModifyListener(this);
	}

	private String getFilterScope() {
		return IFilterScopes.IMPORT_TRACE_FILTER_SCOPE;
	}

	/**
	 * The Finish button was pressed.  Try to do the required work now and answer
	 * a boolean indicating success.  If <code>false</code> is returned then the
	 * wizard will not close.
	 *
	 * @return boolean
	 */
	public boolean finish() {
		fInputLog = sourceNameField.getText().trim();
		File log = new File(sourceNameField.getText().trim());

		if (!log.exists()) {
			fError = NLS.bind(TraceMessages.ERR_XMLNF, fInputLog);

			Status err1 = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.INTERNAL_ERROR, fError, null);
			ErrorDialog.openError(getShell(), TraceMessages.TRC_MSGT, TraceMessages.ERR_LXML, err1);

			return false;
		}

		if (!log.canRead()) {
			fError = NLS.bind(TraceMessages.ERR_XMLRD, fInputLog);

			Status err1 = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.INTERNAL_ERROR, fError, null);
			ErrorDialog.openError(getShell(), TraceMessages.TRC_MSGT, TraceMessages.ERR_LXML, err1);

			return false;
		}

		fError = null;

		if (rangeUI.getInRangeSelection()) {
			fStartPercentage = rangeUI.getStartRange();
			fStopPercentage = rangeUI.getStopRange();
		} else {
			fStartPercentage = 0;
			fStopPercentage = 100;
		}

		return loadXMLTrace();
	}

	/**
	 *  Handle all events and enablements for widgets in this dialog
	 *
	 * @param event Event
	 */
	public void handleEvent(Event event) {
		if (event.widget == sourceBrowseButton) {
			handleSourceBrowseButtonPressed();
		}
	}

	/**
	 * Sent when the text is modified.
	 *
	 * @param e an event containing information about the modify
	 */
	public void modifyText(ModifyEvent e) {
		setPageComplete(determinePageCompletion());
	}

	private void prepareAndImport(IProgressMonitor mon, File log, InputStream checkStream, InputStream readStream, String fullFilename) throws IOException {

		byte[] magic = new byte[10];
		if (checkStream.read(magic) < 0) {
			throw new IOException("Empty or invalid file - no magic exists");
		}
		XMLLoader processor = null;
		if (PDCoreUtil.isTBFFile(magic)) { // TBF (binary)
			logLength = log.length();
			processor = getNewProcessor(true);
			processor.setCollectionMode(collectionMode);
			importFile(mon, processor, log, readStream, fullFilename, true);
		} else if (PDCoreUtil.isXMLFile(magic)) { // XML
			logLength = log.length();
			processor = getNewProcessor(false);
			processor.setCollectionMode(collectionMode);
			importFile(mon, processor, log, readStream, fullFilename, false);
		} else {
			throw new IOException("Invalid trace file format. Supported formats are XML and TBF (binary)");
		}
		if (CommunicationDebug.INSTANCE.debugUseEventMode)
			processor.cleanUp();
	}
	
	public void run(IProgressMonitor mon) {
		Display d = Display.getDefault();

		d.syncExec(new Runnable() {
			public void run() {
				if (rangeUI.getshowHeapStatistic().getSelection()) {
					collectionMode = TRCCollectionMode.HEAP_STATISTICS_ONLY;
				} else if (rangeUI.getshowThreadStatistic().getSelection()) {
					collectionMode = TRCCollectionMode.EXECUTION_FULL;
				} else if (rangeUI.getshowExecStatistic().getSelection()) {
					collectionMode = TRCCollectionMode.HEAP_AND_EXECUTION_STATISTICS_ONLY;
				} else if (rangeUI.getshowExecFlow().getSelection()) {
					collectionMode = TRCCollectionMode.HEAP_AND_EXECUTION_FULL;
				}
			}
		});

		try {
			File log = new File(fInputLog);

			if (PDCoreUtil.isZipFile(fInputLog)) { // Zipped XML
				ZipFile zf = new ZipFile(log);

				Enumeration entries = zf.entries();

				while (entries.hasMoreElements()) {
					ZipEntry zipEntry = (ZipEntry) entries.nextElement();
					logLength = zipEntry.getSize();
					if (logLength < log.length())
						logLength = Long.MAX_VALUE;
					prepareAndImport(mon, log, zf.getInputStream(zipEntry), zf.getInputStream(zipEntry), zipEntry.getName());
				}
			} else {
				prepareAndImport(mon, log, new BufferedInputStream(new FileInputStream(log)), new BufferedInputStream(new FileInputStream(log)), log.getAbsolutePath());
			}
		} catch (IOException e) {
			fError = NLS.bind(TraceMessages.IMP_IOEXC, e.getMessage());
		}
	}

	/**
	 * @return
	 */
	private XMLLoader getNewProcessor(boolean binary) {
		XMLLoader processor = null;
		if(binary) {
			processor = new BinaryLoader(fMonitor);
			((BinaryLoader)processor).setLoadXml(false);
		} else {
			processor = new XMLLoader(fMonitor);
		}
		if (filterUI.getSelectedFilter() != null) {
			processor.getContext().setImportFilter(filterUI.getSelectedFilter());
			processor.getContext().getFilterEngine().init();
		}
		return processor;
	}

	public void widgetDefaultSelected(SelectionEvent e) {
	}

	public void widgetSelected(SelectionEvent e) {
		setPageComplete(determinePageCompletion());
	}

	protected boolean allowNewContainerName() {
		return true;
	}

	/**
	 * Creates a new button with the given id.
	 * <p>
	 * The <code>Dialog</code> implementation of this framework method
	 * creates a standard push button, registers for selection events
	 * including button presses and registers
	 * default buttons with its shell.
	 * The button id is stored as the buttons client data.
	 * Note that the parent's layout is assumed to be a GridLayout and
	 * the number of columns in this layout is incremented.
	 * Subclasses may override.
	 * </p>
	 *
	 * @param parent the parent composite
	 * @param id the id of the button (see
	 *  <code>IDialogConstants.*_ID</code> constants
	 *  for standard dialog button ids)
	 * @param label the label from the button
	 * @param defaultButton <code>true</code> if the button is to be the
	 *   default button, and <code>false</code> otherwise
	 */
	protected Button createButton(Composite parent, int id, String label, boolean defaultButton) {
		// increment the number of columns in the button bar
		((GridLayout) parent.getLayout()).numColumns++;

		Button button = new Button(parent, SWT.PUSH);

		GridData data = new GridData(GridData.HORIZONTAL_ALIGN_FILL);

		//        data.heightHint = convertVerticalDLUsToPixels(IDialogConstants.BUTTON_HEIGHT);
		data.widthHint = convertHorizontalDLUsToPixels(IDialogConstants.BUTTON_WIDTH + 2);
		button.setLayoutData(data);

		button.setData(new Integer(id));
		button.setText(label);

		if (defaultButton) {
			Shell shell = parent.getShell();

			if (shell != null) {
				shell.setDefaultButton(button);
			}

			button.setFocus();
		}

		button.setFont(parent.getFont());

		return button;
	}

	/**
	 *  Create the import source specification widgets
	 */
	protected void createSourceGroup(Composite parent) {
		Composite sourceContainerGroup = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout();

		layout.numColumns = 3;
		sourceContainerGroup.setLayout(layout);
		sourceContainerGroup.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.GRAB_HORIZONTAL));

		new Label(sourceContainerGroup, SWT.NONE).setText(TraceMessages.IMP_XML_SR);

		// source name entry field
		sourceNameField = new Text(sourceContainerGroup, SWT.BORDER);

		GridData data = new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.GRAB_HORIZONTAL);

		data.widthHint = SIZING_TEXT_FIELD_WIDTH;
		sourceNameField.setLayoutData(data);

		// source browse button
		sourceBrowseButton = new Button(sourceContainerGroup, SWT.PUSH);
		sourceBrowseButton.setText(TraceMessages.BROWSE4);
		sourceBrowseButton.addListener(SWT.Selection, this);
		sourceBrowseButton.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL));

		sourceNameField.setFocus();

		PlatformUI.getWorkbench().getHelpSystem().setHelp(parent, UIPlugin.getPluginId() + ".xmli0000");
	}

	/**
	 * Returns whether this page is complete. This determination is made based upon
	 * the current contents of this page's controls.  Subclasses wishing to include
	 * their controls in this determination should override the hook methods
	 * <code>validateSourceGroup</code> and/or <code>validateOptionsGroup</code>.
	 *
	 * @return <code>true</code> if this page is complete, and <code>false</code> if
	 *   incomplete
	 * @see #validateSourceGroup
	 * @see #validateOptionsGroup
	 */
	protected boolean determinePageCompletion() {
		boolean complete = validateSourceGroup() && validateDestinationGroup() && validateOptionsGroup();

		// Avoid draw flicker by not clearing the error
		// message unless all is valid.
		if (complete) {
			setErrorMessage(null);
		}

		return complete;
	}

	/**
	 *  Open an appropriate source browser so that the user can specify a source
	 *  to import from
	 */
	protected void handleSourceBrowseButtonPressed() {
		String currentSource = this.sourceNameField.getText();

		FileDialog dlg = new FileDialog(sourceNameField.getShell());

		dlg.setFilterPath(currentSource);
		dlg.setFilterExtensions(new String[] { "*.trcxml", "*.trcbin", "*.*" });
		dlg.open();

		String fileName = dlg.getFileName();

		if ((fileName != null) && !fileName.equals("")) {
			sourceNameField.setText(dlg.getFilterPath() + File.separator + fileName);
		}
	}

	/**
	 * Check if widgets are enabled or disabled by a change in the dialog.
	 * @param event Event
	 */
	protected void updateWidgetEnablements() {
		boolean pageComplete = determinePageCompletion();

		setPageComplete(pageComplete);

		if (pageComplete) {
			setMessage(null);
		}

		super.updateWidgetEnablements();
	}

	/**
	 * Returns whether this page's options group's controls currently all contain
	 * valid values.
	 * <p>
	 * The <code>WizardDataTransferPage</code> implementation of this method returns
	 * <code>true</code>. Subclasses may reimplement this hook method.
	 * </p>
	 *
	 * @return <code>true</code> indicating validity of all controls in the options
	 *   group
	 */
	protected boolean validateOptionsGroup() {
		return (!_location.getLocation().getText().trim().equals("") && !_location.getMonitor().getText().trim().equals("") && (rangeUI.getWholeFileSelection() || (rangeUI.getInRangeSelection() && (rangeUI.checkRegionValue() == null))));
	}

	/**
	 * Returns whether this page's source specification controls currently all
	 * contain valid values.
	 * <p>
	 * The <code>WizardDataTransferPage</code> implementation of this method returns
	 * <code>true</code>. Subclasses may reimplement this hook method.
	 * </p>
	 *
	 * @return <code>true</code> indicating validity of all controls in the
	 *   source specification group
	 */
	protected boolean validateSourceGroup() {
		return !sourceNameField.getText().trim().equals("");
	}

	private String createModel(TRCMonitor monitor, String inputLog) {
		fMonitor = monitor;
		fInputLog = inputLog;

		try {
			getWizard().getContainer().run(true, true, this);
		} catch (InterruptedException e) {
			return "";
		} catch (InvocationTargetException e) {
			return e.getTargetException().getMessage();
		}

		return fError;
	}

	private void handleOutOfMemoryError() {
		/* Inform the user we are running out of memory */
		String text = TraceMessages.IOUTMEM_ERRT;

		;

		//System.out.println(text);
		if (shell != null) {
			final Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.INTERNAL_ERROR, text, null);

			shell.getDisplay().asyncExec(new Runnable() {
				public void run() {
					ErrorDialog.openError(shell, TraceMessages.TRC_MSGT, TraceMessages.OUTMEM_ERRT, err);
				}
			});
		}
	}

	private void importFile(IProgressMonitor mon, final XMLLoader processor, final File log, InputStream readStream, final String entryName, final boolean binary) throws OutOfMemoryError {
		try {
			if (!CommunicationDebug.INSTANCE.debugUseEventMode) {

				final MyBufferedInputStream inputStream = new MyBufferedInputStream(readStream,16*1024);
					if (!binary) {
					inputStream.mark(1000);
					byte[] rootElement = new byte[1000];
					rootElementRequired=false;
					try {
						int readBytes = inputStream.read(rootElement);
						String root = new String(rootElement,0,readBytes);
						root = root.toUpperCase();
						if(root.indexOf("<TRACE>") !=-1 || root.indexOf("<COMMONBASEEVENTS ") !=-1 || root.indexOf("<COMMONBASEEVENTS>") !=-1)
						{
							rootElementRequired=false;
						}
						else
						{
							rootElementRequired=true;
						}
						inputStream.reset();
					} catch (IOException e1) {
						e1.printStackTrace();
					}
	
					if(rootElementRequired)
						inputStream.addRootTRACEElement();
				}
				final String jobName = MessageFormat.format(TraceWizardMessages.IMPORT_FILE, new String[] { "" + log.length(), log.getAbsolutePath() });
				final long entryLength = logLength == Long.MAX_VALUE ? log.length() * 10 : logLength;
				Job job = new Job(jobName) {
					MonitoredInputStream monitoredInputStream = null;

					long prevTotalBytesRead = 0;

					long totalBytesRead = 0;

					int prevProcessedFragments = 0;

					long prevTime;

					long unitSize = entryLength / 100;

					int processedUnits = 0;

					int progress = 0;

					long startTime, deltaTime, partDeltaTime;

					String taskName;
					int collectingState;
					
					protected IStatus run(final IProgressMonitor monitor) {
						Timer refresh = new Timer(false);
						refresh.scheduleAtFixedRate(new TimerTask() {


							public void run() {
								if (unitSize <= 0 || monitor.isCanceled()) {
									if(ModelDebugger.INSTANCE.debug)
										ModelDebugger.log("Import refresh thread canceled!");
									cancel();
									return;
								}
								if(processor.getContext().getAgentProxy() != null)
								{	
									updateMonitor(monitor);
									updateCollecting();
								}
							}

							void updateMonitor(final IProgressMonitor monitor) {
								deltaTime = System.currentTimeMillis();
								partDeltaTime = (deltaTime - prevTime) / 1000;
								prevTime = deltaTime;
								deltaTime = (deltaTime - startTime) / 1000;
								if (deltaTime == 0)
									deltaTime = 1;
								if (partDeltaTime == 0)
									partDeltaTime = 1;
								final int deltaEvents = processor.getProcessedFragments() - prevProcessedFragments;
								prevProcessedFragments = processor.getProcessedFragments();
								if (unitSize != 0) {
									int temp = processedUnits;
									processedUnits = (int) (totalBytesRead / unitSize);
									monitor.worked(processedUnits - temp);
									progress += processedUnits - temp;
									//taskName = MessageFormat.format(TraceWizardMessages.IMPORT_PROGRESS, new String[] { "" + progress, "" + prevProcessedFragments, "" + (deltaEvents / partDeltaTime), "" + deltaTime, "" + totalBytesRead, "" + ((totalBytesRead - prevTotalBytesRead) / partDeltaTime) });
									//monitor.subTask(taskName);

								}
								/*
								Display d = Display.getDefault();
								if (d == null || d.isDisposed())
									return;
								d.syncExec(new Runnable() {
									public void run() {
										INavigator navigator = HyadesUtil.getActiveNavigator();
										if (navigator == null)
											return;
										if (HyadesUtil.getMofObject() != processor.getContext().getAgentProxy())
											return;

										String statusBarMessage = "";
										if (unitSize > 0)
											statusBarMessage = MessageFormat.format(TraceWizardMessages.IMPORT_PROGRESS, new String[] { "" + progress, "" + prevProcessedFragments, "" + (deltaEvents / partDeltaTime), "" + deltaTime, "" + totalBytesRead, "" + ((totalBytesRead - prevTotalBytesRead) / partDeltaTime) });
										IStatusLineManager statusLineManager = navigator.getViewSite().getActionBars().getStatusLineManager();
										statusLineManager.setMessage(statusBarMessage);
										if(ModelDebugger.INSTANCE.debug)
											ModelDebugger.log("Import refresh thread - update status bar");
									}
								});*/

							}

							void updateCollecting() {
								try {
									if (prevTotalBytesRead == totalBytesRead) {
										if (collectingState != 1) {
											if (processor.getContext().getAgentProxy().isCollectionData()) {
												processor.getContext().getAgentProxy().setCollectionData(false);
											}
											Display d = Display.getDefault();
											if (d == null || d.isDisposed())
												return;
											d.syncExec(new Runnable() {
												public void run() {
													ProfileEvent event;

													if(collectingState==0)
													{
														event = UIPlugin.getDefault().getProfileEvent();

														event.setSource(processor.getContext().getAgentProxy());
														event.setType(ProfileEvent.REFRESH_PD_EXPLORER);
														UIPlugin.getDefault().notifyProfileEventListener(event);

														event = UIPlugin.getDefault().getProfileEvent();

														event.setSource(processor.getContext().getAgentProxy());
														event.setType(ProfileEvent.START_MONITOR);
														UIPlugin.getDefault().notifyProfileEventListener(event);
													}
													
													event = UIPlugin.getDefault().getProfileEvent();

													event.setSource(processor.getContext().getAgentProxy());
													event.setType(ProfileEvent.STOP_COLLECTING);
													UIPlugin.getDefault().notifyProfileEventListener(event);
													if(ModelDebugger.INSTANCE.debug)
														ModelDebugger.log("Import refresh thread - set STOP_COLLECTING "+processor.getContext().getAgentProxy());

												}
											});
											collectingState = 1;
										}
									} else if (totalBytesRead > prevTotalBytesRead) {
										if (collectingState != 2) {
											if (!processor.getContext().getAgentProxy().isCollectionData()) {
												processor.getContext().getAgentProxy().setCollectionData(true);
											}
											Display d = Display.getDefault();
											if (d == null || d.isDisposed())
												return;
											d.syncExec(new Runnable() {
												public void run() {
													ProfileEvent event;

													if(collectingState==0)
													{
														event = UIPlugin.getDefault().getProfileEvent();
														event.setSource(null);
														event.setType(ProfileEvent.REFRESH_PD_EXPLORER);
														UIPlugin.getDefault().notifyProfileEventListener(event);
														
														event = UIPlugin.getDefault().getProfileEvent();

														event.setSource(processor.getContext().getAgentProxy());
														event.setType(ProfileEvent.START_MONITOR);
														UIPlugin.getDefault().notifyProfileEventListener(event);
													}
													
													event = UIPlugin.getDefault().getProfileEvent();
													
													event.setSource(processor.getContext().getAgentProxy());
													event.setType(ProfileEvent.COLLECTING);
													UIPlugin.getDefault().notifyProfileEventListener(event);
													if(ModelDebugger.INSTANCE.debug)
														ModelDebugger.log("Import refresh thread - set COLLECTING "+processor.getContext().getAgentProxy());
												}
											});
											collectingState = 2;
										}
										prevTotalBytesRead = totalBytesRead;
									}
								} catch (RuntimeException e) {
									e.printStackTrace();
								}
							}

							public boolean cancel() {
								notifyStopCollecting();
								//								System.out.println("ImportTracePage1.importFile() - cancel");
								/*Display d = Display.getDefault();
								if (d != null && !d.isDisposed()) {
									d.syncExec(new Runnable() {
										public void run() {
											INavigator navigator = HyadesUtil.getActiveNavigator();
											if (navigator == null)
												return;
											if (HyadesUtil.getMofObject() != processor.getContext().getAgentProxy())
												return;

											String statusBarMessage = "";
											IStatusLineManager statusLineManager = navigator.getViewSite().getActionBars().getStatusLineManager();
											statusLineManager.setMessage(statusBarMessage);
										}
									});
								}
								*/
								return super.cancel();
							}
						}, 0, 1000);

						if (fStartPercentage == 0 && fStopPercentage == 100) {

							monitoredInputStream = new MonitoredInputStream(inputStream) {
								protected int afterRead(int readBytes) {
									totalBytesRead += readBytes;
									return readBytes;
								}

								protected boolean isCanceled() {
									if (monitor.isCanceled()) {
										unitSize = 0;
										notifyStopCollecting();
										return true;
									} else
										return false;
								}
								public void close() throws IOException {
									super.close();
									unitSize=-1;
								}
							};

						} else {
							final long startPos = (long) (entryLength * (fStartPercentage / 100));
							final long endPos = (long) (entryLength * (fStopPercentage / 100));

							if (startPos > 0)
								processor.getContext().setLoadToModel(false);
							monitoredInputStream = new MonitoredInputStream(inputStream) {
								int state = 0;

								protected int afterRead(int readBytes) {
									if (state == 2)
										return -1;
									totalBytesRead += readBytes;
									return readBytes;
								}

								protected int beforeRead(int maxBytesToRead) {
									if (state == 2)
										return 0;
									switch (state) {
									case 0:
										if (totalBytesRead == startPos) {
											processor.getContext().setLoadToModel(true);
											state++;
										} else if (totalBytesRead + maxBytesToRead > startPos) {
											maxBytesToRead = (int) (startPos - totalBytesRead);
										}
										break;
									case 1:
										if (totalBytesRead == endPos) {
											processor.getContext().setLoadToModel(false);
											unitSize = -1;
											maxBytesToRead = 0;
											state++;
										} else if (totalBytesRead + maxBytesToRead > endPos) {
											maxBytesToRead = (int) (endPos - totalBytesRead);
										}
										break;
									default:
										break;
									}
									return maxBytesToRead;
								}

								protected boolean isCanceled() {
									if (monitor.isCanceled()) {
										unitSize = 0;
										notifyStopCollecting();
										return true;
									} else
										return false;
								}
								
								public void close() throws IOException {
									super.close();
									unitSize=-1;
								}
							};
						}
						startTime = System.currentTimeMillis();
						prevTime = startTime;
						taskName = entryName + " " + MessageFormat.format(TraceWizardMessages.IMPORT_ENTRY, new String[] { "" + entryLength });
						monitor.beginTask(taskName, 100);
						//taskName = MessageFormat.format(TraceWizardMessages.IMPORT_PROGRESS, new String[] { "0", "0", "0", "0","0", "0" });
						//monitor.subTask(taskName);

						try {
							synchronized (ImportTracePage1.this.importJobLock) {
								processor.loadEvents(monitoredInputStream, 0, -1);
							}
						} catch (Exception e) {
							notifyStopCollecting();
							if (!(monitor.isCanceled() || unitSize == -1)) {
								unitSize = -1;
								long deltaTime = (System.currentTimeMillis() - startTime) / 1000;
								if (deltaTime == 0)
									deltaTime = 1;
								progress = (int)(totalBytesRead*100/entryLength);
								taskName = entryName + " " + MessageFormat.format(TraceWizardMessages.IMPORT_ENTRY, new String[] { "" + entryLength });
								taskName = taskName + " " + MessageFormat.format(TraceWizardMessages.IMPORT_PROGRESS, new String[] { "" + progress, "" + processor.getProcessedFragments(), "" + (processor.getProcessedFragments() / deltaTime), "" + deltaTime, "" + totalBytesRead, "" + (totalBytesRead / deltaTime) });
								processor.cleanUp();
								return new Status(IStatus.ERROR, UIPlugin.getDefault().getName(), IStatus.ERROR, "[" + TraceWizardMessages.IMPORT_ERROR + "] " + jobName + " - " + taskName, e);
							}
						}
						monitor.done();
						long deltaTime = (System.currentTimeMillis() - startTime) / 1000;
						if (deltaTime == 0)
							deltaTime = 1;
						progress = (int)(totalBytesRead*100/entryLength);
						taskName = entryName + " " + MessageFormat.format(TraceWizardMessages.IMPORT_ENTRY, new String[] { "" + entryLength });
						taskName = taskName + " " + MessageFormat.format(TraceWizardMessages.IMPORT_PROGRESS, new String[] { "" + progress, "" + processor.getProcessedFragments(), "" + (processor.getProcessedFragments() / deltaTime), "" + deltaTime, "" + totalBytesRead, "" + (totalBytesRead / deltaTime) });
						if (unitSize == 0) {
							unitSize = -1;
							IStatus s = new Status(IStatus.CANCEL, UIPlugin.getDefault().getName(), IStatus.CANCEL, "[" + TraceWizardMessages.IMPORT_CANCELED + "] " + jobName + " - " + taskName, null);
							if(ModelDebugger.INSTANCE.debug)
								UIPlugin.getDefault().log(s);
							notifyStopCollecting();
							processor.cleanUp();
							return s;
						} else {
							unitSize = -1;
							IStatus s = new Status(IStatus.OK, UIPlugin.getDefault().getName(), IStatus.OK, "[" + TraceWizardMessages.IMPORT_DONE + "] " + jobName + " - " + taskName, null);
							if(ModelDebugger.INSTANCE.debug)
								UIPlugin.getDefault().log(s);
							notifyStopCollecting();
							processor.cleanUp();
							return s;
						}
					}


					protected void notifyStopCollecting() {
						Display d = Display.getDefault();
						if (d != null && !d.isDisposed()) {
							d.syncExec(new Runnable() {
								public void run() {
									if (processor.getContext().getAgentProxy() != null && processor.getContext().getAgentProxy().isCollectionData()) {
										processor.getContext().getAgentProxy().setCollectionData(false);
									}
									ProfileEvent event = UIPlugin.getDefault().getProfileEvent();
									if(unitSize==-1)
									{
	
										event.setSource(null);
										event.setType(ProfileEvent.REFRESH_PD_EXPLORER);
										UIPlugin.getDefault().notifyProfileEventListener(event);
										if(collectingState==0)
										{
											event = UIPlugin.getDefault().getProfileEvent();
											event.setSource(processor.getContext().getAgentProxy());
											event.setType(ProfileEvent.START_MONITOR);
											UIPlugin.getDefault().notifyProfileEventListener(event);
										}

										event = UIPlugin.getDefault().getProfileEvent();
									}

									event.setSource(processor.getContext().getAgentProxy());
									event.setType(ProfileEvent.STOP_COLLECTING);
									UIPlugin.getDefault().notifyProfileEventListener(event);
									if(ModelDebugger.INSTANCE.debug)
										ModelDebugger.log("Import refresh thread - set STOP_COLLECTING "+processor.getContext().getAgentProxy());
									
									event = UIPlugin.getDefault().getRefreshViewEvent(processor.getContext().getAgentProxy());
									UIPlugin.getDefault().notifyProfileEventListener(event);
								}
							});
						}
					}
				};

				mon.done();
				job.schedule();

				return;
			}

			boolean cancelOperation = false;
			int l = 1;
			int lastValidLine = 1;

			try {
				int bSize = 65536; //length of each pipe
				int bufferSize = bSize; //total size of pipes.
				byte[] buffer = new byte[bufferSize];
				byte[] forwardBuffer = new byte[bufferSize];

				long position = 0;
				int bytesRead = 0;
				double startLength = logLength / 100;

				startLength = startLength * fStartPercentage;

				double endLength = logLength;

				if (fStopPercentage != 100) {
					endLength = endLength / 100;
					endLength = endLength * fStopPercentage;
				}

				mon.beginTask(fInputLog, (int) (endLength - startLength));

				int forwardOffset = 0;
				int offset = 0;

				while (bytesRead > -1) {
					if (mon.isCanceled()) {
						readStream.close();
						cancelOperation = true;
						throw new OperationCanceledException();
					}

					offset = 0;
					bytesRead = readStream.read(buffer);

					readLine: while (offset < bytesRead) {
						forwardBuffer[forwardOffset++] = buffer[offset];

						if (forwardOffset == bufferSize) {
							byte[] tempForwardBuffer = forwardBuffer;

							bufferSize = bufferSize + bSize;
							forwardBuffer = new byte[bufferSize];
							System.arraycopy(tempForwardBuffer, 0, forwardBuffer, 0, forwardOffset);

							continue readLine;
						}

						if (buffer[offset++] == '\n') {
							l++;

							if (forwardBuffer[1] != '?') {
								try {
									position += forwardOffset;

									if ((position > endLength) && (endLength != log.length())) {
										byte[] end = (new String("</TRACE>")).getBytes();

										processor.loadEvent(end, 8, true);

										// 									processor.resetDeferredData();
										mon.done();
										readStream.close();

										return;
									}

									boolean loadToModel = loadLine(l, position, startLength, endLength);

									processor.loadEvent(forwardBuffer, forwardOffset, loadToModel);

								} catch (InvalidXMLException e) {

									fError = NLS.bind(TraceMessages.INVXML_ER, ((InvalidXMLException) e).getEnclosedException().getMessage());

									mon.done();
									readStream.close();
									return;

								} catch (java.lang.OutOfMemoryError me) {
									readStream.close();
									handleOutOfMemoryError();
									throw me;
								}
							}

							lastValidLine = l;
							mon.worked(forwardOffset);
							forwardOffset = 0;
						}
					}

					fError = null;
				}

				if (lastValidLine != l) {
					readStream.close();
					fError = NLS.bind(TraceMessages.EOFR_ERR, new Object[] { fInputLog, String.valueOf(lastValidLine) });

					return;
				}
			} catch (IOException e) {
				fError = NLS.bind(TraceMessages.IMP_IOEXC, e.getMessage());
			} catch (Exception e) {
				//System.out.println(e.getMessage());
				//e.printStackTrace();
				if (cancelOperation) {
					fError = "";
				} else {
					fError = TraceMessages.IMP_EXCERR;
				}

				fError = NLS.bind(fError, new Object[] { fInputLog, String.valueOf(lastValidLine), e.getMessage() });
			}

			if (readStream != null) {
				try {
					readStream.close();
					readStream = null;
				} catch (IOException ioexc) {
				}
			}

			mon.done();
		} finally {
			if (CommunicationDebug.INSTANCE.debugUseEventMode) {
				if (readStream != null) {
					try {
						processor.cleanUp();
					} catch (Exception e) {
					}

					try {
						readStream.close();
						readStream = null;
					} catch (IOException ioexc) {
					}
				}
			}
			mon.done();
		}
	}


	private boolean loadLine(int line, double position, double startLength, double endLength) {
		if (line <= 7) {
			return true; //trace start events
		}

		if ((position >= startLength) && (position <= endLength)) {
			return true;
		}

		return false;
	}

	/**
	 * Insert the method's description here.
	 * Creation date: (8/18/2001 2:50:10 PM)
	 * @return boolean
	 */
	private boolean loadXMLTrace() {
		/* find out if the perspective should change */
		IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
		boolean shouldSwitch = CommonUIPlugin.getDefault().openPerspectiveSwitchDialog(window.getShell(), PDPerspective.ID_TRACE_PERSPECTIVE, CommonUIPlugin.getDefault().getPreferenceStore(), HyadesUI.SWITCH_TO_PERSPECTIVE_KEY, TraceWizardMessages.CONFIRM_PSWITCH);
		if (shouldSwitch) {
			UIPlugin.getActivePage();
		}
		if (PDPerspective.ID_TRACE_PERSPECTIVE.equals(window.getActivePage().getPerspective().getId())) {
			try {
				PDProjectExplorer nav = (PDProjectExplorer) UIPlugin.getActivePage().showView(PDPerspective.ID_PD_NAVIGATOR_VIEW);
				nav.getViewer().refresh();
			} catch (Exception e) {
				CommonPlugin.logError(e);
			}
		}

		//create destination container
		IContainer res = PDCoreUtil.createContainer(new Path(_location.getLocation().getText().trim()));

		if (res == null) {
			return false;
		}

		//create monitor
		TRCMonitor mon = PDCoreUtil.createMonitor(res, _location.getMonitor().getText().trim());

		fError = createModel(mon, sourceNameField.getText().trim());

		ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

		event.setSource(null);
		event.setType(ProfileEvent.UNSPECIFIED);
		UIPlugin.getDefault().notifyProfileEventListener(event);

		if (fError != null) {
			if (!fError.equals("")) {
				Status err1 = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.INTERNAL_ERROR, fError, null);

				ErrorDialog.openError(getShell(), TraceMessages.TRC_MSGT, TraceMessages.ERR_LXML, err1);
			}

			return false;
		}

		// Bug 322951
		ITraceModelInteraction[] contributors = TraceModelInteraction.getContributors();
		// Allow implementers of the model interaction extension point to contribute to the load action
		// such as on import of a profile file.
        if(contributors != null && contributors.length > 0) {
        	for(ITraceModelInteraction contributor : contributors) {
        		contributor.traceModelLoad(mon.eResource());
        	}
        }
		return true;
	}

	//~ Inner Classes ------------------------------------------------------------------------------

	static final class FileInfo {
		InputStream inputStream;

		long fileLength;
	}

	public void setTraceFile(String fullPath) {
		sourceNameField.setText(fullPath);
	}
}

class MyBufferedInputStream extends BufferedInputStream{

	public MyBufferedInputStream(InputStream in, int size) {
		super(in, size);
	}
	public void addRootTRACEElement() {
		try {
			byte[] buf1 = new byte[buf.length+8*1024];
			System.arraycopy("<TRACE>".getBytes(), 0,buf1,0,"<TRACE>".getBytes().length);
			System.arraycopy(buf, 0,buf1,"<TRACE>".getBytes().length,count);
			count+="<TRACE>".getBytes().length;
			pos=0;
			buf=buf1;
		} catch (Exception e) {
			e.printStackTrace();
		}
}
	
}
