/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ProfileSetWizard.java,v 1.9 2005/08/09 02:48:49 popescu Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.internal.wizard;

import java.util.Iterator;
import java.util.List;

import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.core.TraceProfileOverviewUI;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfilingSet;
import org.eclipse.hyades.trace.ui.internal.util.FilterTableElement;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.hyades.trace.ui.launcher.ProfilingSetsManagerCopy;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.IWorkbench;

public abstract class ProfileSetWizard extends Wizard {
	
	private ProfileSetWizardPage1 wizPage1;
	private ProfileSetWizardPage2 wizPage2;
	private ProfileSetWizardPage3 wizPage3;
	private ProfilingSetsManagerCopy original;
	private ProfilingSetsManagerCopy copy;
	
	private String _name;
	private ProfilingSet set;

	
	private TraceProfileOverviewUI _overviewUI;
	
	public ProfileSetWizard(TraceProfileOverviewUI overviewUI)
	{
		super();
		original = overviewUI.getManagerCopy();
		copy = original.createCopy();
		_overviewUI = overviewUI;			
	}
	
	public void initializeSet(ProfilingSet profilingSet) {
		set = profilingSet;
		copy.setDefaultSet(set);		
	}
	
	public void init(IWorkbench workbench, IStructuredSelection selection) {
	}
	
	protected abstract String getSetId();
	
	public boolean profileNameExists(String name)
	{
		if (_name != null && name.equals(_name.trim()))
			return false;
		else
		{
			Iterator iter = copy.getProfilingSets().values().iterator();
			
			while (iter.hasNext())
			{
				/*
				 * This is a copy of an original set, so skip the original when checking for
				 * duplicate names.
				 */
				ProfilingSet next = (ProfilingSet)iter.next();
				if ((set == null ||  (set != null && !set.getId().equals(next.getId()))) && name.equals(next.getDisplayName().trim()))
					return true;
			}
			
			return false;
		}
	}
	
	public void setProfileSetName (String name) {
		if (!set.getDisplayName().equals(name)) {
			set.setName(name);
		}
	}
	
	public void setProfileSetDescription (String description) {	
		if (!set.getDisplayDescription().equals(description)) {
			set.setDescription(description);
		}
	}	
	
	public void setNameToPages()
	{
		wizPage2.setProfilingSetName(set.getDisplayName());
	}
	
	public boolean performFinish() {
		if (!validateFilterPatterns(copy.getDefaultFilterSet().getChildren()))
			return false;
		
		
		original.resetFrom(copy);
		original.applyChanges();

		_overviewUI.dataChanged();
		_overviewUI.getTableViewer().refresh(true);
		_overviewUI.getTableViewer().setSelection(new StructuredSelection(set));
		
		return true;
	}
	
	public ProfilingSetsManagerCopy getProfilingSetsManagerCopy() {
		return copy;
	}	
	
	public void addPages() {
		super.addPages();
		
		wizPage1 = new ProfileSetWizardPage1("name", set.getDisplayName(), set.getDisplayDescription(), copy);			
		wizPage1.setTitle(TraceMessages.APRFS_ND);
		wizPage1.setDescription(TraceMessages.APRFS_N);
		addPage(wizPage1);
		
		wizPage2 = new ProfileSetWizardPage2("profilingTypes", set.getDisplayName(), copy, _overviewUI);			
		wizPage2.setTitle(TraceMessages.EPRFS_D);
		wizPage2.setDescription(TraceMessages.EPRF_D);
		addPage(wizPage2);		
		
		wizPage3 = new ProfileSetWizardPage3("filters", _overviewUI);
		wizPage3.setTitle(TraceMessages.EFS);
		wizPage3.setDescription(TraceMessages.EFS_DE);
		addPage(wizPage3);
	}	
	
	private boolean isBadFilterPattern(String filterStr)
	{
		if (filterStr.length() <= 1)
			return false;
		else if (filterStr.indexOf('*') != filterStr.lastIndexOf('*'))
			return true;		
		else if (filterStr.substring(1, filterStr.length()-1).indexOf('*') >= 0)
			return true;
		
		return false;
	}
	
	private boolean validateFilterPatterns(List filterList)
	{
		String badFilters = "";
		String nl = System.getProperty("line.separator"); 
		
		if (filterList != null)
		{
			for (int i = 0; i < filterList.size(); i++)
			{
				FilterTableElement fte = (FilterTableElement)filterList.get(i);
				
				if (isBadFilterPattern(fte.getText()))
					badFilters = badFilters + fte.getText() + nl;
				
				if (isBadFilterPattern(fte.getMethod()))
					badFilters = badFilters + fte.getMethod() + nl;
				
			}
		}
		
		if ("".equals(badFilters))
			return true;
		else
		{
			String message = TraceMessages.ST_FLTPW;
			message = message + nl + nl+ badFilters;
			message = message + nl + TraceMessages.ST_FLTPWC;
			
			String[] buttons = {
					UIPlugin
							.getResourceString(TraceMessages.ST_FLTPWY),
					UIPlugin
							.getResourceString(TraceMessages.ST_FLTPWN) };
			
			MessageDialog messageDialog = new MessageDialog(
					getShell(),
					TraceMessages.ST_FLTPWT,
					null, message, MessageDialog.WARNING, buttons, 0);
			
			messageDialog.create();

			return messageDialog.open() == MessageDialog.OK;
		}		
	}	
}
