/**********************************************************************
 * Copyright (c) 2005, 2006 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ProfilingSetsManagerCopy.java,v 1.9 2006/12/29 19:16:46 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.trace.ui.launcher;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.hyades.trace.ui.internal.launcher.IProfileLaunchConfigurationConstants;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfilingSet;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfilingSetType;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfilingSetTypeGroup;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfilingSetsManager;
import org.eclipse.hyades.trace.ui.internal.util.FilterSetElement;
import org.eclipse.hyades.trace.ui.internal.util.FilterTableElement;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.tptp.platform.common.internal.CommonPlugin;
import org.eclipse.tptp.trace.ui.internal.launcher.core.AnalysisType;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherConstants;

/**
 * This class is used to store changes to the LCD(Launch Configuration Dialog), before the user hits apply.
 * Necessary for the case when user reverts, or cancel.
 * During a save action, this manager copy instance gets sent to the manager.
 * During a cancel action, the manager copy info is not persisted.
 * 
 * @author znensi
 */
public class ProfilingSetsManagerCopy extends ProfilingSetsManager {
	
	private HashMap filterSets;

	/**
	 * This class will be used to keep temporary information regarding profiling sets
	 */
	public ProfilingSetsManagerCopy() {
		super();
		initialize();
	}
	
	/**
	 * Creates the copy information given a launch configuration.
	 * @param conf an ILaunchConfiguration instance, the launch configuration
	 */
	public void initializeFrom(ILaunchConfiguration conf)
	{
		String defaultProfilingSet = getDefaultSet().getId();
		
		try 
		{
			String profilingSetId = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_SET, defaultProfilingSet);
			String filterSetId = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_FILTER_SET, LauncherConstants.DEFAULT_FILTER_ID);
			
			IProfilingSet profilingSet = (IProfilingSet)getProfilingSets().get(profilingSetId);
			FilterSetElement filterSet = (FilterSetElement)getFilterSets().get(filterSetId);
			
			if (profilingSet == null) {
				profilingSet = getInitialDefaultSet(null);
			}
			
			/* Filter set is null */
			if (filterSet == null) 
				filterSet = findNextAvailableFilterSet();
			
			
			setDefaultSet(profilingSet);
			setDefaultFilterSet(filterSet);
		}
		catch(CoreException exc) {
			CommonPlugin.logError(exc);
		}
	}
	
	/**
	 * Creates an instance of the ProfilingSetsManager, this should not be called
	 * for ProfilingSetsManagerCopy. 
	 */
	public static ProfilingSetsManager instance()
	{
		//this method is not supposed to be called on a copy instance
		return null;
	}
	
	/**
	 * Apply any changes to the main profiling set manager
	 */
	public void applyChanges()
	{
		//update profiling set manager
		applyManagerChanges();		
		writeSetsToPreferences();				
	}
	
	
	public void applyManagerChanges()
	{
		//update profiling set manager
		ProfilingSetsManager manager = ProfilingSetsManager.instance();
		
		manager.getProfilingSets().clear();
		manager.getProfilingSets().putAll(getProfilingSets());
		
		manager.getProfilingTypes().clear();
		manager.getProfilingTypes().putAll(getProfilingTypes());
		
		manager.getProfilingTypesGroups().clear();
		manager.getProfilingTypesGroups().putAll(getProfilingTypesGroups());
		
		manager.setDefaultSet(manager.getInitialDefaultSet(null));
		
		manager.setDefaultFilterSet(getDefaultFilterSet());		
	}
		
	/**
	 * Initialize the profiling types groups.
	 * @deprecated
	 */
	protected void initializeProfilingTypesGroups()
	{
		profilingTypesGroups = new HashMap();
		
		Map groups = ProfilingSetsManager.instance().getProfilingTypesGroups();
		Iterator keys = groups.keySet().iterator();
		while(keys.hasNext())
		{
			String key = keys.next().toString();
			
			IProfilingSetTypeGroup group = (IProfilingSetTypeGroup) groups.get(key);
			profilingTypesGroups.put(key, new ProfilingSetTypeGroup(group.getId(), group.getName(), group.getDescription(), group.getIcon()));
		}		
	}
	
	/**
	 * Initialize the profiling types.
	 */
	protected void initializeProfilingTypes()
	{
		profilingTypes = new HashMap();
		
		Map types = ProfilingSetsManager.instance().getProfilingTypes();
		Iterator keys = types.keySet().iterator();
		while(keys.hasNext())
		{
			String key = keys.next().toString();
			
			IProfilingSetType type = (IProfilingSetType) types.get(key);
			
			/* Ali M. -- We need to maintain the type */
			Object value;
			if (type instanceof AnalysisType)
				value = ((AnalysisType)type).clone();
			else
				value =  new ProfilingSetType(type.getId(), type.getName(), type.getDescription()
			    		, type.getImage(), type.isUseFilters(), type.getGroup()
						, type.getProfilingType());
												
			profilingTypes.put(key, value);
		}		
	}

	/**
	 * Initialize the profiling sets.
	 */
	protected void initializeProfilingSets()
	{
		profilingSets = new HashMap();
		
		IProfilingSet defaultSet = ProfilingSetsManager.instance().getDefaultSet();
		Map sets = ProfilingSetsManager.instance().getProfilingSets();
		Iterator keys = sets.keySet().iterator();
		while(keys.hasNext())
		{
			String key = keys.next().toString();
			
			IProfilingSet set = (IProfilingSet) sets.get(key);
			
			IProfilingSet copy = new ProfilingSet(set.getId(), set.getName(), set.getDescription());
            copy.getProfilingTypes().addAll(set.getProfilingTypes());
            copy.getAttributes().putAll(set.getAttributes());
			
			profilingSets.put(key, copy);
			
			if (set == defaultSet) {
				setDefaultSet(copy);
			}
		}				
	}
	
	/**
	 * Initialize the filter sets.
	 *
	 */
	protected void initializeFilterSets()
	{
		filterSets = new HashMap();
		
		Map sets = ProfilingSetsManager.instance().getFilterSets();
		Iterator keys = sets.keySet().iterator();
		while(keys.hasNext())
		{
			String key = keys.next().toString();
			
			FilterSetElement set = (FilterSetElement) sets.get(key);
			
			FilterSetElement copy = new FilterSetElement(set.getId());
			copy.setKey(set.getKey());
			copy.setName(set.getName());
			ArrayList children = set.getChildren();
			ArrayList ccopy = new ArrayList();
			for(int idx=0; idx<children.size(); idx++)
			{
				FilterTableElement child = (FilterTableElement)children.get(idx);
				ccopy.add(new FilterTableElement(child.getText(), child.getMethod(), child.getVisibility()));
			}
			copy.setChildren(ccopy);
			
			filterSets.put(key, copy);
		}		
	}
	
	/**
	 * Applies the given launch configuration working copy to this profiling sets manager copy.
	 * @param wc an ILaunchConfigurationWorkingCopy instance, the given launch configuration working copy
	 */
	public void performApply(ILaunchConfigurationWorkingCopy wc)
	{
		IProfilingSet set = getDefaultSet();

		if(set != null) {
			FilterSetElement filterSet = getDefaultFilterSet();
			String filterSetId = null;
			if (filterSet != null) {
				filterSetId = filterSet.getId();
			}
			
			/*
			 * #59620: All we should be saving in the launch configuration
			 * is the reference to the profiling and filter set, not their
			 * actual contents.
			 */
			wc.setAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_SET, set.getId());
			wc.setAttribute(IProfileLaunchConfigurationConstants.ATTR_FILTER_SET, filterSetId);

		}
	}
	
	/**
	 * Creates a copy of this ProfilingSetsManagerCopy instance.
	 * @return a ProfilingSetsManagerCopy instance, the copied instance
	 */
	public ProfilingSetsManagerCopy createCopy() {
		ProfilingSetsManagerCopy copy = new ProfilingSetsManagerCopy();
		copy.resetFrom(this);
		return copy;
	}
	
	/**
	 * Resets the data in this instance from another ProfilingSetsManagerCopy instance.
	 * @param other the other ProfilingSetsManagerCopy instance
	 */
	public void resetFrom(ProfilingSetsManagerCopy other) {
		getProfilingSets().clear();
		getProfilingSets().putAll(other.getProfilingSets());
		
		getProfilingTypes().clear();
		getProfilingTypes().putAll(other.getProfilingTypes());
		
		getProfilingTypesGroups().clear();
		getProfilingTypesGroups().putAll(other.getProfilingTypesGroups());

		
		setDefaultSet(other.getDefaultSet());
		setDefaultFilterSet(other.getDefaultFilterSet());
	}
	
	
}
