/**********************************************************************
 * Copyright (c) 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.tptp.trace.ui.internal.control.provider;

import java.util.ArrayList;
import java.util.Hashtable;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;
import org.eclipse.tptp.trace.ui.provisional.control.provider.IControlItem;
import org.eclipse.tptp.trace.ui.provisional.control.provider.IControlProvider;


/**
 * Loads and managers the extension org.eclipse.hyades.trace.ui.controllableEntity
 * 
 * @author Ali Mehregani
 */
public class ControlProviderManager
{	
	/* Indicates an all inclusive input id */
	public static final String ALL = "ALL";
	
	/* Supported entities */
	public static final String PROCESS_ENTITY = "process";
	public static final String AGENT_ENTITY = "agent";
	public static final String ANALYSIS_ENTITY = "analysis";
	public static final String TOOLBAR_ENTITY = "toolbar";

	/* The key used to determine the ordering of the control items stored in a hashtable */
	public static final String INSERTED_ORDER = "ORDER";
	
	/* For backward compatibility reasons, we use a default set of control options if a particular entry
	 * deosn't have an associated entry */
	private static final String DEFAULT_AGENT_DECLARATION_ID = "org.eclipse.tptp.trace.ui.jvmpiAgent";
	
	/* Groups supported entities */
	public static final ArrayList SUPPORTED_ENTITIES = new ArrayList();
	
	static
	{
		SUPPORTED_ENTITIES.add(PROCESS_ENTITY);
		SUPPORTED_ENTITIES.add(AGENT_ENTITY);
		SUPPORTED_ENTITIES.add(ANALYSIS_ENTITY);
		SUPPORTED_ENTITIES.add(TOOLBAR_ENTITY);
	}
	

	/* The instance for this singleton class */
	private static ControlProviderManager instance;
	
	/* Stores the information for each extension
	 * KEY = agent declaration id
	 * VALUE = Hashtable 
	 * 		   { KEY = supported entities (see SUPPORTED_ENTITIES)
	 * 			 VALUE = Hashtable
	 * 				{
	 * 					KEY = input id
	 * 					VALUE = A list of control providers
	 * 				}
	 * 		   } 
	 * 		
	 */
	private Hashtable controllableEntities;
	
	
	/**
	 * Hide the constructor
	 */
	private ControlProviderManager()
	{
		controllableEntities = new Hashtable();
		initialize();
	}
	

	/**
	 * Return the instance of this singleton class
	 */
	public static ControlProviderManager getInstance()
	{
		if (instance == null)
			instance = new ControlProviderManager();
		return instance;
	}
	
	
	private void initialize()
	{
		IConfigurationElement[] elements = Platform.getExtensionRegistry().getConfigurationElementsFor("org.eclipse.hyades.trace.ui.controllableEntity");
	
		for (int i = 0; i < elements.length; i++)
		{
			if (!"control".equalsIgnoreCase(elements[i].getName()))
				continue;
			
			String agentDeclarationId = elements[i].getAttribute("associatedAgentId");
			if (agentDeclarationId == null)
				continue;
			
			IConfigurationElement[] entities = elements[i].getChildren("entity");			
			for (int j = 0; j < entities.length; j++)
			{
				ControlProviderLoader controllableEntity = ControlProviderLoader.constructInstance(entities[j]);			
				if (controllableEntity != null)
				{
					Hashtable entitiesHash = (Hashtable)LauncherUtility.getHashtableEntry(controllableEntities, agentDeclarationId, Hashtable.class, true);
					
					/* We don't support the entity type */
					if (!SUPPORTED_ENTITIES.contains(controllableEntity.getType()))
						continue;

					Hashtable inputIdHash = (Hashtable)LauncherUtility.getHashtableEntry(entitiesHash, controllableEntity.getType(), Hashtable.class, true);
					String inputId = entities[j].getAttribute("inputId");
					if (inputId == null)
						inputId = ALL;
					
					ArrayList controlProviders = (ArrayList)LauncherUtility.getHashtableEntry(inputIdHash, inputId, ArrayList.class, true);
					controlProviders.add(controllableEntity);
				}				
			}			
		}
	}


	/**
	 * For backward compatability reasons, we'll need to return a default set
	 * of control options for entities that don't have controllable options
	 * contributed. 
	 * @param entity One of the entities supported
	 * @param input The input
	 * @return The default set of control options for the entry passed in.
	 */
	public Hashtable getDefaultControlItemSet(String entity, StructuredSelection input)
	{
		return getControlItemSet (DEFAULT_AGENT_DECLARATION_ID, entity, input);
	}


	/**
	 * Returns the control item set for the agent declaration and the entity
	 * passed in.
	 * 
	 * @param agentDeclarationId The id of the agent declaration.
	 * @param entity One of the supported entities.
	 * @param input The input
	 * @param inputId The input Id
	 * @return Control items for the agent and entity passed in.
	 */
	public Hashtable getControlItemSet(String agentDeclarationId, String entity, StructuredSelection input, String inputId)
	{
		Hashtable controlOptions = (Hashtable)controllableEntities.get(agentDeclarationId);
		Hashtable controlItemsSet = new Hashtable();
		ArrayList orderQueue = new ArrayList();
		controlItemsSet.put(INSERTED_ORDER, orderQueue);
		
		if (controlOptions == null)
			return controlItemsSet;
		
		Hashtable inputIdHash = (Hashtable)controlOptions.get(entity);
		if (inputIdHash == null)
			return controlItemsSet;
		
		ArrayList controlProviders = (ArrayList)inputIdHash.get(inputId);
		
		if (!inputId.equals(ALL))
		{
			if (controlProviders == null)
				controlProviders = new ArrayList();
			ArrayList allControlProviders = (ArrayList)inputIdHash.get(ALL);
			if (allControlProviders != null)
				controlProviders.addAll(allControlProviders);
		}
		if (controlProviders == null)
			return controlItemsSet;
		
		/* Walk through each of the control providers and get a mutual set of control options that have
		 * been contributed */
		for (int i = 0, controlProviderSize = controlProviders.size(); i < controlProviderSize; i++)
		{
			ControlProviderLoader currentControlProvider = (ControlProviderLoader)controlProviders.get(i);
			IControlProvider controlProvider = currentControlProvider.getControlProvider();
			
			controlProvider.setInput(input);
			IControlItem[] controlItems = controlProvider.createControlItems();
			
			for (int j = 0; j < controlItems.length; j++)
			{
				String key = controlItems[j].getText();
				if (key == null)
					key = String.valueOf(orderQueue.size());
				
				if (controlItemsSet.get(key) == null)
				{					
					controlItemsSet.put(key, controlItems[j]);
					orderQueue.add(key);
				}
					
			}			
		}
				
		return controlItemsSet;
	}
	
	
	public Hashtable getControlItemSet(String agentDeclarationId, String entity, StructuredSelection input)
	{
		return getControlItemSet(agentDeclarationId, entity, input, ALL);
	}
}
