/**********************************************************************
 * Copyright (c) 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.tptp.trace.ui.internal.control.provider;

import java.util.ArrayList;
import java.util.Hashtable;

import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.trace.internal.ui.PDProjectViewer;
import org.eclipse.hyades.trace.ui.internal.navigator.ProfileDetailItem;
import org.eclipse.hyades.ui.extension.INavigatorItem;
import org.eclipse.hyades.ui.internal.navigator.INavigator;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.tptp.trace.ui.internal.launcher.core.AgentDeclarationManager;
import org.eclipse.tptp.trace.ui.internal.launcher.core.AnalysisType;
import org.eclipse.tptp.trace.ui.provisional.control.provider.IControlItem;


/**
 * Reads in the contributed items for the selective-sensitive entities.
 * 
 * @author Ali Mehregani
 */
public class TraceControlItemManager
{	
	/** Constants identifying the container type for the controls */
	private static final byte CONTEXT_MENU = 0x00;
	private static final byte TOOLBAR = 0x01;
	
	
	/** Points to the navigator */
	private static INavigator navigator;
		
	/** The tree of the profiling monitor view */
	private Tree profilingMonitorTree;
	
	public TraceControlItemManager (INavigator navigator)
	{
		TraceControlItemManager.navigator = navigator;		
	}

	private IControlItem[] getContributedItems(byte containerType)
	{
		if (profilingMonitorTree == null)
			profilingMonitorTree = ((PDProjectViewer)navigator.getViewer()).getTree();
		
		/* We'll need to construct the list of control items based on the tree selection */
		TreeItem[] treeItemSelection = profilingMonitorTree.getSelection();
		
		/* A mutual set from all sets from each selected element:
		 * e.g.  If A and B are selected elements with the following control items sets respectively:
		 * {a, b, c} and {c, b, d}, then the mutual set {b, c} will be displayed */
		Hashtable controlSubset = new Hashtable();		
		for (int i = 0; i < treeItemSelection.length; i++)
		{
			Hashtable currentSet = null;
			
			/* Find the control items of the current item selected */
			currentSet = getControlItemSet (containerType, treeItemSelection[i]);
			
			/* This is the first control item set that's been resolved */
			if (i == 0)
				controlSubset = currentSet;
			/* We'll need to find a mutual subset between what we currently have and the new set */
			else
				controlSubset = findMutualSubset (controlSubset, currentSet);
		}
				
		/* Add the elements contributed */		
		ArrayList orderedList = (ArrayList)controlSubset.get(ControlProviderManager.INSERTED_ORDER);
		IControlItem[] controlItems = new IControlItem[0];
				
		if (orderedList != null)
		{
			controlItems = new IControlItem[orderedList.size()];
			for (int i = 0; i < controlItems.length; i++)
				controlItems[i] = (IControlItem)controlSubset.get(orderedList.get(i));			
		}
		
		return controlItems;
	}
	
	
	public IControlItem[] getContextMenuItems()
	{
		return getContributedItems(CONTEXT_MENU);
	}
	
	
	public IControlItem[] getToolbarItems()
	{
		return getContributedItems(TOOLBAR);
	}

	/**
	 * Return the mutual subset of the two subsets returned.
	 * 
	 * @param firstSet The first set
	 * @param secondSet The second set
	 * @return The mutual subset of firstSet and secondSet.
	 */
	private Hashtable findMutualSubset(Hashtable firstSet, Hashtable secondSet) 
	{
		Hashtable mutualSet = new Hashtable();
		ArrayList orderQueue = new ArrayList();
		mutualSet.put(ControlProviderManager.INSERTED_ORDER, orderQueue);
		
		if (firstSet == null || secondSet == null)
			return mutualSet;
		
		
		ArrayList orderedKeys = (ArrayList)firstSet.get(ControlProviderManager.INSERTED_ORDER);
		if (orderedKeys != null)
		{
			for (int i = 0, setSize = orderedKeys.size(); i < setSize; i++)
			{		
				Object currentKey = orderedKeys.get(i);
				Object currentValue =  secondSet.get(currentKey);
				if (currentValue != null && currentValue.equals(firstSet.get(currentKey)))
				{
					mutualSet.put(currentKey, currentValue);
					orderQueue.add(currentKey);
				}
			}
		}		
		return mutualSet;
	}

	
	/**
	 * Return the control item set of the tree item passed in.
	 * 
	 * @param containerType The type of container that the control items will be applied to
	 * @param treeItem The item whose control item will be returned.
	 * @return The control item of treeItem.
	 */
	private Hashtable getControlItemSet (byte containerType, TreeItem treeItem)
	{
		/* Is there a contributed set of items for this item ? */
		Object data = treeItem.getData();
		
		Hashtable controlItemsSet = new Hashtable();
		TRCAgentProxy agentProxy;
		ProfileDetailItem navigatorItem;
		String entityType; 
		
		/* A process */
		if (data instanceof TRCProcessProxy)
		{
			entityType = containerType == CONTEXT_MENU ? ControlProviderManager.PROCESS_ENTITY : ControlProviderManager.TOOLBAR_ENTITY;
			/* Look up the agents that this item has */
			EList agentList = ((TRCProcessProxy)data).getAgentProxies();
			
			/* If the process has no agents, then return the default control item set */
			int agentListCount = agentList.size();
			if (agentListCount <= 0)
				return ControlProviderManager.getInstance().getDefaultControlItemSet (entityType, new StructuredSelection(data));
			
			/* Iterate through the agents under this process and find their contributed items.
			 * Return the mutual set of the items defined for the process' agent */
			for (int i = 0; i < agentListCount; i++)
			{				
				agentProxy = (TRCAgentProxy)agentList.get(i);
												
				if (i == 0)
					controlItemsSet = getAgentControlItems(entityType, agentProxy, createInput (entityType, agentProxy));
				else
					controlItemsSet = findMutualSubset(controlItemsSet, getAgentControlItems(entityType, agentProxy, createInput (entityType, agentProxy)));
			}
		}
		
		/* An agent */
		if (data instanceof TRCAgentProxy)
		{
			entityType = containerType == CONTEXT_MENU ? ControlProviderManager.AGENT_ENTITY : ControlProviderManager.TOOLBAR_ENTITY;
			agentProxy = (TRCAgentProxy) data;
			controlItemsSet = getAgentControlItems(entityType, agentProxy, createInput (entityType, agentProxy));
		}
		
		/* An analysis type */	
		else if (data instanceof ProfileDetailItem && (navigatorItem = (ProfileDetailItem)data).isAnalysisType())
		{
			entityType = containerType == CONTEXT_MENU ? ControlProviderManager.ANALYSIS_ENTITY : ControlProviderManager.TOOLBAR_ENTITY;
			agentProxy = (TRCAgentProxy)treeItem.getParentItem().getData();
			controlItemsSet = getAgentControlItems(entityType, agentProxy, createInput (entityType, agentProxy), navigatorItem.getAnalysisType().getId());
		}
			
		
		return controlItemsSet;
	}

	
	private StructuredSelection createInput(String entity, TRCAgentProxy agentProxy)
	{
		TreeItem[] treeItemSelection = profilingMonitorTree.getSelection();
		ArrayList inputList = new ArrayList();
		
		String agentName = agentProxy.getName();
		String agentType = agentProxy.getType();
		
		/* Process */
		if (entity.equals(ControlProviderManager.PROCESS_ENTITY))
		{
			for (int i = 0; i < treeItemSelection.length; i++)
			{
				Object data = treeItemSelection[i].getData();
				if (!(data instanceof TRCProcessProxy))
					continue;
				
				addProcess((TRCProcessProxy)data, inputList, agentName, agentType);
			}
		}
		
		/* Agent */
		else if (entity.equals(ControlProviderManager.AGENT_ENTITY))
		{
			for (int i = 0; i < treeItemSelection.length; i++)
			{
				Object data = treeItemSelection[i].getData();
				if (!(data instanceof TRCAgentProxy))
					continue;
								
				addAgent(inputList, agentName, agentType, (TRCAgentProxy)data, data);									
			}
		}
		
		/* Analysis */
		else if (entity.equals(ControlProviderManager.ANALYSIS_ENTITY))
		{
			for (int i = 0; i < treeItemSelection.length; i++)
			{
				Object data = treeItemSelection[i].getData();				
				if (!(data instanceof INavigatorItem && ((INavigatorItem)data).getData() instanceof AnalysisType))
					continue;
							
				addAgent(inputList, agentName, agentType, (TRCAgentProxy)treeItemSelection[i].getParentItem().getData(), data);		
			}
		}
		
		/* Toolbar */
		else if (entity.equals(ControlProviderManager.TOOLBAR_ENTITY))
		{
			for (int i = 0; i < treeItemSelection.length; i++)
			{
				Object data = treeItemSelection[i].getData();
				if (data instanceof TRCProcessProxy)
					addProcess((TRCProcessProxy)data, inputList, agentName, agentType);
				else if (data instanceof TRCAgentProxy)
					addAgent(inputList, agentName, agentType, (TRCAgentProxy)data, data);
				else if (data instanceof ProfileDetailItem)
				{
					ProfileDetailItem profileDetailItem = (ProfileDetailItem)data;
					if (profileDetailItem.isAnalysisType())
					{
						TRCAgentProxy associatedAgentProxy = (TRCAgentProxy)treeItemSelection[i].getParentItem().getData();
						addAgent(inputList, agentName, agentType, associatedAgentProxy, associatedAgentProxy);
					}									
				}
					
			}
		}
		
		
		Object[] elements = new Object[inputList.size()];
		inputList.toArray(elements);
		return new StructuredSelection(elements);
	}

	private void addProcess(TRCProcessProxy processProxy, ArrayList inputList, String agentName, String agentType)
	{		
		EList agentProxies = processProxy.getAgentProxies();		
		for (int j = 0, agentCount = agentProxies.size(); j < agentCount; j++)
		{							
			if (addAgent(inputList, agentName, agentType, (TRCAgentProxy)agentProxies.get(j), processProxy))
				break;
			
		}
	}
	
	
	private boolean addAgent(ArrayList inputList, String targetAgentName, String targetAgentType, TRCAgentProxy agent, Object data)
	{
		if (agent.getName().equals(targetAgentName) && agent.getType().equals(targetAgentType))
		{
			inputList.add(data);
			return true;
		}
		
		return false;
	}
	/**
	 * Returns the control item for the entity type that is associated with an agent declaration
	 * that matches "agentProxy".
	 * 
	 * @param entityType The entity type
	 * @param agentProxy The agent proxy
	 * @param input The input
	 * @param inputId The id of the input that the control items should be applied to
	 * @return The control item of entity type associated with a declaration that matches agentProxy.
	 */
	private Hashtable getAgentControlItems(String entityType, TRCAgentProxy agentProxy, StructuredSelection input, String inputId)
	{
		Hashtable controlItemSet = new Hashtable();
		
		if (agentProxy == null)
			return controlItemSet;
		
		String agentName = agentProxy.getName();
		String agentType = agentProxy.getType();
		String agentDeclarationId = AgentDeclarationManager.getInstance().getAgentDeclarationId (agentName, agentType);
		
		if (agentDeclarationId == null)
			controlItemSet = ControlProviderManager.getInstance().getDefaultControlItemSet (entityType, input);
		else
			controlItemSet = ControlProviderManager.getInstance().getControlItemSet (agentDeclarationId, entityType, input, inputId);
		
		return controlItemSet;
	}
	
	private Hashtable getAgentControlItems(String entityType, TRCAgentProxy agentProxy, StructuredSelection input)
	{
		return getAgentControlItems(entityType, agentProxy, input, ControlProviderManager.ALL);
	}

	
	public static INavigator getNavigator()
	{
		return navigator;
	}

}
