/**********************************************************************
 * Copyright (c) 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.trace.ui.internal.control.provider.application;


import java.util.Iterator;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.hyades.trace.ui.internal.actions.CollectObjRefAction;
import org.eclipse.hyades.trace.ui.internal.actions.NewSnapshotAction;
import org.eclipse.hyades.trace.ui.internal.actions.RunGCAction;
import org.eclipse.hyades.trace.ui.internal.core.TraceUIImages;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceConstants;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;
import org.eclipse.tptp.trace.ui.provisional.control.provider.AbstractAgentControlProvider;
import org.eclipse.tptp.trace.ui.provisional.control.provider.ControlItem;
import org.eclipse.tptp.trace.ui.provisional.control.provider.IAgentStateModifier;
import org.eclipse.tptp.trace.ui.provisional.control.provider.IControlItem;
import org.eclipse.tptp.trace.ui.provisional.control.provider.StandardAgentStateModifier;


/**
 * The control provider for the PI agent 
 * @author Ali Mehregani
 */
public class PIAgentControlProvider extends AbstractAgentControlProvider
{	
	/** The id of the contributed items */
	private static final String RESET_ITEM = "org.eclipse.hyades.trace.ui.internal.popupMenu.NewSnapshot";
	private static final String HEAP_DUMP_ITEM = "org.eclipse.hyades.trace.ui.monitorsview.popupMenu.ObjRef";
	private static final String GC_ITEM = "org.eclipse.hyades.trace.ui.monitorsview.popupMenu.GC";
	
	/** The agent state modifier */
	private IAgentStateModifier agentStateModifier;
	
	/** The collect object reference control item */
	private IControlItem dumpHeapControlItem;
	
	/** The garbage collection control item */
	private IControlItem garbageCollectionControlItem;
	
	/** The restart control item */
	private IControlItem restartControlItem;

	
	public PIAgentControlProvider()
	{
		agentStateModifier = new StandardAgentStateModifier(); 
	}
	
	
    protected void initializeControlItems()
	{
    	super.initializeControlItems();
    	addControlItem(createResetControlItem());
    	addControlItem(createHeapDumpControlItem());
    	addControlItem(createGCControlItem());
	}

	public IAgentStateModifier getAgentStateModifier()
	{
		return agentStateModifier;
	}
    	
	
	/**
	 * Create the collect object reference control item
	 *  
	 * @return The collect object reference control item
	 */
	protected IControlItem createHeapDumpControlItem()
	{
		if (dumpHeapControlItem != null)
			return dumpHeapControlItem;
		
		dumpHeapControlItem = new HeapDumpControlItem();		
		return dumpHeapControlItem;
	}

	
	/**
	 * Create the garbage collection control item
	 *  
	 * @return The garbage collection control item
	 */
	protected IControlItem createGCControlItem()
	{
		if (garbageCollectionControlItem != null)
			return garbageCollectionControlItem;
		
		garbageCollectionControlItem = new GCControlItem();		
		return garbageCollectionControlItem;
	}
	
	
	/**
	 * Create the reset/new snap shot control item
	 *  
	 * @return The reset/new snap shot control item
	 */
	protected IControlItem createResetControlItem()
	{
		if (restartControlItem != null)
			return restartControlItem;
		
		class ResetControlItem extends ControlItem
		{
			private NewSnapshotAction resetAction;
			public ResetControlItem()
			{
				super(RESET_ITEM, CommonUITraceConstants.PROFILE_NEW_SNAPSHOT_GROUP, ControlMessages.CONTROL_ITEM_NEW_SNAPSHOT, null);
				resetAction = new NewSnapshotAction();
			}
			
			public void run()
			{
				try
				{
					for (Iterator agents = input.iterator(); agents.hasNext();)
						resetAction.doAction(agents.next());
				}
				catch (Exception e)
				{
					LauncherUtility.openMessageWithDetail(IStatus.ERROR, "", e.getMessage(), e);
				}
			}
			
			
			public boolean isEnabled()
			{
				boolean isEnable = true;
				for (Iterator agents = input.iterator(); isEnable && agents.hasNext();)
					isEnable = isEnable && resetAction.isEnabledFor(agents.next());
				return isEnable;
			}
				
		}
		
		restartControlItem = new ResetControlItem();		
		return restartControlItem;
	}
	
	
	public static class HeapDumpControlItem extends ControlItem
	{
		CollectObjRefAction collectObjRef;
		public HeapDumpControlItem()
		{
			super(HEAP_DUMP_ITEM, CommonUITraceConstants.PROFILE_GC_GROUP, ControlMessages.CONTROL_ITEM_DUMP, TraceUIImages.INSTANCE.getImageDescriptor("c", TraceUIImages.IMG_DUMP));
			setDisabledImageDescriptor(TraceUIImages.INSTANCE.getImageDescriptor("d", TraceUIImages.IMG_DUMP));
			collectObjRef = new CollectObjRefAction();
		}
		
		public void run()
		{
			try
			{
				for (Iterator agents = input.iterator(); agents.hasNext();)
					collectObjRef.doAction(agents.next());
			}
			catch (Exception e)
			{
				LauncherUtility.openMessageWithDetail(IStatus.ERROR, "", e.getMessage(), e);
			}
		}
		
		
		public boolean isEnabled()
		{
			if (input == null || input.size() == 0)
				return false;
			
			boolean isEnable = true;
			for (Iterator agents = input.iterator(); isEnable && agents.hasNext();)
				isEnable = isEnable && collectObjRef.isEnabledFor(agents.next());
			return isEnable;
		}
			
	}
	
	
	public static class GCControlItem extends ControlItem
	{
		private RunGCAction gcAction;
		public GCControlItem()
		{
			super(GC_ITEM, CommonUITraceConstants.PROFILE_GC_GROUP, ControlMessages.CONTROL_ITEM_GC_ACTION, TraceUIImages.INSTANCE.getImageDescriptor("c", TraceUIImages.IMG_GC));
			setDisabledImageDescriptor(TraceUIImages.INSTANCE.getImageDescriptor("d", TraceUIImages.IMG_GC));
			gcAction = new RunGCAction();
		}
		
		public void run()
		{
			try
			{
				for (Iterator agents = input.iterator(); agents.hasNext();)
					gcAction.doAction(agents.next());
			}
			catch (Exception e)
			{
				LauncherUtility.openMessageWithDetail(IStatus.ERROR, "", e.getMessage(), e);
			}
		}
		
		
		public boolean isEnabled()
		{
			if (input == null || input.size() == 0)
				return false;
			
			boolean isEnable = true;
			for (Iterator agents = input.iterator(); isEnable && agents.hasNext();)
				isEnable = isEnable && gcAction.isEnabledFor(agents.next());
			return isEnable;
		}
			
	}
}
