/**********************************************************************
 * Copyright (c) 2005, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.trace.ui.internal.launcher.application;

import java.util.Map;
import java.util.Vector;

import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.hyades.trace.ui.launcher.ProfilingAttribute;
import org.eclipse.tptp.trace.ui.provisional.launcher.DataCollectionEngineAttribute;
import org.eclipse.tptp.trace.ui.provisional.launcher.IConfiguration;
import org.eclipse.tptp.trace.ui.provisional.launcher.IConfigurationPage;

/**
 * An abstract configuration wizard used for the analysis types
 * of the JVMPI data collector.
 *  
 * @author Ali Mehregani
 */
public abstract class AbstractConfiguration implements IConfiguration
{
	/* The configuration pages */
	private Class[] configurationPages;
	
	/* The configuration page container */
	private IExtendedConfigurationPage[] confPageContainer;
	
	/**
	 * Constructor - The configuration page classes are expected to implement
	 * IExtendedConfigurationPage and have a 0-argument constructor.
	 * 
	 * @param confPages The configuration pages that should be added
	 * to this wizard
	 */	
	public AbstractConfiguration (Class[] confPages)
	{
		this.configurationPages = confPages;
	}
	
	
	/**
	 * Initialize the configuration pages
	 * of the wizard.
	 */
	public void initialize()
	{
		/* Walk through each page */
		Vector confPageVec = new Vector();
		for (int i = 0; i < configurationPages.length; i++)
		{
			Object confPage = null;
			try
			{
				confPage = configurationPages[i].newInstance();
			} 
			catch (Exception e)
			{
				confPage = null;
			} 
			
			if (confPage != null && confPage instanceof IExtendedConfigurationPage)
				confPageVec.add(confPage);
		}
		confPageContainer = new IExtendedConfigurationPage[confPageVec.size()];
		confPageVec.toArray(confPageContainer);
	}

	public IConfigurationPage[] getConfigurationPages()
	{
		return confPageContainer;
	}

	
	/**
	 * Returns the attributes for the configuration wizard.
	 */
	public DataCollectionEngineAttribute[] getAttributes()
	{
		Vector attributeContainer = new Vector();
		for (int i = 0; i < confPageContainer.length; i++)
			attributeContainer.addAll(confPageContainer[i].getAttributes());
		
		DataCollectionEngineAttribute[] profilingAtts = new DataCollectionEngineAttribute[attributeContainer.size()];
		attributeContainer.toArray(profilingAtts);
		
		return profilingAtts;
	}

	/**
	 * Finishes the wizard
	 */
	public boolean finishConfiguration(ILaunchConfigurationWorkingCopy workingCopy)
	{
		/* Give all pages the chance of storing their own attributes */
		boolean status = true;
		for (int i = 0; i < confPageContainer.length; i++)
			status = status && confPageContainer[i].storeAttributes(workingCopy);
		
		return status;
	}
	
	
	/**
	 * Extends the configuration page to add APIs that can be
	 * used for storing and retrieving attributes. <br/>  
	 * Pre-condition:
	 * <ul>
	 * 	<li> The implementor is expected to have a 0-argument default constructor </li>
	 * </ul>
	 *  
	 * @author Ali Mehregani
	 */
	public interface IExtendedConfigurationPage extends IConfigurationPage
	{
		/**
		 * Invoked when the wizard finishes.  This is the chance that the page has in 
		 * storing the user's settings into the working copy of the configuration.
		 *    
		 * @param configurationWC The working copy of the configuration
		 * @return true if successful; false otherwise.
		 */
		public boolean storeAttributes (ILaunchConfigurationWorkingCopy configurationWC);
		
		
		/**
		 * Invoked to retrieve the profiling attributes that corresponds to 
		 * user's settings for this page.  Each entry must be of type ProfilingAttribute 
		 *  
		 * @return The profiling attributes of this page.
		 */
		public Vector getAttributes();
				
	}
	
	
	
	/**
	 * Abstract implementation of IExtendedConfiguration
	 * 
	 * @author Ali Mehregani
	 */
	public static abstract class ExtendedConfigurationPage implements IExtendedConfigurationPage
	{
		
		/**
		 * A convenience method for extracting an option from the option map passed in.
		 * 
		 * @param options The option container
		 * @param optionKey The option key
		 * @param defaultValue The default value
		 * @return The option value if found; otherwise defaultValue
		 */
		protected String getOption(Map options, String optionKey, String defaultValue) 
		{
			ProfilingAttribute option = (ProfilingAttribute)options.get(optionKey);
			if(option != null)
			{
				String optionValue = option.getValue();
				if (optionValue != null)
					return optionValue;
			}
			
			return defaultValue;
		}
	}
	
}
