/**********************************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.trace.ui.internal.launcher.application;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.Vector;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.core.TraceUIImages;
import org.eclipse.hyades.trace.ui.internal.util.AbstractChangeable;
import org.eclipse.hyades.trace.ui.internal.util.FilterList;
import org.eclipse.hyades.trace.ui.internal.util.FilterListContentProvider;
import org.eclipse.hyades.trace.ui.internal.util.FilterListLabelProvider;
import org.eclipse.hyades.trace.ui.internal.util.FilterTableElement;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ColumnLayoutData;
import org.eclipse.jface.viewers.ColumnPixelData;
import org.eclipse.jface.viewers.ComboBoxCellEditor;
import org.eclipse.jface.viewers.IBasicPropertyConstants;
import org.eclipse.jface.viewers.ICellModifier;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Spinner;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceImages;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherMessages;
import org.eclipse.tptp.trace.ui.provisional.launcher.IStatusListener;


/**
 * The configuration for the log agent discoverer data collector.
 * 
 * @author Ali Mehregani
 */
public class AgentDiscovererConfiguration extends AbstractConfiguration
{
	public static final int DEFAULT_FREQUENCY = 500;
	public static final String DEFAULT_FILTERS = ".*,1";

	/**
	 * Add the configuration pages to the parent class.
	 */
	public AgentDiscovererConfiguration()
	{
		super(new Class[]{AgentDiscovererConfPage.class});
	}

	public static AgentDiscovererFilterEntry[] unserializeFilters(String filtersStr)
	{
		StringTokenizer filters = new StringTokenizer(filtersStr, ";"); 
		Vector filtersContainer = new Vector();
		while (filters.hasMoreTokens())
		{
			StringTokenizer filterTokens = new StringTokenizer(filters.nextToken(), ",");
			if (filterTokens.countTokens() != 2)
				continue;
			filtersContainer.add(new AgentDiscovererFilterEntry(filterTokens.nextToken(), filterTokens.nextToken().equals("1")));
		}
		
		AgentDiscovererFilterEntry[] entries = new AgentDiscovererFilterEntry[filtersContainer.size()];
		filtersContainer.toArray(entries);
		
		return entries;
	}
	
	
	/**
	 * Provides a page that will allow the user to set the configurable
	 * options of this collector and its filters.
	 * 
	 * @author Ali Mehregani
	 */
	public static class AgentDiscovererConfPage extends AbstractChangeable implements IExtendedConfigurationPage, SelectionListener
	{
		private ILaunchConfiguration launchConfiguration;
		
		/* Stores the frequency period for querying Agent Controller */
		private Spinner frequencyValue;

		/* Stores the filter set */
		private Table filterTable;

		/* Stores the table viewer */
		private CheckboxTableViewer tableViewer;
		
		/* The filter control buttons */
		private Button removeButton, upButton, downButton;	
		
		private FilterList filterList;
		
		/* The cell editor properties of the filter table */
		private static final String IMG_ITEM_PROP = "0";
		private static final String LOG_AGENT_PROP = "1";
		private static final String VISIBILITY_PROP = "2";

		public void reset(ILaunchConfiguration launchConfiguration)
		{
			this.launchConfiguration = launchConfiguration;
		}
		
		
		/**
		 * Create the controls of the page and restore the settings based 
		 * on the configuration used to reset the page
		 * 
		 * @param parent The parent composite.
		 */
		public void createControl(Composite parent)
		{
			/* Create the option group */
			Composite optionComposite = new Composite(parent, SWT.NONE);
			optionComposite.setLayout(new GridLayout());
			optionComposite.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));
			Group optionGroup = createGroup (optionComposite, LauncherMessages.CONFIGURATION_LOG_AGENT_DISC_TITLE);
			createOptions(optionGroup);
			
			Composite filterGroupComposite = new Composite(parent, SWT.NONE);
			GridLayout gl = new GridLayout();
			gl.marginTop = 30;
			filterGroupComposite.setLayout(gl);
			filterGroupComposite.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));
			
			/* Create the filter group */
			Group filterGroup = createGroup (filterGroupComposite, LauncherMessages.CONFIGURATION_LOG_AGENT_DISC_DESC);
			createFilters(filterGroup);
			
			setButtonStatus(false);
		}

		
		private Group createGroup(Composite parent, String text)
		{			
			Group group = new Group(parent, SWT.NONE);
			group.setLayout(new GridLayout());			
			group.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));			
			group.setText(text);
			
			return group;
		}

		
		private void createOptions(Group optionGroup)
		{
			Composite optionComposite = new Composite(optionGroup, SWT.NONE);
			optionComposite.setLayout(new GridLayout(2, false));
			optionComposite.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));
			
			/* The frequency period option */
			Label frequencyLabel = new Label(optionComposite, SWT.NONE);
			frequencyLabel.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));
			frequencyLabel.setText(LauncherMessages.CONFIGURATION_LOG_AGENT_FREQUENCY);
			
			frequencyValue = new Spinner(optionComposite, SWT.BORDER);
			GridData gd = new GridData(SWT.DEFAULT, SWT.DEFAULT, false, false);
			gd.widthHint = 100;
			frequencyValue.setLayoutData(gd);
			
			frequencyValue.setIncrement(100);			
			frequencyValue.setMaximum(20000000);
			frequencyValue.setMinimum(100);
			frequencyValue.setDigits(0);
			
			/* Set the value of the frequency spinner */
			int freqVal;
			try 
			{
				freqVal = launchConfiguration.getAttribute(IDiscovererLaunchConfConstants.ATTR_OPT_FREQUENCY, -1);
			} 
			catch (CoreException e) 
			{
				freqVal = DEFAULT_FREQUENCY;
			}
			
			if (freqVal < 0)
				freqVal = DEFAULT_FREQUENCY;
			
			frequencyValue.setSelection(freqVal);
			frequencyValue.addModifyListener(new ModifyListener(){

				public void modifyText(ModifyEvent e)
				{
					changed();
				}});
		}
		
		private void createTable(Composite parent) {
			filterTable = new Table(parent, SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER | SWT.MULTI | SWT.FULL_SELECTION);
			filterTable.setLinesVisible(true);
			GridData tableGD = new GridData(SWT.FILL, SWT.FILL, true, true);
			tableGD.widthHint = 360;
			tableGD.heightHint = 70;
			filterTable.setLayoutData(tableGD);
		}
		
		private void createColumns() 
		{
			TableLayout layout = new TableLayout();
			filterTable.setLayout(layout);
			filterTable.setVisible(true);
			filterTable.addSelectionListener(this);
			String headers[] = 
				{
					"",
					UIPlugin.getResourceString(LauncherMessages.CONFIGURATION_LOG_AGENT_NAME),
					UIPlugin.getResourceString(LauncherMessages.CONFIGURATION_LOG_AGENT_VIS)
				};

			ColumnLayoutData layouts[] =
				{
					new ColumnPixelData(20, true),
					new ColumnPixelData(200, true),
					new ColumnPixelData(70, true)
				};

			final TableColumn columns[] = new TableColumn[headers.length];

			for (int i = 0; i < headers.length; i++)
			{
				layout.addColumnData(layouts[i]);
				TableColumn tc = new TableColumn(filterTable, SWT.NONE, i);
				tc.setResizable(layouts[i].resizable);
				tc.setText(headers[i]);				
				columns[i] = tc;
			}
		}
		
		private void setTableCellEditors() 
		{
			/* Add the cell editors */
			CellEditor editors[] = new CellEditor[filterTable.getColumnCount()];

			editors[0] = new TextCellEditor(filterTable);
			editors[1] = new TextCellEditor(filterTable);
			editors[2] = new ComboBoxCellEditor(filterTable, new String[] {LauncherMessages.CONFIGURATION_LOG_AGENT_EXCLUDE, LauncherMessages.CONFIGURATION_LOG_AGENT_INCLUDE});
			

			tableViewer.setCellEditors(editors);
		}
		
		private void setTableCellModifier() 
		{
			tableViewer.setCellModifier(new ICellModifier(){

				public boolean canModify(Object element, String property)
				{
					if (property.equals(IMG_ITEM_PROP))
						return false;
					return true;
				}

				public Object getValue(Object element, String property)
				{
					TableItem[] tableItem = filterTable.getSelection();
					if (tableItem == null || tableItem.length != 1)
						return null;
					
					if (property.equals(LOG_AGENT_PROP))
						return tableItem[0].getText(1);
					else if (property.equals(VISIBILITY_PROP))
					{
						String tableItemTxt = tableItem[0].getText(2);
						if (tableItemTxt.equals(LauncherMessages.CONFIGURATION_LOG_AGENT_EXCLUDE))
							return new Integer(0);
						else if (tableItemTxt.equals(LauncherMessages.CONFIGURATION_LOG_AGENT_INCLUDE))
							return new Integer(1);
					}
					
					return null;
				}

				public void modify(Object element, String property, Object value)
				{
					TableItem[] tableItem = filterTable.getSelection();
					
					if (tableItem == null || tableItem.length != 1)
						return;
					
					AgentDiscovererFilterEntry entry = (AgentDiscovererFilterEntry) filterList.getFilterList().get(filterTable.indexOf(tableItem[0]));
					
					if (property.equals(LOG_AGENT_PROP)) {
						entry.setPattern((String)value);
						tableItem[0].setText(1, (String)value);
					}
					else if (property.equals(VISIBILITY_PROP))
					{
						int visibility = ((Integer)value).intValue();
						entry.setVisibility(visibility == 1);
						if (visibility == 0)
							tableItem[0].setText(2, LauncherMessages.CONFIGURATION_LOG_AGENT_EXCLUDE);
						else if (visibility == 1)
							tableItem[0].setText(2, LauncherMessages.CONFIGURATION_LOG_AGENT_INCLUDE);
					}
					
					changed();
					
				}});
			
		}
		
		private void initializeTable(Composite parent) {
			createTable(parent);
			tableViewer = new CheckboxTableViewer(filterTable);
			createColumns();

			tableViewer.setContentProvider(new ADContentProvider());
			tableViewer.setLabelProvider(new ADLabelProvider());

			filterList = new FilterList();
			tableViewer.setInput(filterList);

			setTableCellEditors();
			setTableCellModifier();
			tableViewer.setColumnProperties(new String[] {IMG_ITEM_PROP, LOG_AGENT_PROP, VISIBILITY_PROP});

			populateTable();			
			
			tableViewer.refresh();
		}
		
		
		private void createFilters(Group filterGroup)
		{
			Composite filterComposite = new Composite(filterGroup, SWT.NONE);
			filterComposite.setLayout(new GridLayout(2, false));
			filterComposite.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));
			
			initializeTable(filterComposite);
			
			Composite buttonComposite = new Composite(filterComposite, SWT.NONE);
			buttonComposite.setLayout(new GridLayout());
			buttonComposite.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));
			
			/* Add button */
			createButton (buttonComposite, LauncherMessages.CONFIGURATION_LOG_AGENT_ADD, 
			new SelectionAdapter(){

				public void widgetSelected(SelectionEvent e)
				{
					createTableItem(".*", true, 0);
					updateButtons();
					changed();

				}});
			
			
			/* Remove button */
			removeButton = createButton (buttonComposite, LauncherMessages.CONFIGURATION_LOG_AGENT_REMOVE, 
			new SelectionAdapter(){
				public void widgetSelected(SelectionEvent e)
				{
					TableItem[] selectedItems = filterTable.getSelection();
					Object [] remainEntries = filterList.getFilterList().toArray();
					int indexOfFirstElement = -1;
					if (selectedItems.length > 0)
						indexOfFirstElement = filterTable.indexOf(selectedItems[0]);
					for (int i = 0; i < selectedItems.length; i++) {
						remainEntries[filterTable.indexOf(selectedItems[i])] = null;
					}

					for (int i = 0; i < selectedItems.length; i++) {
						filterTable.remove(filterTable.indexOf(selectedItems[i]));
					}
					
					filterList.getFilterList().clear();
					
					for(int i = 0; i < remainEntries.length; i++)
						if(remainEntries[i] != null)
							filterList.getFilterList().add((AgentDiscovererFilterEntry)remainEntries[i]);
					
					tableViewer.refresh();
					
					if (filterTable.getItemCount() <= 0)
						setButtonStatus(false);
					
					if (indexOfFirstElement >= 0 && indexOfFirstElement < filterTable.getItemCount())
						filterTable.setSelection(indexOfFirstElement);
					
					changed();
					updateButtons();
				}});
			
			/* Up button */
			upButton = createButton (buttonComposite, LauncherMessages.CONFIGURATION_LOG_AGENT_UP, 
					new SelectionAdapter(){
						public void widgetSelected(SelectionEvent e)
						{
							moveItem(true);					
						}});
			
			/* Down button */
			downButton = createButton (buttonComposite, LauncherMessages.CONFIGURATION_LOG_AGENT_DOWN, 
					new SelectionAdapter(){
						public void widgetSelected(SelectionEvent e)
						{
							moveItem(false);					
						}});
			
			if (filterTable.getItemCount() <= 0)
				setButtonStatus(false);
		}
		


		private void setButtonStatus(boolean status)
		{
			removeButton.setEnabled(status);
			upButton.setEnabled(status);
			downButton.setEnabled(status);
		}
		
		
		private void moveItem(boolean upward)
		{
			TableItem[] selectedItems = filterTable.getSelection();
			for (int i = 0; i < selectedItems.length; i++)
			{			
				int inx = filterTable.indexOf(selectedItems[i]);
				if (upward)
				{
					if (inx <= 0)
						return;
					AgentDiscovererFilterEntry top = (AgentDiscovererFilterEntry) filterList.getFilterList().get(inx-1);
					filterList.getFilterList().remove(inx-1);
					filterList.getFilterList().add(inx,top);
				}
				else
				{
					if (inx >= filterTable.getItemCount() - 1)
						return;
					AgentDiscovererFilterEntry top = (AgentDiscovererFilterEntry) filterList.getFilterList().get(inx);
					filterList.getFilterList().remove(inx);
					filterList.getFilterList().add(inx+1,top);
				}
		
				tableViewer.refresh();
				updateButtons();
			}			
		}
		
		
		private Button createButton(Composite parent, String text, SelectionAdapter listener)
		{
			Button button = new Button (parent, SWT.NONE);
			button.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));
			button.setText(text);
			button.addSelectionListener(listener);
			
			return button;
		}


		/**
		 * Populate the table based on the launch configuration
		 */
		private void populateTable() 
		{
					
			String filtersStr = null;
			
			try 
			{
				filtersStr = launchConfiguration.getAttribute(IDiscovererLaunchConfConstants.ATTR_FILTER, (String)null);
			} catch (CoreException e) 
			{
				/* Doesn't need to be handled */
			}
			
			if (filtersStr == null || filtersStr.length() <= 0)
				filtersStr = DEFAULT_FILTERS;
			
			AgentDiscovererFilterEntry[] filters = unserializeFilters(filtersStr);
			for (int i = 0; i < filters.length; i++)
			{
				filterList.getFilterList().add(filters[i]);
			}
		}


		private AgentDiscovererFilterEntry createTableItem(String logAgentName, boolean visibility, int inx) 
		{
			AgentDiscovererFilterEntry entry = new AgentDiscovererFilterEntry(logAgentName,visibility); 
			filterList.getFilterList().add(entry);
			tableViewer.refresh();
			return entry;
		}

		private AgentDiscovererFilterEntry createTableItem(String logAgentName, boolean visibility) 
		{
			return createTableItem(logAgentName, visibility, filterTable.getItemCount());
		}
		
		public boolean storeAttributes(ILaunchConfigurationWorkingCopy configurationWC)
		{
			/* Store the frequency */
			int frequency = frequencyValue.getSelection();
			if (frequency > 0)
				configurationWC.setAttribute(IDiscovererLaunchConfConstants.ATTR_OPT_FREQUENCY, frequency);
						
			/* Store the filters */
			String filters = serializeFilters();
			if (filters != null)
				configurationWC.setAttribute(IDiscovererLaunchConfConstants.ATTR_FILTER, filters);
			
			return true;
		}

		
		/**
		 * Serializes the content of the filter table into a string that can be written
		 * to the launch configuration.  The format used is as follows:
		 * <p>
		 * 	LOG_AGENT_NAME1,VISIBILITY1;LOG_AGENT_NAME2,VISIBILITY2...
		 * </p>
		 * 
		 * @return A string serialization of the filter table content
		 */
		private String serializeFilters() 
		{
			TableItem[] tableItems = filterTable.getItems();
			String filterStr = "";
			for (int i = 0; i < tableItems.length; i++) 
			{				
				/* Make sure that the values are valid */
				if (errorInFilterItem(tableItems[i]) != null)
					continue;
				
				filterStr += tableItems[i].getText(1) + "," + (tableItems[i].getText(2).equals(LauncherMessages.CONFIGURATION_LOG_AGENT_EXCLUDE) ? "0" : "1") + ";";
			}
						
			return filterStr;
		}
		
		
		public Vector getAttributes()
		{
			/* Don't need to send over any back-end attributes */
			return null;
		}

		public String getPageName()
		{
			return this.getClass().getName();
		}

		public String getTitle()
		{
			return LauncherMessages.CONFIGURATION_LOG_AGENT_DISC_TITLE;
		}

		public ImageDescriptor getWizardBanner()
		{
			return TraceUIImages.INSTANCE.getImageDescriptor(TraceUIImages.IMG_LOG_WIZ_BAN);
		}

		public String getDescription()
		{
			return LauncherMessages.CONFIGURATION_LOG_AGENT_DISC_DESC;
		}

		public void addErrorListener(final IStatusListener statusListener)
		{
			if (statusListener != null)
				super.addListener(new Listener(){

					public void handleEvent(Event event)
					{
						String error = null;
						if (frequencyValue.getSelection() <= 0)
							statusListener.handleErrorEvent(LauncherMessages.ERROR_LOG_DISCOVER_FREQUENCY);
						else if (filterTable.getItemCount() <= 0)
							statusListener.handleErrorEvent(LauncherMessages.ERROR_LOG_DISCOVER_NO_FILTER);						
						else if ((error = errorInFilters()) != null)
							statusListener.handleErrorEvent(error);						
						else
							statusListener.handleOKEvent();
						
					}});
		}
		
		private String errorInFilters()
		{
			TableItem[] tableItems = filterTable.getItems();
			String error = null;
			for (int i = 0; i < tableItems.length; i++)
			{
				if ((error = errorInFilterItem(tableItems[i]))!= null)
					return error;
			}
			
			return null;
		}
		
		private String errorInFilterItem(TableItem filterItem)
		{
			String agentName = filterItem.getText(1);
			String visibility = filterItem.getText(2); 
			
			if (!validString(agentName) || !validString(visibility))
				return LauncherMessages.ERROR_LOG_DISCOVER_FILTER;
			 
			try
			{
				Pattern.compile(agentName);
			}
			catch (Exception e)
			{
				String exceptionMsg = e.getMessage();
				return LauncherMessages.ERROR_LOG_DISCOVER_FILTER + (exceptionMsg != null && exceptionMsg.length() > 0 ? " [" + exceptionMsg + "]" : ""); 
			}
			return null;
		}


		private boolean validString(String text)
		{
			return text != null && text.length() > 0 && text.indexOf(";") == -1 && text.indexOf(",") == -1;
		}
		
		private class FilterList {

			private List fList;

			private FilterList() {
				fList = new ArrayList();
			}

			public List getFilterList() {
				return fList;
			}
		}
		
		public class ADLabelProvider extends LabelProvider implements ITableLabelProvider
		{

			public ADLabelProvider() {
				super();
			}
			public Image getColumnImage(Object element, int columnIndex)
			{
				if(columnIndex == 0) 
					return CommonUITraceImages.INSTANCE.getImage(CommonUITraceImages.IMG_UI_AGENT);
				else
					return null;
			}
			public Image getColumnImage(Viewer viewer, Object element, int columnIndex)
			{
				return getColumnImage(element, columnIndex);
			}
			
			public String getColumnText(Object element, int columnIndex)
			{
				if(columnIndex == 0)
					return "";
				if(columnIndex == 1)
					return ((AgentDiscovererFilterEntry)element).getAgentName();
				if(columnIndex == 2)
					return (((AgentDiscovererFilterEntry)element).getVisibility()) ? LauncherMessages.CONFIGURATION_LOG_AGENT_INCLUDE : LauncherMessages.CONFIGURATION_LOG_AGENT_EXCLUDE;
				return  "";
			}
			
			public String getColumnText(Viewer viewer, Object element, int columnIndex)
			{
				return getColumnText(element,columnIndex);
			}
		}


		public class ADContentProvider implements IStructuredContentProvider {

			private CheckboxTableViewer fViewer;
			protected AgentDiscovererFilterEntry fChangedElement;

			public ADContentProvider() {

				fViewer = tableViewer;
				
			}

			public Object[] getElements(Object inputElement) {
				return ((FilterList)inputElement).getFilterList().toArray();
			}

			public void dispose() {
			}

			
			public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
			}

		}

		public void updateButtons() {
			TableItem[] selectedItems = filterTable.getSelection();
			int minIndex = Integer.MAX_VALUE;
			int maxIndex = -1;
			if(selectedItems.length == 0) {
				setButtonStatus(false);
				return;
			}
			for (int i = 0; i < selectedItems.length; i++)
			{			
				int inx = filterTable.indexOf(selectedItems[i]);
				if(inx < minIndex)
					minIndex = inx;
				if(inx > maxIndex)
					maxIndex = inx;
			}			
			setButtonStatus(true);
			if(minIndex == 0)
				upButton.setEnabled(false);
			if(maxIndex == filterTable.getItemCount() - 1)
				downButton.setEnabled(false);
		}

		public void widgetDefaultSelected(SelectionEvent e) {
			
			
		}


		public void widgetSelected(SelectionEvent e) {
			updateButtons();
		}
	}
	
	
	/**
	 * Represents a filter entry
	 * 
	 * @author Ali Mehregani
	 */
	public static class AgentDiscovererFilterEntry
	{		
		public static final int INCLUDE = 0;
		public static final int EXCLUDE = 1;
		public static final int INDETERMINATE = 2;
		
		private String logAgentName;
		private Pattern logAgentNamePattern;
		private boolean visibility;
		
		public AgentDiscovererFilterEntry(String logAgName, boolean vis)
		{
			try
			{
				logAgentNamePattern = Pattern.compile(logAgName);
				logAgentName = logAgName;
				this.visibility = vis;
			}
			catch (Exception e)
			{
				logAgentNamePattern = null;
			}
		}
		
		public void setPattern(String logAgName) {
			try {
				logAgentNamePattern = Pattern.compile(logAgName);
				logAgentName = logAgName;
			}
			catch (Exception e)
			{
				logAgentNamePattern = null;
			}
		}
		
		public void setVisibility(boolean vis) {
			this.visibility = vis;
		}
		
		public boolean getVisibility()
		{
			return visibility;
		}


		public String getAgentName()
		{
			return logAgentName;
		}


		/**
		 * Returns three possible values: 
		 * <ul>
		 * 	<li> INCLUDE - The filter entry explicitly includes the logging agent name </li>
		 *  <li> EXCLUDE - The filter entry explicitly excludes the logging agent name </li>
		 *  <li> INDETERMINATE - The filter entry doesn't include/exclude the agent name </li>
		 * </ul>
		 * 
		 * @param logAgName The logging agent name
		 * @return A value indicating whether the logging agent is included/excluded/indeterminate by
		 * the filter entry
		 */
		public int match (String logAgName)
		{
			if (logAgentName == null)
				return EXCLUDE;
			try
			{
				Matcher m = logAgentNamePattern.matcher(logAgName);
				boolean matches = m.matches();				
				return !matches ? INDETERMINATE : visibility ? INCLUDE : EXCLUDE;
			}
			catch (Exception e)
			{
				return EXCLUDE;
			}
		}
	}
}
