/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: FilterConfigurationPage.java,v 1.5 2007/05/19 01:06:01 jkubasta Exp $
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.tptp.trace.ui.internal.launcher.application;

import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.core.TraceFilterManager;
import org.eclipse.hyades.trace.ui.internal.core.TraceProfileFiltersUI;
import org.eclipse.hyades.trace.ui.internal.core.TraceUIImages;
import org.eclipse.hyades.trace.ui.internal.launcher.IProfileLaunchConfigurationConstants;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfilingSetsManager;
import org.eclipse.hyades.trace.ui.internal.util.FilterSetElement;
import org.eclipse.hyades.trace.ui.internal.util.FilterTableElement;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.tptp.trace.ui.internal.launcher.application.AbstractConfiguration.IExtendedConfigurationPage;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherConstants;
import org.eclipse.tptp.trace.ui.provisional.launcher.DataCollectionEngineAttribute;
import org.eclipse.tptp.trace.ui.provisional.launcher.ILightConfiguration;
import org.eclipse.tptp.trace.ui.provisional.launcher.IStatusListener;

/**
 * This is the filter configuration page that is commonly used by
 * data collectors requiring a filtering mechanism.
 * 
 * @author Ali Mehregani
 */
public class FilterConfigurationPage implements IExtendedConfigurationPage
{	
	/* Re-use the old instance for the filteration page */		
	private TraceProfileFiltersUI filterUI;
	
	/* The associated launch configuration */ 
	private ILaunchConfiguration launchConfiguration;
	
	/* The attribute name that should be used to store the filter set selection */
	private String attributeName;
	
	
	/**
	 * Default constructor - equivalent to FilterConfigurationPage(IProfileLaunchConfigurationConstants.
	 * ATTR_FILTER_SET)
	 *
	 */
	public FilterConfigurationPage()
	{
		this(IProfileLaunchConfigurationConstants.ATTR_FILTER_SET);
	}
	
	
	/**
	 * The constructor
	 *  
	 * @param attributeName The attribute name that should be used to store the 
	 * filter set selection.
	 */
	public FilterConfigurationPage(String attributeName)
	{
		filterUI = createFilterUI();
		this.attributeName = attributeName;
	}
	
	public void reset(ILaunchConfiguration launchConfiguration)
	{
		this.launchConfiguration = launchConfiguration;				
	}

	
	public Vector getAttributes()
	{
		Vector options = new Vector();			
		try
		{
			String filterSetId = launchConfiguration.getAttribute(attributeName, getDefautltFilterSet());
			options.addElement(new DataCollectionEngineAttribute(ILightConfiguration.AC_OPT_PREFIX_NAME + attributeName, filterSetId));
			
			/* Get the filter set and store its entries as options */
			FilterSetElement filterSet = (FilterSetElement)TraceFilterManager.getInstance().getFilterSet().get(filterSetId);
			
			if (filterSet == null)
				return options;

			List filterElements = filterSet.getChildren();
			if (filterElements == null)
				return options;
			
			for (int i = 0, filterElementCount = filterElements.size(); i < filterElementCount; i++)
			{
				FilterTableElement filterElement = (FilterTableElement)filterElements.get(i);

				String value = filterElement.getVisibility().equals(TraceMessages.INCLUDE) ? "INCLUDE" : "EXCLUDE";
				options.add(new DataCollectionEngineAttribute(filterElement.getMethod().length() <= 0 ? 
							LauncherConstants.SET_FILTER_PREFIX + filterElement.getText() :
							LauncherConstants.SET_METHOD_FILTER_PREFIX + filterElement.getText() + ' ' + filterElement.getMethod(), value));
				
			}
		} 
		catch (CoreException e)
		{
			UIPlugin.getDefault().log(e);
		}
		
		return options;
	}


	protected String getDefautltFilterSet()
	{
		return LauncherConstants.DEFAULT_FILTER_ID;	
	}


	public boolean storeAttributes(ILaunchConfigurationWorkingCopy workingCopy)
	{
		TraceFilterManager.getInstance().storeFilterSetList(filterUI.getFilterSet());
		
		workingCopy.setAttribute(attributeName, filterUI.getActiveFilterSet());
		return true;
	}

			
	
	public void createControl(Composite parent)
	{
		filterUI.resetFields();
		filterUI.createControl(parent);
		if (launchConfiguration == null)
			return;
		
		/* Load the filters from the preference page */
		filterUI.setFilterSet(TraceFilterManager.getInstance().getFilterSetCopy());
								
		String activeFilterSetId = null;
		FilterSetElement activeFilterSet = null;
		if (launchConfiguration != null)
		{
			try
			{
				activeFilterSetId = launchConfiguration.getAttribute(attributeName, LauncherConstants.DEFAULT_FILTER_ID);					
			} 
			catch (CoreException e)
			{
				/* Don't need to do anything */
			}
		}			
		

		activeFilterSet = activeFilterSetId == null ? null : findFilterSet(activeFilterSetId);		
		if(activeFilterSet == null)
		{	
			FilterSetElement nextAvailableFilterSet = ProfilingSetsManager.instance().findNextAvailableFilterSet();
			activeFilterSet = nextAvailableFilterSet == null ? null : findFilterSet(nextAvailableFilterSet.getId());
		}
		
		if (activeFilterSet != null)
			filterUI.setActiveFilterSet(activeFilterSet);

	}

	protected FilterSetElement findFilterSet(String activeFilterSetId)
	{
		ArrayList filterList = filterUI.getFilterSet();
		for (int i = 0, listCount = filterList.size(); i < listCount; i++)
		{
			FilterSetElement fse = (FilterSetElement)filterList.get(i);
			if (fse.getId().equals(activeFilterSetId))
				return fse;
		}
		return null;
	}


	public String getPageName()
	{
		return this.getClass().getName();
	}

	public String getTitle()
	{
		return TraceMessages.EFS;
	}

	public ImageDescriptor getWizardBanner()
	{
		return TraceUIImages.INSTANCE.getImageDescriptor(TraceUIImages.IMG_PROF_SET_WIZ_BAN);
	}

	public String getDescription()
	{
		return TraceMessages.EFS_DE;
	}

	public void addErrorListener(final IStatusListener statusListener)
	{
		/* There are no error conditions that need to be reported */
		statusListener.handleOKEvent();
	}


	/**
	 * @return the filterUI
	 */
	public TraceProfileFiltersUI getFilterUI()
	{
		return filterUI;
	}


	/**
	 * @return the launchConfiguration
	 */
	public ILaunchConfiguration getLaunchConfiguration()
	{
		return launchConfiguration;
	}
	
	protected TraceProfileFiltersUI createFilterUI() {
		return new TraceProfileFiltersUI();
	}
}