/**********************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.trace.ui.internal.launcher.application;

import java.util.Vector;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.hyades.internal.execution.local.common.Options;
import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.core.TraceProfileCollectionsUI;
import org.eclipse.hyades.trace.ui.internal.core.TraceUIImages;
import org.eclipse.hyades.trace.ui.internal.launcher.IProfileLaunchConfigurationConstants;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherMessages;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;
import org.eclipse.tptp.trace.ui.provisional.launcher.DataCollectionEngineAttribute;
import org.eclipse.tptp.trace.ui.provisional.launcher.ILightConfiguration;
import org.eclipse.tptp.trace.ui.provisional.launcher.IStatusListener;

/**
 * This configuration is used to configure the JVMPI data collector.
 * It provides the following wizard pages:
 * <ol>
 *  <li> The limitation options (formerly included as a nested tab of the profile tab) </li>
 *  <li> The filteration page (formerly included as part of the profiling set wizard) </li>
 * </ol> 
 *   	
 * @author Ali Mehregani
 */
public class PIConfiguration extends AbstractConfiguration
{
	
	/**
	 * Add the configuration pages to the parent class.
	 */
	public PIConfiguration()
	{
		this(FilterConfigurationPage.class);
	}
	
	/**
	 * Add the configuration pages to the parent class.
	 */
	protected PIConfiguration(Class FilterConfigurationPageClass)
	{
		super(new Class[] {FilterConfigurationPageClass, LimitsConfigurationPage.class});
	}
	
	/**
	 * Represetns the limits configruation page.  This is the first configuration
	 * page that is displayed when users attempt ot edit the options for the JVMPI
	 * data collector.
	 * 
	 * @author Ali Mehregani
	 */
	private static class LimitsConfigurationPage implements IExtendedConfigurationPage
	{
		/* Re-use the old instance for the limitation configuration */		
		private TraceProfileCollectionsUI limitsUI;
		
		/* The associated launch configuration */ 
		private ILaunchConfiguration launchConfiguration;
		
		
		public LimitsConfigurationPage()
		{
			limitsUI = new TraceProfileCollectionsUI();			
		}
		
		/**
		 * Reset the UI based on the launch configuration
		 * 
		 * @param launchConfiguration The launch configuration
		 */
		public void reset(ILaunchConfiguration launchConfiguration)
		{
			this.launchConfiguration = launchConfiguration;			
		}

		public void createControl(Composite parent)
		{
			limitsUI.createControl(parent);
			if (launchConfiguration == null)
				return;
			
			IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();
					
			try
			{
				limitsUI.setLimitInvocations(launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_LIMIT_INVOCATIONS, store.getBoolean(TraceConstants.LIMIT_TRACE_INVOC_OPTION)));
				limitsUI.setLimitInvocNb(String.valueOf(launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_LIMIT_INVOCATIONS_NUM, store.getInt(TraceConstants.LIMIT_INVOC_NB))));
				limitsUI.setLimitTime(launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_LIMIT_SECONDS, store.getBoolean(TraceConstants.LIMIT_TRACE_TIME_OPTION)));
				limitsUI.setLimitTimeNb(String.valueOf(launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_LIMIT_SECONDS_NUM, store.getInt(TraceConstants.LIMIT_TIME_NB))));
				limitsUI.setAutoMonitoring(launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_AUTO_MONITORING, store.getBoolean(TraceConstants.PROF_AUTO_MONITOR)));
			}
			catch (Exception e)
			{
				LauncherUtility.openWarningWithDetail(LauncherMessages.LAUNCHER_COMMON_WARNING_TITLE, 
						LauncherMessages.ERROR_CONFIG_RESTORE, e);
				limitsUI.setDefaultOptions();
			}					
		}

		public String getPageName()
		{
			return this.getClass().getName();
		}

		public String getTitle()
		{
			return LauncherMessages.CONFIGURATION_LIMIT_TITLE;
		}

		public ImageDescriptor getWizardBanner()
		{
			return TraceUIImages.INSTANCE.getImageDescriptor(TraceUIImages.IMG_PROF_SET_WIZ_BAN);
		}

		public String getDescription()
		{
			return LauncherMessages.CONFIGURATION_LIMIT_DESC;			
		}

		
		public boolean storeAttributes(ILaunchConfigurationWorkingCopy launchConfigurationWC)
		{
			launchConfigurationWC.setAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_LIMIT_INVOCATIONS, limitsUI.getLimitInvocations());
			launchConfigurationWC.setAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_LIMIT_INVOCATIONS_NUM, limitsUI.getLimitInvocNb());
			launchConfigurationWC.setAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_LIMIT_SECONDS, limitsUI.getLimitTime());
			launchConfigurationWC.setAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_LIMIT_SECONDS_NUM, limitsUI.getLimitTimeNb());
			launchConfigurationWC.setAttribute(IProfileLaunchConfigurationConstants.ATTR_AUTO_MONITORING, limitsUI.getAutoMonitoring());
			
			return true;
		}
		
		public Vector getAttributes()
		{
			Vector options = new Vector();
			try
			{				
				IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();
				
				boolean limitMethodInvocation = launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_LIMIT_INVOCATIONS, false);
				boolean limitTime = launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_LIMIT_SECONDS, false);
				int invocationLimitNo = launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_LIMIT_INVOCATIONS_NUM, store.getInt(TraceConstants.LIMIT_INVOC_NB));
				int timeLimitNo = launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_LIMIT_SECONDS_NUM, store.getInt(TraceConstants.LIMIT_TIME_NB));
				
				/* Invocation limit is enabled */			
				if (limitMethodInvocation) 
				{
					options.addElement(new DataCollectionEngineAttribute(ILightConfiguration.AC_OPT_PREFIX_NAME + Options.OPTION_BURST_MODE, limitTime ? Options.OPTION_VALUE_SECONDSANDINVOCATIONS : Options.OPTION_VALUE_INVOCATIONS));
					options.addElement(new DataCollectionEngineAttribute(ILightConfiguration.AC_OPT_PREFIX_NAME + Options.OPTION_BURST_INVOCATIONS, String.valueOf(invocationLimitNo)));
					
					/* Time limit is enabled */ 
					if (limitTime) 
						options.addElement(new DataCollectionEngineAttribute(ILightConfiguration.AC_OPT_PREFIX_NAME + Options.OPTION_BURST_SECONDS, String.valueOf(timeLimitNo)));
				} 
				/* Time limit is enabled */
				else if (limitTime) 
				{
					options.addElement(new DataCollectionEngineAttribute(ILightConfiguration.AC_OPT_PREFIX_NAME +  Options.OPTION_BURST_MODE, Options.OPTION_VALUE_SECONDS));
					options.addElement(new DataCollectionEngineAttribute(ILightConfiguration.AC_OPT_PREFIX_NAME +  Options.OPTION_BURST_SECONDS, String.valueOf(timeLimitNo)));
				}
			} 
			catch (CoreException e)
			{
				UIPlugin.getDefault().log(e);
			}
			
			return options;
		}

		public void addErrorListener(final IStatusListener statusListener)
		{
			limitsUI.addListener(new Listener(){

				public void handleEvent(Event event)
				{
					String error = limitsUI.validate();
					if (error != null)
						statusListener.handleErrorEvent (error);
					else
						statusListener.handleOKEvent();
				}});			
		}
		
	}

}
