/**********************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.trace.ui.internal.launcher.application;

import java.util.Vector;

import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.hyades.internal.execution.local.common.Options;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.core.TraceUIImages;
import org.eclipse.hyades.trace.ui.internal.launcher.IProfileLaunchConfigurationConstants;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherMessages;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;
import org.eclipse.tptp.trace.ui.provisional.launcher.DataCollectionEngineAttribute;
import org.eclipse.tptp.trace.ui.provisional.launcher.ILightConfiguration;
import org.eclipse.tptp.trace.ui.provisional.launcher.IStatusListener;

/**
 * The configuration wizard for the memory analysis type wizard.  
 * 
 * @author Ali Mehregani
 */
public class PIMemoryAnalysisConf extends AbstractConfiguration
{
	/**
	 * Constructor - Intialize with only JVMPIMemoryAnalysisConf as the
	 * configuration page.
	 * 
	 * @param confPages The configuration pages.
	 */
	public PIMemoryAnalysisConf()
	{
		super(new Class[]{MemoryConfigurationPage.class});		
	}
	
		
	/**
	 * The configuration page showing the memory analysis
	 * type options.
	 * 
	 * @author Ali Mehregani
	 */
	public static class MemoryConfigurationPage extends ExtendedConfigurationPage
	{		
		/** The launch configuraiton */
		private ILaunchConfiguration launchConfiguration;

		/** The collect instance level option */
		private Button collectInstanceCheckBox;

		/** Indicates whether instance level information should be collected */
		private boolean collectInstanceLevel;

		/** Indicates whether the controls have been created */
		private boolean controlsCreated;
		
		public void reset(ILaunchConfiguration launchConfiguration)
		{
			this.launchConfiguration = launchConfiguration;
			
			/* Reset the options based on the configuration passed in */
			try
			{
				collectInstanceLevel = launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_SHOW_EXEC_INSTANCES, false);
			}
			catch (Exception e)
			{
				UIPlugin.getDefault().log(e);
			}		
		}		


		/**
		 * Create the controls of the page and restore the settings based 
		 * on the configuration used to reset the page
		 */
		public void createControl(Composite parent)
		{
			controlsCreated = true;
			/* Create the group */
			Group memoryAnalysisOptGroup = new Group(parent, SWT.NONE);
			memoryAnalysisOptGroup.setLayout(new GridLayout());
			memoryAnalysisOptGroup.setLayoutData(new GridData(SWT.FILL, SWT.DEFAULT, true, false));
			memoryAnalysisOptGroup.setText(LauncherMessages.CONFIGURATION_MEM_GROUP_TITLE);
			
			/* The option */
			collectInstanceCheckBox = new Button(memoryAnalysisOptGroup, SWT.CHECK);
			collectInstanceCheckBox.setText(TraceMessages.SH_INSTLI);
			
			/* The warning message */
			Label info = new Label(memoryAnalysisOptGroup, SWT.WRAP);
			GridData data = new GridData();
			data.widthHint = 375;	
			data.verticalIndent = 10;	
			info.setLayoutData(data);
			info.setForeground(Display.getCurrent().getSystemColor(SWT.COLOR_RED));
			info.setText(TraceMessages.INSTL_INFO);
			
			
			/* Attempt to restore the settings from the launch configuration 
			 * that this page was reseted by */
			if (launchConfiguration != null)
			{
				try
				{
					collectInstanceCheckBox.setSelection(launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_SHOW_HEAP_INSTANCES, false));
				}
				catch (Exception e)
				{
					/* Fall back to the default settings */
					LauncherUtility.openWarningWithDetail(LauncherMessages.LAUNCHER_COMMON_WARNING_TITLE, 
							LauncherMessages.ERROR_CONFIG_RESTORE, e);
					collectInstanceCheckBox.setSelection(false);
				}							
			}
		}

		/**
		 * Return the page name
		 */
		public String getPageName()
		{
			/* Return the class name as the page name */
			return this.getClass().getName();			
		}

		/**
		 * Return the title of the wizard page
		 */
		public String getTitle()
		{
			return UIPlugin.getResourceString("memoryType");
		}

		/**
		 * Return the wizard banner of the page 
		 */
		public ImageDescriptor getWizardBanner()
		{
			return TraceUIImages.INSTANCE.getImageDescriptor(TraceUIImages.IMG_PROF_SET_WIZ_BAN);
		}

		/**
		 * Return the description of the page
		 */
		public String getDescription()
		{
			return LauncherMessages.CONFIGURATION_MEM_ANALYSIS_DESC;
		}

		/**
		 * An error condition can't be caused by this page
		 */
		public void addErrorListener(IStatusListener statusListener)
		{
			/* An error condition can't be caused */
			statusListener.handleOKEvent();
		}

		/**
		 * Store the attributes of this page
		 */
		public boolean storeAttributes(ILaunchConfigurationWorkingCopy configurationWC)
		{			
			configurationWC.setAttribute(IProfileLaunchConfigurationConstants.ATTR_SHOW_HEAP_INSTANCES, controlsCreated ? collectInstanceCheckBox.getSelection() : collectInstanceLevel);
			return true;
		}

		
		/**
		 * Return the profiling attributes that are understandable
		 * by the Agent Controller. 
		 */
		public Vector getAttributes()
		{					
			Vector options = new Vector();
			
			try
			{
				String[][] heapData = Options.OPTIONS_ANALYZE_HEAP;		
				for (int idx = 0; idx < heapData.length; idx++) 
				{
					options.add(new DataCollectionEngineAttribute(ILightConfiguration.AC_OPT_PREFIX_NAME + heapData[idx][0], heapData[idx][1]));
				}
				
				String showHeap = String.valueOf(launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_SHOW_HEAP_INSTANCES, false));
				options.add(new DataCollectionEngineAttribute(IProfileLaunchConfigurationConstants.ATTR_SHOW_HEAP_INSTANCES, showHeap));
			}
			catch (Exception e)
			{
				LauncherUtility.openErrorWithDetail(
						LauncherMessages.LAUNCHER_COMMON_ERROR_TITLE,
						LauncherMessages.ERROR_LAUNCH_ATTRIBUTES, e);
			}
			return options;
		}
		
	}
}
