/**********************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.trace.ui.internal.launcher.application;

import java.util.Vector;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.hyades.internal.execution.local.common.Options;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.core.TraceUIImages;
import org.eclipse.hyades.trace.ui.internal.launcher.IProfileLaunchConfigurationConstants;
import org.eclipse.hyades.trace.ui.internal.launcher.defaults.ExecutionTimeUI;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherMessages;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;
import org.eclipse.tptp.trace.ui.provisional.launcher.DataCollectionEngineAttribute;
import org.eclipse.tptp.trace.ui.provisional.launcher.ILightConfiguration;
import org.eclipse.tptp.trace.ui.provisional.launcher.IStatusListener;

public class PITimeAnalysisConf extends AbstractConfiguration
{	
	/**
	 * Constructor - Add the execution time configuration page
	 */
	public PITimeAnalysisConf()
	{
		super(new Class[] {ExecutionTimeConfigurationPage.class});
	}

	
	/**
	 * The execution time configuration page.
	 * 
	 * @author Ali Mehregani
	 */
	public static class ExecutionTimeConfigurationPage extends ExtendedConfigurationPage
	{
		/** The default polling frequency in manual mode */
		public static final int DEFAULT_POLLING_PERIOD = 60;
		
		/** Re-use the UI for the execution time */
		private ExecutionTimeUI executionTimeUI;
		
		/** The launch configuration */
		private ILaunchConfiguration launchConfiguration;

		/** Collect CPU time option */
		private boolean collectCPUTime;

		/** Collect boundary class option */
		private boolean collectBoundaryClass;

		/** The boundary depth */
		private int boundaryDepth;

		/** Collect execution flow */
		private boolean showExecutionFlow;

		/** Collect instance level information */
		private boolean collectInstanceLevel;

		/** Set when controls of this page are created */
		private boolean controlsCreated;
		
		/** A flag indicating the mode of the polling frequency */
		private boolean isPollingAutomated;
		
		/** The value of the polling frequency */
		private int pollingFrequency;
		
		public ExecutionTimeConfigurationPage()
		{
			executionTimeUI = new ExecutionTimeUI(false); 
		}
		
		/**
		 * Reset this page using the launch configuration passed in
		 * 
		 * @param launchConfiguration The launch configuration
		 */
		public void reset(ILaunchConfiguration launchConfiguration)
		{
			this.launchConfiguration = launchConfiguration;
			
			try
			{
				collectCPUTime = launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_SHOW_EXEC_CPU_TIME, false);
				collectBoundaryClass = launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_SHOW_EXEC_BOUNDARY_CLASS, false);
				boundaryDepth = launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_SHOW_EXEC_BOUNDARY_CLASS_DEPTH, 1);
				showExecutionFlow = launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_SHOW_EXECUTION_FLOW, false);
				isPollingAutomated = launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_EXEC_POLLING_FREQ_MODE, true);
				pollingFrequency = launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_EXEC_POLLING_FREQ, DEFAULT_POLLING_PERIOD);
				collectInstanceLevel = launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_SHOW_EXEC_INSTANCES, false);				
			} 
			catch (CoreException e)
			{
				UIPlugin.getDefault().log(e);
			}
		}

		
		/**
		 * Create the controls of the page and restore the settings. 
		 */
		public void createControl(Composite parent)
		{
			controlsCreated = true;
			executionTimeUI.createControl(parent);
						
			try
			{
				executionTimeUI.setCPUtime(collectCPUTime);
				executionTimeUI.setExecutionBoundary(collectBoundaryClass);
				executionTimeUI.setCollectionDepth(String.valueOf(boundaryDepth));				
				executionTimeUI.setExecutionStatistic(!showExecutionFlow);
				executionTimeUI.setExecutionFlow(showExecutionFlow);
				executionTimeUI.setPollingFrequencyMode(isPollingAutomated);
				executionTimeUI.setPollingFrequency(pollingFrequency);
				executionTimeUI.setInstance(collectInstanceLevel);
			}
			catch (Exception e)
			{
				/* Fall back to the default settings */
				LauncherUtility.openWarningWithDetail(LauncherMessages.LAUNCHER_COMMON_WARNING_TITLE, 
						LauncherMessages.ERROR_CONFIG_RESTORE, e);
				executionTimeUI.setDefaultSettings();
			}
		}

		/**
		 * Return the class name as the page name
		 */
		public String getPageName()
		{
			return this.getClass().getName();
		}

		/**
		 * Return the title of the page
		 */
		public String getTitle()
		{
			return UIPlugin.getResourceString("executionGroup");
		}

		/**
		 * Return the banner of this page
		 */
		public ImageDescriptor getWizardBanner()
		{
			return TraceUIImages.INSTANCE.getImageDescriptor(TraceUIImages.IMG_PROF_SET_WIZ_BAN);
		}

		/**
		 * Return the description of this page
		 */
		public String getDescription()
		{
			return LauncherMessages.CONFIGURATION_EXEC_DESC;
		}

		/**
		 * Notify the status listener of any possible error.
		 */
		public void addErrorListener(final IStatusListener statusListener)
		{
			executionTimeUI.addStatusListener(statusListener);
		}
		
		
		/**
		 * Store the attributes of the launch configuration
		 * 
		 * @param configurationWC The working copy of the launch configuration
		 */
		public boolean storeAttributes(ILaunchConfigurationWorkingCopy configurationWC)
		{
			configurationWC.setAttribute(IProfileLaunchConfigurationConstants.ATTR_SHOW_EXEC_CPU_TIME, controlsCreated ? executionTimeUI.getCPUtime() : collectCPUTime);
			configurationWC.setAttribute(IProfileLaunchConfigurationConstants.ATTR_SHOW_EXEC_BOUNDARY_CLASS, controlsCreated ? executionTimeUI.getExecutionBoundary() : collectBoundaryClass);
			configurationWC.setAttribute(IProfileLaunchConfigurationConstants.ATTR_SHOW_EXEC_BOUNDARY_CLASS_DEPTH, controlsCreated ? executionTimeUI.getCollectionDepth() : boundaryDepth);
			configurationWC.setAttribute(IProfileLaunchConfigurationConstants.ATTR_SHOW_EXECUTION_FLOW, controlsCreated ? executionTimeUI.getExecutionFlow() : showExecutionFlow);
			configurationWC.setAttribute(IProfileLaunchConfigurationConstants.ATTR_EXEC_POLLING_FREQ_MODE, controlsCreated ? executionTimeUI.isPollingFreqModeAutomated() : isPollingAutomated);
			configurationWC.setAttribute(IProfileLaunchConfigurationConstants.ATTR_EXEC_POLLING_FREQ, controlsCreated ? executionTimeUI.getPollingFrequency() : pollingFrequency);
			configurationWC.setAttribute(IProfileLaunchConfigurationConstants.ATTR_SHOW_EXEC_INSTANCES, controlsCreated ? executionTimeUI.getInstance() : collectInstanceLevel);			
			
			return true;
		}

		
		/**
		 * Returns the attributes that are understandable by the
		 * Agent Controller.
		 */
		public Vector getAttributes()
		{
			/* Map my UI options to options for the RAC to be recognized as  
			 * RAC options, the name should be prefixed by SETOPTION_ */
			Vector options = new Vector();
			
			/* If the execution boundary option is enabled */
			try
			{
				boolean collectBoundary = launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_SHOW_EXEC_BOUNDARY_CLASS, false);
				boolean collectInstanceLevelInfo = launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_SHOW_EXEC_INSTANCES, false);
				
				if (collectBoundary)
				{
					String[][] optionSet = null;					
					if (collectInstanceLevelInfo) 
						optionSet = Options.OPTIONS_EXECUTION_FLOW_BOUNDARY_INSTANCES;
					else 
						optionSet = Options.OPTIONS_EXECUTION_FLOW_BOUNDARY;
					
	
					for (int idx = 0; idx < optionSet.length; idx++) 				
						options.add(new DataCollectionEngineAttribute(ILightConfiguration.AC_OPT_PREFIX_NAME + optionSet[idx][0], optionSet[idx][1]));
					
					
					/* add boundary depth */
					options.add(new DataCollectionEngineAttribute(ILightConfiguration.AC_OPT_PREFIX_NAME + Options.OPTION_BOUNDARY_DEPTH, String.valueOf(launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_SHOW_EXEC_BOUNDARY_CLASS_DEPTH, 1))));
				}
				
				/* Otherwise */
				else
				{
					String[][] optionSet = null;
					if (collectInstanceLevelInfo) 
						optionSet = Options.OPTIONS_EXECUTION_FLOW_INSTANCES;
					else 
						optionSet = Options.OPTIONS_EXECUTION_FLOW;
					
					
					for (int idx = 0; idx < optionSet.length; idx++) 
						options.add(new DataCollectionEngineAttribute(ILightConfiguration.AC_OPT_PREFIX_NAME + optionSet[idx][0], optionSet[idx][1]));
							
				}
	
				boolean flow = launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_SHOW_EXECUTION_FLOW, false);  //determine full or compressed				
				options.add(new DataCollectionEngineAttribute(IProfileLaunchConfigurationConstants.ATTR_SHOW_EXECUTION_FLOW, String.valueOf(flow)));
				if (flow)
					options.add(new DataCollectionEngineAttribute(ILightConfiguration.AC_OPT_PREFIX_NAME + Options.OPTION_COMPRESS, Options.OPTION_VALUE_NONE));
				else
					options.add(new DataCollectionEngineAttribute(ILightConfiguration.AC_OPT_PREFIX_NAME + Options.OPTION_COMPRESS, Options.OPTION_VALUE_AGGREGATE));
				
				
				/* If we happen to be in compress mode, then store the polling frequency 
				 * mode and possibly the polling frequency value */
				if (!flow)
				{
					boolean automaticPoll = launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_EXEC_POLLING_FREQ_MODE, true);
					
					/* Store the polling frequency value if we're in manual mode */
					if (!automaticPoll)
						options.add(new DataCollectionEngineAttribute(IProfileLaunchConfigurationConstants.ATTR_EXEC_POLLING_FREQ, String.valueOf(launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_EXEC_POLLING_FREQ, DEFAULT_POLLING_PERIOD))));
					options.add(new DataCollectionEngineAttribute(IProfileLaunchConfigurationConstants.ATTR_EXEC_POLLING_FREQ_MODE, String.valueOf(automaticPoll)));
				}
				
				
				String showHeap = String.valueOf(collectInstanceLevelInfo);   //determine collect instance
				options.add(new DataCollectionEngineAttribute(IProfileLaunchConfigurationConstants.ATTR_SHOW_EXEC_INSTANCES, showHeap));
	
				/* If CPU time option is enabled */
				boolean collectCPUTime = launchConfiguration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_SHOW_EXEC_CPU_TIME, false);
				if (collectCPUTime) 
				{
					String[][] optionSet = Options.OPTIONS_COLLECT_CPU_TIME;
					
					for (int idx = 0; idx < optionSet.length; idx++) 
						options.add(new DataCollectionEngineAttribute(ILightConfiguration.AC_OPT_PREFIX_NAME + optionSet[idx][0], optionSet[idx][1]));				
				}
			}
			catch (Exception e)
			{
				LauncherUtility.openErrorWithDetail(
						LauncherMessages.LAUNCHER_COMMON_ERROR_TITLE,
						LauncherMessages.ERROR_LAUNCH_ATTRIBUTES, e);
			}
			
			return options;
		}
	}
}
