/**********************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.trace.ui.internal.launcher.core;

import java.util.Hashtable;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSetType;
import org.eclipse.hyades.trace.ui.launcher.IProfilingType;
import org.eclipse.hyades.trace.ui.launcher.ProfilingAttribute;
import org.eclipse.hyades.trace.ui.launcher.ProfilingSetsManagerCopy;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.tptp.platform.common.ui.trace.internal.util.CommonUITraceUtil;
import org.eclipse.tptp.trace.ui.provisional.launcher.ICollectorFiltration;
import org.eclipse.tptp.trace.ui.provisional.launcher.ILaunchValidator;

/**
 * This class represents an analysis type.  The class implements {@link org.eclipse.hyades.trace.ui.launcher.IProfilingSetType}
 * to be able to support backward compatibility.
 * 
 * @author Ali Mehregani
 */
public class AnalysisType implements IProfilingSetType, Cloneable
{
	/**
	 * The associated id of this analysis type
	 */
	private String id;
		
	/**
	 * The associated name of this analysis type
	 */
	private String name;
	
	/**
	 * The associated description of this analysis type
	 */
	private String description;
	
	/**
	 * Stores the violators of this analysis type.
	 * KEY = analysis type id
	 * VALUE = Boolean.TRUE 
	 */
	private Map coexistanceViolators;
	
	/**
	 * The associated icon of this analysis type
	 */
	private ImageDescriptor icon;
	
	/**
	 * The associated configuration of this analysis type
	 */
	private LightConfigurationLoader configuration;
	
	/**
	 * The associated validator of this analysis type
	 */
	private ILaunchValidator supplement;
		
	/**
	 * The associated filter of this analysis type
	 */
	private ICollectorFiltration filter;
	
	/**
	 * This analysis type is converted to a dummy profiling type to support backward compatibility 
	 */
	private IProfilingType dummyProfilingType;

	
	/**
	 * Restrict the visibility of this constructor
	 */
	private AnalysisType()
	{

	}
	
	public static AnalysisType constructInstance(IConfigurationElement configurationElement)
	{
		String name = configurationElement.getAttribute("name");
		ImageDescriptor icon = CommonUITraceUtil.resolveImageDescriptor(configurationElement, "icon");	
		
		
		String configurationId = configurationElement.getAttribute("configurationId");
		LightConfigurationLoader configuration = null;
		if (configurationId != null)
			configuration = ConfigurationManager.getInstance().getConfiguration(configurationId);
		
		ILaunchValidator validator = null;
		if (configurationElement.getAttribute("validator") != null)
		{
			try
			{
				validator = (ILaunchValidator) configurationElement.createExecutableExtension("validator");
			}
			catch (Exception e)
			{
				validator = null;
			}
		}
		
		String description = configurationElement.getAttribute("description");
		String id = configurationElement.getAttribute("id");
		
		/* Check to ensure that all the required fields are present */
		if (id == null || name == null)
			return null;
			
		IConfigurationElement[] coexistanceViolatorsConf = configurationElement.getChildren("coexistanceViolation");
		Map coexistanceViolators = new Hashtable();		
		if (coexistanceViolatorsConf != null && coexistanceViolatorsConf.length == 1)
		{
			IConfigurationElement[] rule = coexistanceViolatorsConf[0].getChildren("rule");
			for (int i = 0; i < rule.length; i++)
			{		
				String coexistanceViolator = null;
				coexistanceViolator = rule[i].getAttribute("analysisTypeId");
				if (coexistanceViolator != null)
					coexistanceViolators.put(coexistanceViolator, Boolean.TRUE);
			}
		}

		
		if (coexistanceViolators.size() <= 0)
			coexistanceViolators = null;
		
		ICollectorFiltration filter = null;
		try
		{
			if (configurationElement.getAttribute("filter") != null)
				filter = (ICollectorFiltration)configurationElement.createExecutableExtension("filter");
		}
		catch (CoreException e)
		{
			filter = null;
		}
			
		AnalysisType analysisType = new AnalysisType();
		analysisType.setName(name);
		analysisType.setId (id);
		analysisType.setIcon(icon);
		analysisType.setConfiguration(configuration);
		analysisType.setSupplement (validator);
		analysisType.setDescription (description);
		analysisType.setCoexistanceViolators(coexistanceViolators);
		analysisType.setFilter(filter);
		
		return analysisType;
	}

	public Map getCoexistanceViolators()
	{
		return coexistanceViolators;
	}

	public void setCoexistanceViolators(Map coexistanceViolators)
	{
		this.coexistanceViolators = coexistanceViolators;
	}

	public LightConfigurationLoader getConfigurationLoader()
	{
		return configuration;
	}

	public void setConfiguration(LightConfigurationLoader configuration)
	{
		this.configuration = configuration;
	}

	public String getDescription()
	{
		return description;
	}

	public void setDescription(String description)
	{
		this.description = description;
	}

	public ImageDescriptor getIcon()
	{
		return icon;
	}

	public void setIcon(ImageDescriptor icon)
	{
		this.icon = icon;
	}

	public String getId()
	{
		return id;
	}

	public void setId(String id)
	{
		this.id = id;
	}

	public String getName()
	{
		return name;
	}

	public void setName(String name)
	{
		this.name = name;
	}

	public ILaunchValidator getValidator()
	{
		return supplement;
	}

	public void setSupplement(ILaunchValidator validator)
	{
		this.supplement = validator;
	}

	public ImageDescriptor getImage()
	{
		return getIcon();
	}

	public void setImage(ImageDescriptor image)
	{
		setIcon(image);
	}

	
	
	/**
	 * @return the filter
	 */
	public ICollectorFiltration getFilter()
	{
		return filter;
	}

	/**
	 * @param filter the filter to set
	 */
	public void setFilter(ICollectorFiltration filter)
	{
		this.filter = filter;
	}

	public IProfilingType getProfilingType()
	{
		if (dummyProfilingType == null)
			dummyProfilingType = createDummyProfilingType();
		return dummyProfilingType;
	}



	public void setProfilingType(IProfilingType profilingType)
	{
		dummyProfilingType = profilingType;		
	}

	public boolean isUseFilters()
	{		
		return false;
	}

	public void setUseFilters(boolean useFilters)
	{
		/* Not Supported */
	}

	public String getGroup()
	{
		/* Not Supported */
		return null;
	}

	public void setGroup(String group)
	{
		/* Not Supported */
	}
	
	
	/**
	 * Constructs a dummy profiling type based on this analysis type
	 *  
	 * @return A profiling type based on this analysis type
	 */
	private IProfilingType createDummyProfilingType()
	{
		IProfilingType profilingType = new IProfilingType(){

			private Composite parent;
			public Control createControl(Composite parent, ProfilingSetsManagerCopy config)
			{
				this.parent = parent;
				return parent;
			}

			public Control getControl(String profilingSetName)
			{
				return parent;
			}

			public ProfilingAttribute[] getAttributes()
			{
				LightConfigurationLoader configurationLoader = getConfigurationLoader();
				if (configurationLoader != null)
					return configurationLoader.getConfigurationClass().getAttributes();				
				return null;
			}

			public String validateConfiguration(ProfilingSetsManagerCopy managerCopy)
			{
				return null;
			}

			public String launch(ILaunchConfiguration config)
			{	
				return null;
			}

			public String getDescription(ProfilingSetsManagerCopy managerCopy)
			{
				return AnalysisType.this.getDescription();
			}};
			
		return profilingType;
	}

	/**
	 * Provides a shallow clone of this object
	 */
	public Object clone()
	{
		try
		{
			return super.clone();
		} 
		catch (CloneNotSupportedException e)
		{
			return null;
		}
	}
	
	
	/**
	 * Compares this object with the passed object and returns true iff the
	 * passed object is of type AnalysisType OR IProfilingSetType and has the same ID.
	 * 
	 * @param obj The object that will be compared with this object
	 * @return True if this object is equalled to this object
	 */
	public boolean equals(Object obj) 
	{
		if (obj instanceof AnalysisType)
			return id != null && id.equals(((AnalysisType)obj).getId());			
		
		/* For backward compatibility */
		else if (obj instanceof IProfilingSetType)
			return id != null && id.equals(((IProfilingSetType)obj).getId());
		
		return false;
	}
}
