/**********************************************************************
 * Copyright (c) 2006 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.trace.ui.internal.launcher.core;

import java.util.Hashtable;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.tptp.trace.ui.provisional.launcher.IAvailabilityTester;
import org.eclipse.tptp.trace.ui.provisional.launcher.IConfiguration;

/**
 * Represents a contributed configuration wizard
 * 
 * @author Ali Mehregani
 */
public class ConfigurationLoader extends LightConfigurationLoader
{
	/**
	 * The supported attributes
	 */
	public static final String ATTRIBUTE_WIDTH = "width";
	public static final String ATTRIBUTE_HEIGHT = "height";
	
	/**
	 *  The title of the dialog 
	 */
	private String dialogTitle;
	
	/**
	 * The attributes associated with this configuration
	 */
	private Hashtable attributes;
	
	
	/**
	 * Restrict the visibility of the default constructor
	 */
	private ConfigurationLoader(String id, String dialogTitle, IConfiguration mainClass)
	{
		super(id);
		super.editable = true;
		super.configurationClass = mainClass;
		this.dialogTitle = dialogTitle;		
	}
	
	
	/**
	 * Construct an instance of this class based on the configuration element passed in.
	 * 
	 * @param configuration The configuration element
	 * @return An instance of this class based on the details of the configuration element
	 * passed in.  If the element happens to be invalid, then null will be returned.
	 */
	public static LightConfigurationLoader constructInstance(IConfigurationElement configuration)
	{
		/* Return if the element name is not what is expected */
		if (!"configuration".equalsIgnoreCase(configuration.getName()))
			return null;
		
		String[] attributeValues = LauncherUtility.getAttributes(configuration, new String[] {	"id", "dialogTitle", "class",
																								"testAvailabilityClass"});
		/* Ensure that the attribute values for the required attributes are present */
		for (int i = 0; i < 2; i++)
		{
			if (attributeValues[i] == null)
				return null;
		}
		
		IConfiguration configurationClass = null;
		IAvailabilityTester availabilityTester = null;
		
		try
		{
			if (attributeValues[2] != null)
				configurationClass = (IConfiguration)configuration.createExecutableExtension("class");
			
			if (attributeValues[3] != null)
				availabilityTester = (IAvailabilityTester)configuration.createExecutableExtension("testAvailabilityClass");
		} 
		catch (Exception e)
		{
			return null;
		}
		
		ConfigurationLoader configurationLoader = new ConfigurationLoader(attributeValues[0], attributeValues[1], configurationClass);		
		configurationLoader.setAvailabilityTester(availabilityTester);
		
		/* Read in the attributes of the configuration */
		IConfigurationElement[] children = configuration.getChildren();
		for (int i = 0; i < children.length; i++)
		{
			IConfigurationElement child = children[i];
			if ("attribute".equalsIgnoreCase(child.getName()))
			{
				String[] attribute = LauncherUtility.getAttributes(child, new String[] {"name", "value"});
				if (attribute == null || attribute.length <= 0 || attribute[0] == null || attribute[1] == null)
					continue;
					
				configurationLoader.getAttributes().put(attribute[0], attribute[1]);
			}
		}
				
		
		/* Initialize the configuration -- This should only be done once. */
		configurationClass.initialize();
		
		return configurationLoader;
	}
	
	

	public String getDialogTitle()
	{
		return dialogTitle;
	}


	public void setDialogTitle(String dialogTitle)
	{
		this.dialogTitle = dialogTitle;
	}

	
	public Hashtable getAttributes()
	{
		if (attributes == null)
			attributes = new Hashtable();
		return attributes;
	}
	

	public String retrieveAttribute(String name)
	{
		return (String)getAttributes().get(name);
	}
}
