/**********************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.trace.ui.internal.launcher.core;

import java.util.Hashtable;
import java.util.Map;
import java.util.Vector;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.tptp.platform.common.ui.trace.internal.util.CommonUITraceUtil;

public class DataCollector
{
	/* The id of the data collector */
	private String id;
	
	/* The name of the data collector */
	private String name;
	
	/* The analysis types associated with this data collector */
	private AnalysisType[] applicableAnalysisTypes;
	
	/* The default analysis types of this data collector */
	private AnalysisType[] defaultAnalysisTypes;
	
	/* Stores the ID of data collectors that this data collector cannot co-exist with
	 * KEY = data collector id
	 * VALUE = Boolean.TRUE */
	private Map coexistanceViolators;
	
	/* The associated description */
	private String description;
	
	/* The associated icon */
	private ImageDescriptor icon;
	
	/* Indicates if analysis types are required by this data collector */
	private boolean requiresAnalysisType;
	
	
	/**
	 * Restrict the visibility of this constructor
	 */
	private DataCollector()
	{
		
	}
	
	/**
	 * Constructs an instance of type 'DataCollector' based on the configuration element 
	 * passed in.  
	 *   
	 * @param configurationElements The configuration elements of the data collector extension point 
	 */
	public static DataCollector constructInstances(IConfigurationElement collector)
	{
		String id = collector.getAttribute("id");
		String name = collector.getAttribute("name");
		String description = collector.getAttribute("description");
		ImageDescriptor icon = CommonUITraceUtil.resolveImageDescriptor(collector, "icon");
		
		/* Name and ID are required attributes */
		if (name == null || id == null)
			return null;
	

		/* Load the applicable analysis types */
		IConfigurationElement[] childElement = null;
		Vector analysisTypesContainer = new Vector();
		Vector defaultAnalysisTypesContainer = new Vector();
	
		if ((childElement = getElement(collector, "analysis", 1)) != null)
			readApplicableAnalysisTypes(analysisTypesContainer, childElement[0]);
					
		/* We need to also read the associated analysis types that can be 
		 * contributed via the analysisTypesAssociations extension */
		readAssociatedAnalysisTypes(analysisTypesContainer, id);
		
		AnalysisType[] analysisTypes = new AnalysisType[analysisTypesContainer.size()];
		analysisTypesContainer.toArray(analysisTypes);
		
		if ((childElement = getElement(collector, "defaultAnalysis", 1)) != null)
			readApplicableAnalysisTypes(defaultAnalysisTypesContainer, childElement[0]);				
		
		AnalysisType[] defaultAnalysisTypes = new AnalysisType[defaultAnalysisTypesContainer.size()];
		defaultAnalysisTypesContainer.toArray(defaultAnalysisTypes);
		boolean requiresAnalysisType = false;
		
		/* Load the co-existance violations */
		Map coexistanceViolators = null;
		if ((childElement = getElement(collector, "configuration", 1)) != null)
		{
			String reqAnalysisType = childElement[0].getAttribute("requiresAnalysisType");
			if ("true".equals(reqAnalysisType))
				requiresAnalysisType = true;

			if ((childElement = getElement(childElement[0], "coexistanceViolation", 1)) != null)
			{
				IConfigurationElement[] coexistanceRules =  getElement (childElement[0], "rule", -1);
				coexistanceViolators = new Hashtable();
				
				for (int j = 0; j < coexistanceRules.length; j++)
				{
					String ruleID = coexistanceRules[j].getAttribute("dataCollectionMechanismID");
					if (ruleID != null)
						coexistanceViolators.put(ruleID, Boolean.TRUE);
				}
				
				if (coexistanceViolators.size() <= 0)
					coexistanceViolators = null;
			}			
		}
		
		DataCollector dataCollector = new DataCollector();
		dataCollector.setId(id);
		dataCollector.setName(name);
		dataCollector.setDescription(description);
		dataCollector.setApplicableAnalysisTypes(analysisTypes);
		dataCollector.setCoexistanceViolators(coexistanceViolators);
		dataCollector.setRequiresAnalysisType(requiresAnalysisType);
		dataCollector.setDefaultAnalysisTypes(defaultAnalysisTypes);
		dataCollector.setIcon(icon);
		
		return dataCollector;
	}

	private static void readApplicableAnalysisTypes(Vector analysisTypesContainer, IConfigurationElement configurationElement)
	{
		IConfigurationElement[] applicableAnalysisTypes =  getElement (configurationElement, "applicableType", -1);
		AnalysisTypeManager analysisTypeManager = AnalysisTypeManager.getInstance();
		
		for (int j = 0; j < applicableAnalysisTypes.length; j++)
		{
			AnalysisType analysisType = analysisTypeManager.getAnalysisType(applicableAnalysisTypes[j].getAttribute("id"));
			if (analysisType != null)
				analysisTypesContainer.add(analysisType);
		}
	}
	
	
	private static void readAssociatedAnalysisTypes(Vector analysisTypesContainer, String dataCollectionMechanismId)
	{
		AnalysisType[] analysisTypes = AnalysisTypeAssociationManager.getInstance().getAssociatedAnalysisTypes(dataCollectionMechanismId);
		for (int i = 0; i < analysisTypes.length; i++) 
		{
			if (!analysisTypesContainer.contains(analysisTypes[i]))
				analysisTypesContainer.add(analysisTypes[i]);
		}
	}
	
	/**
	 * Returns the child element with the name 'elementName' of the parentElement element.
	 * Returns null if no such element is found or the number of children is not equal to
	 * 'expectedChildrenCount'.  If the number of expected children is unbound, then set
	 * 'expectedChildrenCount' to -1
	 * 
	 * @param parentElement The parent element
	 * @param elementName The element name of the child element
	 * @param expectedChildrenCount The expected number of elementName children
	 * @return The elementName children or null if it cannot be resolved.
	 */
	private static IConfigurationElement[] getElement (IConfigurationElement parentElement, String elementName, int expectedChildrentCount)
	{
		IConfigurationElement[] configurationElements = parentElement.getChildren(elementName);
		if (configurationElements == null || (expectedChildrentCount != -1 && configurationElements.length != expectedChildrentCount))
			return null;
		
		return configurationElements;
	}
	
	
	public AnalysisType[] getApplicableAnalysisTypes()
	{
		return applicableAnalysisTypes;
	}


	public void setApplicableAnalysisTypes(AnalysisType[] applicableAnalysisTypes)
	{
		this.applicableAnalysisTypes = applicableAnalysisTypes;
	}


	public Map getCoexistanceViolators()
	{
		return coexistanceViolators;
	}


	public void setCoexistanceViolators(Map coexistanceViolators)
	{
		this.coexistanceViolators = coexistanceViolators;
	}


	public String getId()
	{
		return id;
	}


	public void setId(String id)
	{
		this.id = id;
	}


	public String getName()
	{
		return name;
	}


	public void setName(String name)
	{
		this.name = name;
	}

	public String getDescription()
	{
		return description;		
	}
	
	public void setDescription(String description)
	{
		this.description = description;		
	}

	public AnalysisType[] getDefaultAnalysisTypes()
	{
		return defaultAnalysisTypes;
	}

	public void setDefaultAnalysisTypes(AnalysisType[] defaultAnalysisTypes)
	{
		this.defaultAnalysisTypes = defaultAnalysisTypes;
	}

	public ImageDescriptor getIcon()
	{
		return icon;
	}

	public void setIcon(ImageDescriptor icon)
	{
		this.icon = icon;
	}

	public boolean isAnalysisTypeRequired() {
		return requiresAnalysisType;
	}

	public void setRequiresAnalysisType(boolean requiresAnalysisType)
	{
		this.requiresAnalysisType = requiresAnalysisType;
	}
}
