/**********************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.trace.ui.internal.launcher.core;

import java.util.Hashtable;
import java.util.Iterator;
import java.util.Vector;

import org.eclipse.core.runtime.IConfigurationElement;


/**
 * This class represents each association tag for the launchConfigMechanismAssociator extension point.
 * 
 * @author Ali Mehregani
 */
public class DataCollectorAssociation
{
	/* Describes each mechanism tag defined for the association. 
	 * KEY = Data collector ID
	 * VALUE = Data collector association data (see DataCollectorAssociationData)
	 */
	private Hashtable associatedDataCollectorMechanism;
	
	/* The default data collectors */
	private String[] defaultDataCollectors;
	
	/* Cached values of the data collectors associated with the launch configuration that this object belongs to */
	private DataCollector[] dataCollectors;
	
	/* Stores the launch configuration id */
	private String launchConfigurationId;
	
	/* For backward compatibility, a default association is created that is only affiliated with the PI data collector */
	private static DataCollectorAssociation defaultAssociation;
	
	
	/**
	 * Restrict the visibility of the constructor
	 */
	protected DataCollectorAssociation()
	{
		associatedDataCollectorMechanism = new Hashtable();
	}
	
	/**
	 * Construct an instance of this class based on the configuration elements passed in.  
	 * 
	 * @param associations The association configuration elements
	 * @param associationTable The association table
	 */
	public static DataCollectorAssociation constructInstances(IConfigurationElement association, Hashtable associationTable)
	{	
		IConfigurationElement[] configurationElemChildren;
		String launchConfigID = association.getAttribute("launchConfigID");
		
		/* The launch configuration id is required. */
		if (launchConfigID == null)
			return null;
		
		/* Load the data collectors */
		DataCollectorAssociation datacollectorAssociation = (DataCollectorAssociation)associationTable.get(launchConfigID);
		
		if (datacollectorAssociation == null)
		{
			datacollectorAssociation = new DataCollectorAssociation();
			datacollectorAssociation.setLaunchConfigurationId(launchConfigID);			
		}
		
		Hashtable datacollectorAssociationData = datacollectorAssociation.getAssociatedDataCollectorMechanism();
		configurationElemChildren = association.getChildren("mechanism");
		for (int k = 0; k < configurationElemChildren.length; k++)
		{
			DataCollectorAssociationData datacollectorAssociationDataEntry = DataCollectorAssociationData.constructInstance(configurationElemChildren[k]);
			if (datacollectorAssociationDataEntry != null)
				datacollectorAssociationData.put(datacollectorAssociationDataEntry.getDataCollectorId(), datacollectorAssociationDataEntry);
		}
		
		datacollectorAssociation.setAssociatedDataCollectorMechanism(datacollectorAssociationData);	
		
		/* Load the default data collectors */
		configurationElemChildren = association.getChildren("defaultDataCollectors");
		String[] defaultDataCollectors = null;
		if (configurationElemChildren != null && configurationElemChildren.length == 1)
		{
			configurationElemChildren = configurationElemChildren[0].getChildren("defaultCollector");
			Vector defaultCollectors = new Vector();
			for (int k = 0; k < configurationElemChildren.length; k++)
				defaultCollectors.add(configurationElemChildren[k].getAttribute("datacollectorID"));
			
			defaultDataCollectors = datacollectorAssociation.getDefaultDataCollectors();
			
			/* If the default data collectors doesn't exist, then create it */
			if (defaultDataCollectors == null)
			{
				defaultDataCollectors = new String[defaultCollectors.size()];
				defaultCollectors.toArray(defaultDataCollectors);
				datacollectorAssociation.setDefaultDataCollectors(defaultDataCollectors);
			}
			/* Otherwise add to it */
			else
			{
				int totalLength = defaultCollectors.size() + defaultDataCollectors.length;
				String[] updatedListOfDefaults = new String[totalLength];
				System.arraycopy(defaultDataCollectors, 0, updatedListOfDefaults, 0, defaultDataCollectors.length);
				for (int i = defaultDataCollectors.length; i < totalLength; i++)
					updatedListOfDefaults[i] = (String)defaultCollectors.get(i - defaultDataCollectors.length);
					
				datacollectorAssociation.setDefaultDataCollectors(updatedListOfDefaults);
			}
		}
		
		return datacollectorAssociation;
	}

	public Hashtable getAssociatedDataCollectorMechanism()
	{
		return associatedDataCollectorMechanism;
	}

	public void setAssociatedDataCollectorMechanism(
			Hashtable associatedDataCollectorMechanism)
	{
		this.associatedDataCollectorMechanism = associatedDataCollectorMechanism;
	}

	public String[] getDefaultDataCollectors()
	{
		return defaultDataCollectors;
	}

	public void setDefaultDataCollectors(String[] defaultDataCollectors)
	{
		this.defaultDataCollectors = defaultDataCollectors;
	}
	
	public DataCollector[] getDataCollectors()
	{
		if (dataCollectors != null)
			return dataCollectors;
		
		/* Resolve the data collectors associated with this object */
		Vector datacollectorsContainer = new Vector();
		for (Iterator iterator = associatedDataCollectorMechanism.values().iterator(); iterator.hasNext();)
		{
			DataCollectorAssociationData dataCollectorAssociationData = (DataCollectorAssociationData)iterator.next();
			datacollectorsContainer.add(DataCollectorManager.getInstance().getDataCollector(dataCollectorAssociationData.getDataCollectorId()));			
		}
		
		dataCollectors = new DataCollector[datacollectorsContainer.size()];
		datacollectorsContainer.toArray(dataCollectors);
		return dataCollectors;
	}
	
	
	public DataCollectorAssociationData getDataCollectorAssociationData(String dataCollectorId)
	{
		return (DataCollectorAssociationData)associatedDataCollectorMechanism.get(dataCollectorId);
	}

	public String getLaunchConfigurationId()
	{
		return launchConfigurationId;
	}

	public void setLaunchConfigurationId(String launchConfigurationId)
	{
		this.launchConfigurationId = launchConfigurationId;
	}
	
	
	public static DataCollectorAssociation getDefault()
	{
		if (defaultAssociation == null)
		{
			defaultAssociation = new DataCollectorAssociation();
			defaultAssociation.associatedDataCollectorMechanism.put(LauncherConstants.JVMPI_DATA_COLLECTOR_ID, DataCollectorManager.getInstance().getDataCollectorAssociator(LauncherConstants.LOCAL_JAVA_APP_CONF_ID).getDataCollectorAssociationData(LauncherConstants.JVMPI_DATA_COLLECTOR_ID));
		}
		
		return defaultAssociation;
	}
}
