/**********************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.trace.ui.internal.launcher.core;

import java.util.Vector;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.tptp.trace.ui.provisional.launcher.ICollectorFiltration;
import org.eclipse.tptp.trace.ui.provisional.launcher.IDataCollectorBaseLauncher;
import org.eclipse.tptp.trace.ui.provisional.launcher.ILaunchValidator;

/**
 * This class represents each mechanism entity defined under an association tag for the 
 * launchConfigMechanismAssociator extension point.
 *  
 * @author Ali Mehregani
 */
public class DataCollectorAssociationData 
{
	/**
	 * The launch delegate's configuration element 
	 */
	private IConfigurationElement launchDelegateConfigurationElem;
	
	/**
	 * The configuration association with this association 
	 */
	private LightConfigurationLoader configurationLoader;
	
	/** 
	 * The validator associated with this association 
	 */
	private ILaunchValidator validator;

	/**
	 * The data collection mechanism ID 
	 */
	private String datacollectionMechanismID; 
	
	/** 
	 * Contains the id of the agent declarations that are applicable to this launch 
	 */
	private String[] applicableAgents;
	
	/**
	 * The associated filter 
	 */
	private ICollectorFiltration filter;
	
	
	/**
	 * Restrict the visibility of this constructor
	 */
	private DataCollectorAssociationData()
	{
		
	}
	
	
	/**
	 * A convenient constructor that can be used for setting the fields of this class.  As 
	 * an alternative, clients can use the 0-argument constructor in conjunction with the
	 * setter methods.
	 * 
	 * @param launchDelegateConfigurationElem The configuration element for the launch delegate
	 * @param launchConfiguration The launch configuration
	 * @param validator The validator
	 */
	private DataCollectorAssociationData(IConfigurationElement launchDelegateConfigurationElem, 
										LightConfigurationLoader configuration, ILaunchValidator validator, 
										String dataCollectorID, String[] applicableAgents)
	{
		this.launchDelegateConfigurationElem = launchDelegateConfigurationElem;
		this.configurationLoader = configuration;
		this.validator = validator;
		this.datacollectionMechanismID = dataCollectorID;
		this.applicableAgents = applicableAgents;
	}

	/**
	 * Construct an instance of this class based on the 'mechanism' element that is defined as
	 * a child element of 'association' for the extension point 'launchConfigMechanismAssociator'
	 * 
	 * @param mechanism The mechanism element
	 * @return An instance of this class based on a 'launchConfigMechanismAssociator' extension 
	 */
	public static DataCollectorAssociationData constructInstance(IConfigurationElement mechanism)
	{		
		IConfigurationElement[] configurationElemChildren = mechanism.getChildren("configuration");
		
		if (configurationElemChildren == null || configurationElemChildren.length != 1)
			return null;
		
		try
		{		
			IConfigurationElement configurationElement = configurationElemChildren[0];
					
			String datacollectionMechanismID = mechanism.getAttribute("mechanismID");
			
			/* The data collection mechanism id is a required field */
			if (datacollectionMechanismID == null)
				return null;
			
			String configurationId = configurationElemChildren[0].getAttribute("configurationId");
			LightConfigurationLoader launchConfiguration = null;
			if (configurationId != null)
				launchConfiguration = ConfigurationManager.getInstance().getConfiguration(configurationId);
			
			ILaunchValidator collectorValidator = null;
			if (configurationElemChildren[0].getAttribute("validator") != null)
				collectorValidator = (ILaunchValidator) configurationElemChildren[0].createExecutableExtension("validator");
			
			/* Launch delegate is a required field */
			if (configurationElement.getAttribute("launchDelegate") == null)
				return null;
			
			/* Determine the associated agents of this launch configuration */
			String[] applicableAgentsStr = null;
			
			/* First determine if there is only one associated agent */
			String associatedAgent = configurationElemChildren[0].getAttribute("associatedAgent");
			if (associatedAgent != null && associatedAgent.length() > 0)
				applicableAgentsStr = new String[] {associatedAgent};
			/* Otherwise determine if there are multiple agents associated with this launch config */
			else
			{
				configurationElemChildren = configurationElemChildren[0].getChildren("agentAssociation");
				if (configurationElemChildren != null && configurationElemChildren.length == 1)
				{
					configurationElemChildren = configurationElemChildren[0].getChildren("agent");
					Vector applicableAgents = new Vector();			
					for (int i = 0; i < configurationElemChildren.length; i++)
						applicableAgents.add(configurationElemChildren[i].getAttribute("id"));
					
					applicableAgentsStr = new String[applicableAgents.size()];
					applicableAgents.toArray(applicableAgentsStr);
				}
			}
			
			ICollectorFiltration filter = configurationElement.getAttribute("filter") == null ? null : (ICollectorFiltration)configurationElement.createExecutableExtension("filter");				
			
			DataCollectorAssociationData associationData = new DataCollectorAssociationData();
			associationData.setLaunchDelegateConfigurationElem(configurationElement);
			associationData.setConfigurationLoader(launchConfiguration);
			associationData.setValidator(collectorValidator);
			associationData.setApplicableAgents(applicableAgentsStr);
			associationData.setDatacollectionMechanismID(datacollectionMechanismID);
			associationData.setFilter(filter);
			
			return associationData;
		}
		catch (CoreException e)
		{
			UIPlugin.getDefault().log(e);
		}
		return null;
	}

	public String getDataCollectorId()
	{
		return datacollectionMechanismID;
	}
	
	
	public IDataCollectorBaseLauncher createLaunchDelegate() throws CoreException 
	{		
		return (IDataCollectorBaseLauncher)launchDelegateConfigurationElem.createExecutableExtension("launchDelegate");
	}
	

	public LightConfigurationLoader getConfigurationLoader()
	{
		return configurationLoader;
	}

	public void setConfigurationLoader(LightConfigurationLoader configuration)
	{
		this.configurationLoader = configuration;
	}

	public ILaunchValidator getValidator()
	{
		return validator;
	}

	public void setValidator(ILaunchValidator validator)
	{
		this.validator = validator;
	}


	public String[] getApplicableAgents()
	{
		return applicableAgents;
	}


	public void setApplicableAgents(String[] applicableAgents)
	{
		this.applicableAgents = applicableAgents;
	}


	/**
	 * @return the filter
	 */
	public ICollectorFiltration getFilter()
	{
		return filter;
	}


	/**
	 * @param filter the filter to set
	 */
	public void setFilter(ICollectorFiltration filter)
	{
		this.filter = filter;
	}


	/**
	 * @return the datacollectionMechanismID
	 */
	public String getDatacollectionMechanismID()
	{
		return datacollectionMechanismID;
	}


	/**
	 * @param datacollectionMechanismID the datacollectionMechanismID to set
	 */
	public void setDatacollectionMechanismID(String datacollectionMechanismID)
	{
		this.datacollectionMechanismID = datacollectionMechanismID;
	}


	/**
	 * @param launchDelegateConfigurationElem the launchDelegateConfigurationElem to set
	 */
	public void setLaunchDelegateConfigurationElem(IConfigurationElement launchDelegateConfigurationElem)
	{
		this.launchDelegateConfigurationElem = launchDelegateConfigurationElem;
	}	
	
	
}
