/**********************************************************************
 * Copyright (c) 2006, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DataCollectorManager.java,v 1.2 2010/12/01 17:56:07 mreid Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.trace.ui.internal.launcher.core;

import java.util.Collection;
import java.util.Hashtable;
import java.util.WeakHashMap;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.Platform;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.tptp.trace.ui.provisional.launcher.IDataCollectorBaseLauncher;

/**
 * Loads the registered data collectors and provides accessor methods for
 * clients to retrieve them.
 * 
 * @author Ali Mehregani
 */
public class DataCollectorManager
{
	/* An instance of this class */
	private static DataCollectorManager instance;
	
	/* Contains the registered data collectors.  
	 * KEY = data collector ID
	 * VALUE = data collector */
	private Hashtable datacollectors;

	/* Describes the associations between launch configurations and the data collectors:
	 * 
	 * KEY = Launch configuration ID
	 * VALUE = Data Collector Association
	 */
	private Hashtable datacollectorAssociations;
	
	private WeakHashMap<IDataCollectorFetchListener,Object> collectorListeners;
	private int runningJobs;

	private DataCollectorManager()	
	{
		datacollectors = new Hashtable();
		datacollectorAssociations = new Hashtable();
		
		// Use a weak hash map so that registered listeners don't _have_ to remove themselves
		collectorListeners = new WeakHashMap<IDataCollectorFetchListener,Object>();

		initialize();
	}
	
	/**
	 * Scan the registered data collectors and add them to
	 * data structures.
	 */
	private void initialize()
	{
		/* Find the data collector registerations */
		IConfigurationElement[] elements = Platform.getExtensionRegistry().getConfigurationElementsFor(
				"org.eclipse.hyades.trace.ui.dataCollectionMechanism");
		
		/* Iterate through all collectors and create a data collector instance */
		final String COLLECTOR_ELEMENT_NAME = "collector";
		final String ASSOCIATOR_ELEMENT_NAME = "association";
		
		for (int i = 0; i < elements.length; i++)
		{
			/* Skip the element if it is not what is expected */
			if (!COLLECTOR_ELEMENT_NAME.equalsIgnoreCase(elements[i].getName()))
				continue;
			
			DataCollector datacollector = DataCollector.constructInstances(elements[i]);
			if (datacollector != null)
				this.datacollectors.put(datacollector.getId(), datacollector);
		}		
		
		/* Find the data collector association registerations */
		elements = Platform.getExtensionRegistry().getConfigurationElementsFor(
				"org.eclipse.hyades.trace.ui.launchConfigMechanismAssociator");
		
		/* Iterate through all collectors and create a data collector instance */
		for (int i = 0; i < elements.length; i++)
		{
			/* Skip the element if it is not what is expected */
			if (!ASSOCIATOR_ELEMENT_NAME.equalsIgnoreCase(elements[i].getName()))
				continue;
			
			DataCollectorAssociation datacollectorAssociation = DataCollectorAssociation.constructInstances(elements[i], datacollectorAssociations);
			if (datacollectorAssociation != null)
				this.datacollectorAssociations.put(datacollectorAssociation.getLaunchConfigurationId(), datacollectorAssociation);
		}
	}
	
	public static DataCollectorManager getInstance()
	{
		if (instance == null)
			instance = new DataCollectorManager();
		
		return instance;
	}

	public DataCollector getDataCollector(String datacollectorID)
	{
		return (DataCollector)datacollectors.get(datacollectorID);
	}

	/**
	 * Lookup and return a data collector associator entry for the launchConfigurationID, if it
	 * exists.  If an association is not found, then a default one is returned (to support
	 * backward compatibility)
	 * 
	 * @param launchConfigurationID The launch configuration ID.
	 * @return A DataCollectorAssociation with id 'launchConfigurationID'
	 */
	public DataCollectorAssociation getDataCollectorAssociator (String launchConfigurationID)
	{
		DataCollectorAssociation association = (DataCollectorAssociation)datacollectorAssociations.get(launchConfigurationID);
		if (association == null)
			return DataCollectorAssociation.getDefault();
		return association;
	}
	
	
	/**
	 * Returns the launch delegate of the data collector with the id 'dataCollectorID' for
	 * the launch configuration 'configurationType'.
	 * If the launch delegate cannot be resolved, then null will be returned.
	 * 
	 * @param currentDataCollectorID The ID of the data collector
	 * @param configurationType The launch configuration type
	 * @return The launch delegate of the data collector with id 'dataCollectorID'
	 * @throws CoreException If this method fails
	 */
	public IDataCollectorBaseLauncher getDataCollectorLaunchDelegate(String dataCollectorID, ILaunchConfiguration configurationType) throws CoreException 
	{
		DataCollector dataCollector = (DataCollector)datacollectors.get(dataCollectorID);
		if (dataCollector == null)
			return null;
		
		DataCollectorAssociation dataCollectorAssociation = (DataCollectorAssociation)datacollectorAssociations.get(configurationType.getType().getIdentifier());
		if (dataCollectorAssociation == null)
			return null;
		
		DataCollectorAssociationData datacollectorAssociationData = dataCollectorAssociation.getDataCollectorAssociationData(dataCollectorID);
		if (datacollectorAssociationData == null)
			return null;
		
		return datacollectorAssociationData.createLaunchDelegate();
	}

	
	public DataCollector[] getDataCollectors()
	{
		Collection values = datacollectors.values();
		DataCollector[] datacollectors = new DataCollector[values.size()]; 
		values.toArray(datacollectors);
		
		return datacollectors;
	}

	/**
	 * Bug 331355
	 * 
	 * Blocks the calling thread until any currently running data collector fetch jobs complete or `timeout`
	 * milliseconds of time passes.
	 * 
	 * @param timeout
	 * @return false if the wait timed out and jobs are still running; true otherwise (i.e. no fetch jobs are running)
	 */
	public synchronized boolean waitForDataCollectors( long timeout ) {
			
		final long baseTime = System.currentTimeMillis();
		
		while( isFetching() ) {
			
			long remainingTime = baseTime + timeout - System.currentTimeMillis();
			if( remainingTime < 0 )
				return false;
			
			try {
				
				wait(remainingTime);
				
			} catch (InterruptedException e) {
				// No-op
			}
			
		}
			
		return true;		
	}
	
	/**
	 * Record a new data collector fetch/filter operation. If none are currently in-flight
	 * then notify registered listeners about the start.
	 */
	synchronized void beginFetch() {
		if( 0 == runningJobs )
			// Notify registered listeners that a new fetch job has started.
			for( IDataCollectorFetchListener l : collectorListeners.keySet() ) {
				l.fetchStarted();
			}
		
		runningJobs++;
	}

	/**
	 * Record the end of a data collector fetch/filter operation. If this is the last
	 * running fetch operation, then notify registered listeners about the results.
	 * 
	 * @param result - Array of fetched available data collectors
	 */
	synchronized void endFetch(DataCollector[] result) {
		runningJobs--;
		if( 0 == runningJobs ) {
			
			// Notify listeners
			for( IDataCollectorFetchListener l : collectorListeners.keySet() )
				l.fetchComplete(result);
			
			// Also notify anyone who is waiting in .waitDataCollectors
			notifyAll();
		}
	}
	
	/**
	 * Bug 331355
	 * 
	 * Register the given object to be notified when a fetch job is started. Note this only fires when transitioning
	 * from 'no running jobs' to '1 or more running jobs'. I.e. if there is already 1 running job and another
	 * one starts, the monitors are not notified.
	 *  
	 * @param listener Monitor to notify
	 */
	public synchronized void addDataCollectorFetchListener( IDataCollectorFetchListener listener ) {
		
		collectorListeners.put( listener, null );
		
	}
	
	/**
	 * Bug 331355
	 * 
	 * Remove the given listener from the list of registered data collector fetch listeners. The
	 * passed in listener will no longer receive fetchStarted/fetchCompleted events.
	 */
	public synchronized void removeDataCollectorFetchListener( IDataCollectorFetchListener listener ) {
		
		collectorListeners.remove(listener);
		
	}

	/**
	 * Status query to check if there are currently any data collector fetch operations in flight.
	 */
	public synchronized boolean isFetching() {
		
		return runningJobs > 0;
		
	}

}
