/**********************************************************************
 * Copyright (c) 2006, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.trace.ui.internal.launcher.core;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.IJobChangeEvent;
import org.eclipse.core.runtime.jobs.ISchedulingRule;
import org.eclipse.core.runtime.jobs.JobChangeAdapter;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.hyades.trace.ui.internal.core.TraceProfileUI;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfilingSetsManager;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSetType;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.AbstractTreeViewer;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.tptp.trace.ui.provisional.launcher.ICollectorFiltration;
import org.eclipse.ui.progress.DeferredTreeContentManager;
import org.eclipse.ui.progress.IDeferredWorkbenchAdapter;
import org.eclipse.ui.progress.IElementCollector;
import org.eclipse.ui.progress.WorkbenchJob;


/**
 * Provides the content of the tree that is used to display the registered data collectors
 * and their applicable analysis types.
 *  
 * @author Ali Mehregani
 */
public class DataCollectorTreeContentProvider implements ITreeContentProvider
{		
	/**
	 * The profile UI
	 */
	private TraceProfileUI profileUI;
	
	/**
	 * The deferred content manager
	 */
	private DeferredTreeContentManager deferredContentManager;

	/**
	 * The input
	 */
	private DataCollectorTreeInput input;
	
	public DataCollectorTreeContentProvider(TraceProfileUI profileUI)
	{
		this.profileUI = profileUI;
	}
	
	public Object[] getChildren(Object parentElement)
	{
		if (parentElement instanceof DataCollector)
		{
			int numberOfChildren = 0;
			DataCollector dataCollector = (DataCollector)parentElement;
			AnalysisType[] applicableAnalysisTypes = dataCollector.getApplicableAnalysisTypes();
			
			if (applicableAnalysisTypes == null)
				return null;
			
			numberOfChildren = applicableAnalysisTypes.length; 
			Object[] profilingTypes = null;
			
			/* To support backward compatibility, we'll need to associate orphan profiling types to
			 * the JVMPI data collector. */
			if (dataCollector.getId().equals(LauncherConstants.JVMPI_DATA_COLLECTOR_ID))
			{
				profilingTypes = ProfilingSetsManager.instance().getProfilingTypes().values().toArray();
				
				/* We'll need to filter the analysis types */
				for (int i = 0; i < profilingTypes.length; i++)
				{
					if (profilingTypes[i] instanceof AnalysisType || !(profilingTypes[i] instanceof IProfilingSetType))
						profilingTypes[i] = null;
					else
						numberOfChildren++;
				}				
			}
			
			/* Begin copying all the children */
			List parentChildNodes = new ArrayList(numberOfChildren);
			int storageInx = 0;
			
			/* Copy the analysis types */
			for (; storageInx < applicableAnalysisTypes.length; storageInx++)
			{
				/* Does this analysis type have an associted filter? */
				ICollectorFiltration filter = applicableAnalysisTypes[storageInx].getFilter();
				if (filter != null && !filter.include(applicableAnalysisTypes[storageInx].getId(), input == null ? null : input.getConfiguration(), null))
					continue;
									
				parentChildNodes.add(new ParentChildNode (parentElement, applicableAnalysisTypes[storageInx]));
			}

			/* Copy the profiling types */		
			if (profilingTypes != null)
			{
				for (int i = 0; i < profilingTypes.length; i++)
				{
					if (profilingTypes[i] != null)
						parentChildNodes.add(new ParentChildNode (parentElement, profilingTypes[i]));
				}
			}
			return (ParentChildNode[])parentChildNodes.toArray(new ParentChildNode[parentChildNodes.size()]);
		}
				
		return null;
	}

	public Object getParent(Object element)
	{		
		if (element instanceof ParentChildNode)
			return ((ParentChildNode)element).parent;
		
		return null;
	}

	public boolean hasChildren(Object element)
	{
		if (element instanceof DataCollector)
		{
			AnalysisType[] analysisTypes = ((DataCollector)element).getApplicableAnalysisTypes();
			if (analysisTypes != null && analysisTypes.length > 0)
				return true;
		}
			
		return false;
	}

	public Object[] getElements(Object inputElement)
	{
		return deferredContentManager.getChildren(inputElement);
	}

	public void dispose()
	{
		/* Don't need to handle anything */
	}

	public void inputChanged(Viewer viewer, Object oldInput, Object newInput)
	{
		this.input = newInput instanceof DataCollectorTreeInput ? (DataCollectorTreeInput)newInput : null;
		if (deferredContentManager == null)
		{
			class ExtendedDeferredContentManager extends DeferredTreeContentManager
			{
				private AbstractTreeViewer viewer;

				public ExtendedDeferredContentManager(ITreeContentProvider provider, AbstractTreeViewer viewer)
				{
					super(provider, viewer);
					this.viewer = viewer;
				}
				
				protected void addChildren(final Object parent, final Object[] children, IProgressMonitor monitor)
				{
			        WorkbenchJob updateJob = new WorkbenchJob(TraceMessages.AttachAdding) 
			        {
			            /** 
			             * @see org.eclipse.ui.progress.UIJob#runInUIThread(org.eclipse.core.runtime.IProgressMonitor)
			             */
			            public IStatus runInUIThread(IProgressMonitor updateMonitor) {
			                //Cancel the job if the tree viewer got closed
			                if (viewer.getControl().isDisposed() || updateMonitor.isCanceled()) {
								return Status.CANCEL_STATUS;
							}
			                viewer.add(parent, children);
			                return Status.OK_STATUS;
			            }
			        };
			        
			        updateJob.setSystem(true);
			        updateJob.addJobChangeListener(new JobChangeAdapter() {
						public void done(IJobChangeEvent event) {
							profileUI.initializeAfterFetch();
						}						
			        });
			        updateJob.schedule();
				}
			};
			
			deferredContentManager = new ExtendedDeferredContentManager(this, (AbstractTreeViewer)viewer);
		}	
	}
	
	
	public static class ParentChildNode
	{
		public Object parent, child;
		
		public ParentChildNode (Object parent, Object child)
		{
			this.parent = parent;
			this.child = child;
		}
		
		public boolean equals(Object obj)
		{
			if (obj instanceof ParentChildNode)
			{ 
				ParentChildNode parentChildNode2 = (ParentChildNode)obj;							
				return parentChildNode2.child.equals(child) && parentChildNode2.parent.equals(parent);
			}
			
			return false;
		}
	}
	
	/**
	 * This object will act as the input of the data collector tree.
	 * It will need to adapt to the IDeferredWorkbenchAdapter.
	 */
	public static class DataCollectorTreeInput implements IAdaptable
	{
		private DeferredDataCollectorAdapter deferredAgentAdapter;
		private ILaunchConfiguration configuration;
		
		public DataCollectorTreeInput (ILaunchConfiguration configuration)
		{
			this.configuration = configuration;
			deferredAgentAdapter = new DeferredDataCollectorAdapter(configuration);
		}
		public Object getAdapter(Class adapter)
		{
			if (adapter == IDeferredWorkbenchAdapter.class)
				return deferredAgentAdapter;
			
			return null;
		}
		/**
		 * @return the configuration
		 */
		public ILaunchConfiguration getConfiguration()
		{
			return configuration;
		}
		/**
		 * @param configuration the configuration to set
		 */
		public void setConfiguration(ILaunchConfiguration configuration)
		{
			this.configuration = configuration;
		}		
		
		/**
		 * @see java.lang.Object#equals(java.lang.Object)
		 */
		public boolean equals(Object o)
		{
			if (!(o instanceof DataCollectorTreeInput))
				return false;
			
			if (configuration != null)
				return configuration.equals(((DataCollectorTreeInput)o).configuration);
			
			return this == o;
		}
	}
	
	
	/**
	 * This deferred adapter is used to retrieve the data collectors
	 * available.
	 */
	public static class DeferredDataCollectorAdapter implements IDeferredWorkbenchAdapter 
	{
		private ILaunchConfiguration configuration;
		
		public DeferredDataCollectorAdapter(ILaunchConfiguration configuration)
		{
			this.configuration = configuration;
		}

		public void fetchDeferredChildren(Object object, IElementCollector collector, IProgressMonitor monitor)
		{
			try
			{
				if (configuration == null)
					return;
				
				/* Resolve the ID of the launch configuration */
				String launchConfigurationID = null;;
				try
				{
					launchConfigurationID = configuration.getType().getIdentifier();
		 
				} 
				catch (Exception e)
				{
					LauncherUtility.openErrorWithDetail(TraceMessages.LAUNCHER_COMMON_ERROR_TITLE, TraceMessages.ERROR_LAUNCH_TYPE_NOT_F, e);
					return;
				}
				DataCollectorManager dataCollManager = DataCollectorManager.getInstance();			
				DataCollectorAssociation dataCollectorAssociation = dataCollManager.getDataCollectorAssociator(launchConfigurationID);
				if (dataCollectorAssociation == null)
					return;
				
				// Notify that a fetch is starting
				dataCollManager.beginFetch();
				
				DataCollector[] dataCollectors = dataCollectorAssociation.getDataCollectors();
				List filteredDataCollectorSet = new ArrayList();
				monitor.beginTask(TraceMessages.DataCollectorFiltering, dataCollectors.length*2); //$NON-NLS-1$
				
				/* Apply the filtration */
				for (int i = 0; i < dataCollectors.length; i++)
				{
					DataCollectorAssociationData associationData = dataCollectorAssociation.getDataCollectorAssociationData(dataCollectors[i].getId());					
					ICollectorFiltration filter = associationData == null ? null : associationData.getFilter();
					if (filter == null || filter.include(dataCollectors[i].getId(), configuration, null))
						filteredDataCollectorSet.add(dataCollectors[i]);
					monitor.worked(1);
				}
				dataCollectors = (DataCollector[])filteredDataCollectorSet.toArray(new DataCollector[filteredDataCollectorSet.size()]);
				
				boolean useModifiedList = false;
				ArrayList modifiedListContainer = new ArrayList();
				
				/* We need to exclude data collectors that require an analysis 
				 * type and don't have any associated with them */
				for (int i = 0; i < dataCollectors.length; i++)
				{
					if (dataCollectors[i].isAnalysisTypeRequired())
					{
						if (!useModifiedList && dataCollectors[i].getApplicableAnalysisTypes().length <= 0)
						{
							useModifiedList = true;
							
							/* Copy what we so far have */
							for (int j = 0; j < i; modifiedListContainer.add(dataCollectors[j++]));
						}
						else if (useModifiedList)
							modifiedListContainer.add(dataCollectors[i]);
					}
					else if (useModifiedList)
						modifiedListContainer.add(dataCollectors[i]);
					monitor.worked(1);				
				}
				
				DataCollector[] filteredResult = null;
				if (useModifiedList)
				{
					filteredResult = new DataCollector[modifiedListContainer.size()];
					modifiedListContainer.toArray(filteredResult);
					collector.add(filteredResult, monitor);
				}
				else
				{
					collector.add(dataCollectors, monitor);
					filteredResult = dataCollectors;
				}
				
				// Pass the filtered collectors on to whomever may be listening
				dataCollManager.endFetch(filteredResult);
			}
			finally
			{
				collector.done();
				monitor.done();
			}

		}

		public ISchedulingRule getRule(Object object)
		{			
			return null;
		}

		public boolean isContainer()
		{			
			return true;
		}

		public Object[] getChildren(Object o)
		{			
			return null;
		}

		public ImageDescriptor getImageDescriptor(Object object)
		{			
			return null;
		}

		public String getLabel(Object o)
		{			
			return TraceMessages.DataCollectorChildren; //$NON-NLS-1$
		}

		public Object getParent(Object child)
		{
			if (child instanceof ParentChildNode)
			{
				return ((ParentChildNode)child).parent;
			}
			return null;
		}
		
	}
}
