/**********************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.trace.ui.internal.launcher.core;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.core.ProfileLaunchUtil;
import org.eclipse.hyades.trace.ui.internal.core.TraceUIImages;
import org.eclipse.hyades.trace.ui.internal.launcher.ILaunchConfigurationTabUpdater;
import org.eclipse.hyades.trace.ui.internal.launcher.AttachAgentTabProviders.AgentTreeItem;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSetType;
import org.eclipse.hyades.trace.ui.launcher.IProfilingType;
import org.eclipse.hyades.trace.ui.launcher.ProfilingAttribute;
import org.eclipse.hyades.trace.ui.launcher.ProfilingSetsManagerCopy;
import org.eclipse.jface.viewers.CheckboxTreeViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.tptp.trace.ui.internal.launcher.core.DataCollectorTreeContentProvider.ParentChildNode;
import org.eclipse.tptp.trace.ui.provisional.launcher.IConfiguration;
import org.eclipse.tptp.trace.ui.provisional.launcher.IConfigurationPage;
import org.eclipse.tptp.trace.ui.provisional.launcher.IStatusListener;


/**
 * This class is used to create and display a wizard that corresponds to the contributed
 * configuration that is associated to an entity under the launch configuration dialog.
 * It's a singleton class and its instance is accessible via <code>getInstance()</code>
 * 
 * @author Ali Mehregani
 */
public class EditOptionDelegate implements SelectionListener
{
	/* An instance of this singleton class */
	private static EditOptionDelegate editOptDelegate;
	
	/* Points to the profile tab UI */
	private static ILaunchConfigurationTabUpdater tabUpdater;
	
	/* The working copy of profiling set manager (for backward-compatibility) */
	private static ProfilingSetsManagerCopy profilingSetManageWC;
	
	//The pageWrapper used as a reference for launch validation
	private static ConfigurationPageWrapper confPageWrapperLastPage;
	/**
	 * Hide the visibility of this class.
	 */
	private EditOptionDelegate()
	{		
	}
	
	
	/**
	 * Return an instance of this singleton class.
	 * 
	 * @param tabUpdater The profile tab UI
	 * @param profilingSetsManagerCopy 
	 * @return An instance of this singleton class.
	 */
	public static SelectionListener getInstance(ILaunchConfigurationTabUpdater tabUpdater, ProfilingSetsManagerCopy profilingSetsManagerCopy)
	{
		if (editOptDelegate == null)
			editOptDelegate = new EditOptionDelegate();
		
		EditOptionDelegate.tabUpdater = tabUpdater;
		EditOptionDelegate.profilingSetManageWC = profilingSetsManagerCopy;
		
		return editOptDelegate;
	}

	
	/**
	 * Invoked when the user selects the 'Edit Option' button. <br/>
	 * Pre-condition:
	 * <ul>
	 * 	<li> 
	 * 		se.widget.getData(LAUNCH_CONFIGURATION) is expected to point 
	 * 		to the launch configuration item currently selected.
	 *  </li>
	 *  <li>
	 *  	se.widget.getData(CONFIGURATION_LOADER) is expected to point
	 *  	to the configuration loader that is associated with the entity
	 *  	selected.
	 *  </li>
	 * </ul>
	 * 
	 * @param se The selection event.
	 */
	public void widgetSelected(SelectionEvent se)
	{
		Object launchConfiguration = null;
		Object confLoader = null;
		Object profilingSetType = null;
		Object checkboxTreeViewer = null;
		Object checkstateListener = null;
		
		try
		{	
			launchConfiguration = se.widget.getData(LauncherConstants.LAUNCH_CONFIGURATION);
			confLoader = se.widget.getData(LauncherConstants.CONFIGURATION_LOADER);
			profilingSetType = se.widget.getData(LauncherConstants.PROFILING_SET_TYPE);
			checkboxTreeViewer = se.widget.getData(LauncherConstants.CHECKBOX_TREEVIEWER);
			checkstateListener = se.widget.getData(LauncherConstants.CHECKSTATE_LISTENER);
			
			boolean isInputProfilingType = profilingSetType != null;
			
			/* Disable the button if we don't have the launch configuration OR 
			 * (the input is NOT a profiling type AND we don't have a configuration loader) OR the 
			 * checkbox tree viewer is missing OR the check state listener is missing */
			if (!(launchConfiguration instanceof ILaunchConfiguration) || (!isInputProfilingType && !(confLoader instanceof ConfigurationLoader)) || 
				!(checkboxTreeViewer instanceof CheckboxTreeViewer) || !(checkstateListener instanceof ICheckStateListener))
			{
				((Button)se.widget).setEnabled(false);
				return;
			}
			
			ConfigurationLoader configurationLoader; 
			EditOptionWizard editOptWizard;
			int width = -1, height = -1;
			
			if (isInputProfilingType)
			{
				editOptWizard = new EditOptionWizard(tabUpdater, (ILaunchConfiguration)launchConfiguration, (IProfilingSetType)profilingSetType, profilingSetManageWC, (CheckboxTreeViewer)checkboxTreeViewer, (ICheckStateListener)checkstateListener);
			}
			else
			{
				configurationLoader = (ConfigurationLoader) confLoader;
				editOptWizard = new EditOptionWizard(tabUpdater, (ILaunchConfiguration)launchConfiguration, configurationLoader, (CheckboxTreeViewer)checkboxTreeViewer, (ICheckStateListener)checkstateListener);
				String widthStr = configurationLoader.retrieveAttribute(ConfigurationLoader.ATTRIBUTE_WIDTH);
				String heightStr = configurationLoader.retrieveAttribute(ConfigurationLoader.ATTRIBUTE_HEIGHT);				
				try
				{
					width = Integer.parseInt(widthStr);
					height = Integer.parseInt(heightStr);
				}
				catch (NumberFormatException nfe)
				{
					/* ignore the error */
				}
				

			}
			
			
			final int finalWidth = width;
			final int finalHeight = height;	
			WizardDialog wizardDialog = new WizardDialog(org.eclipse.swt.widgets.Display.getDefault().getActiveShell(), editOptWizard)
			{
				protected Point getInitialSize()
				{	
					Point size = super.getInitialSize();
					Point parentSize = getShell().getParent().getSize();									
					Point preferredSize = getShell().computeSize(SWT.DEFAULT, SWT.DEFAULT, true);
					final float RELATIVE_WIDTH = 0.7f;					
										
					return new Point (finalWidth > 0 ? finalWidth : Math.min((int)Math.round(parentSize.x * RELATIVE_WIDTH), preferredSize.x), finalHeight > 0 ? finalHeight : size.y);					
				}
				protected Point getInitialLocation(Point initialSize)
				{					
					Point parentLocation = getShell().getParent().getLocation();
					Point parentSize = getShell().getParent().getSize();
					int widthDifference = parentSize.x - initialSize.x;
					int heightDifference = parentSize.y - initialSize.y;
					
					/* Try to place the dialog in the middle of the parent dialog */
					if (widthDifference > 0 && heightDifference > 0)
					{
						return new Point(parentLocation.x + widthDifference/2, parentLocation.y + heightDifference/2);
					}
					else 
					{
						return super.getInitialLocation(initialSize);
					}
				}
			};
			wizardDialog.open();
		}
		catch (Exception ex)
		{			
			String configurationID = (confLoader instanceof ConfigurationLoader ? ((ConfigurationLoader)confLoader).getId() : LauncherMessages.LAUNCHER_COMMON_UNKNOWN);
			
			/* Display an error message */
			LauncherUtility.openErrorWithDetail(LauncherMessages.LAUNCHER_COMMON_ERROR_TITLE, 
					NLS.bind(LauncherMessages.ERROR_CONFIG_OPEN, configurationID), 
					ex);
		}
	}

	public void widgetDefaultSelected(SelectionEvent e)
	{
		/* Doesn't need to be implemented */		
	}
	
	
	/**
	 * Represents the edit option wizard that will be created 
	 * and displayed to the user once clicked on the 'Edit Option'
	 * button.
	 * 
	 * @author Ali Mehregani
	 */
	private static class EditOptionWizard extends Wizard
	{	
		/* The configuration loader */
		private ConfigurationLoader configurationLoader;
		
		/* A working copy of the launch configuration */
		private ILaunchConfigurationWorkingCopy launchConfigurationWC;
		
		/* Indicates whether the launch configuration working copy was constructed or not */
		private boolean isWorkingCopyConstructed;
		
		/* Represents the profile tab UI */
		private ILaunchConfigurationTabUpdater tabUpdater;
				
		/* Indicates if the input is a profiling type */
		private boolean isInputProfilingType;
		
		/* Stores the profiling type.  <code>isInputProfilingType</code> must be set for this to have any meaning */
		private IProfilingSetType profilingSetType;
		
		/* The working copy of the profiling set manager (for backward-compatibility).  
		 * <code>isInputProfilingType</code> must be set for this to have any meaning */
		private ProfilingSetsManagerCopy profilingSetManageWC;
		
		/* The dummy configuration page that will be created when the input is a profiling type.
		 * <code>isInputProfilingType</code> must be set for this to have any meaning */
		private ConfigurationPageWrapper dummyConfigurationPage;
		
		/* The tree viewer */
		private CheckboxTreeViewer treeViewer;

		/* The check state listener */
		private ICheckStateListener listener;
		
		/**
		 * Constructs this wizard based on the input.
		 * 
		 * @param tabUpdater The profile tab UI
		 * @param launchConfiguration The launch cofiguration
		 * @param configurationLoader The configuration loader that  
		 * will determine the wizard and its pages 
		 */
		public EditOptionWizard (ILaunchConfigurationTabUpdater tabUpdater, ILaunchConfiguration launchConfiguration, ConfigurationLoader configurationLoader, CheckboxTreeViewer treeViewer, ICheckStateListener listener)
		{
			this.configurationLoader = configurationLoader;
			this.isInputProfilingType = false;
			this.treeViewer = treeViewer;
			this.listener = listener;
			super.setWindowTitle(configurationLoader.getDialogTitle());
			commonInitializer(tabUpdater, launchConfiguration);
			
		}

		
		/**
		 * This constructor should be used when the input is a profiling type.
		 * The purpose of this is for backward compatability.
		 * 
		 * @param tabUpdater The profile tab UI
		 * @param launchConfiguration The launch cofiguration
		 * @param type The profiling set type that will serve as input
		 * @param profilingSetManageWC The working copy of the profiling set manager
		 */
		public EditOptionWizard(ILaunchConfigurationTabUpdater tabUpdater, ILaunchConfiguration launchConfiguration, IProfilingSetType type, ProfilingSetsManagerCopy profilingSetManageWC, CheckboxTreeViewer treeViewer, ICheckStateListener listener)
		{
			this.isInputProfilingType = true;
			this.profilingSetType = type;
			this.profilingSetManageWC = profilingSetManageWC;
			this.treeViewer = treeViewer;
			this.listener = listener;
			super.setWindowTitle(UIPlugin.getResourceString("EDIT_PROF_OPT")); //$NON-NLS-1$
			commonInitializer(tabUpdater, launchConfiguration);
		}

		
		private void commonInitializer(ILaunchConfigurationTabUpdater tabUpdater, ILaunchConfiguration launchConfiguration)
		{
			this.tabUpdater = tabUpdater;			
			
			if (launchConfiguration.isWorkingCopy())
			{
				isWorkingCopyConstructed = false;
				launchConfigurationWC = (ILaunchConfigurationWorkingCopy)launchConfiguration;
			}						
			else
			{
				try
				{
					isWorkingCopyConstructed = true;
					launchConfigurationWC = launchConfiguration.getWorkingCopy();
				} catch (CoreException e)
				{
					LauncherUtility.openErrorWithDetail(LauncherMessages.LAUNCHER_COMMON_ERROR_TITLE, 
							LauncherMessages.ERROR_CONFIG_WC_HANDLE, e);
				}
			}
		}

		/**
		 * Add all the pages of the configuration wizard to this
		 * wizard.
		 */
	    public void addPages() 
	    {		
	    	/* For backward compatibility */
	    	if (isInputProfilingType)
	    	{
	    		/* Wrap the profiling type using a ConfigurationPageWrapper and add the page */
	    		dummyConfigurationPage = new ConfigurationPageWrapper(profilingSetManageWC, profilingSetType);
	    		super.addPage (dummyConfigurationPage);
	    	}
	    	else
	    	{
		    	IConfigurationPage[] confPages = ((IConfiguration)configurationLoader.getConfigurationClass()).getConfigurationPages();
		    	if (confPages == null || confPages.length ==0)
		    		return;
		    	
		    	for (int i = 0; i < confPages.length - 1; i++)
		    	{
		    		addCurrentPage(confPages[i]);
		    	}
		    	
		    	// added for launch message display in JVMTI
		    	confPageWrapperLastPage =addCurrentPage(confPages[confPages.length -1]);
	    	}
	    }
	    
	    
	    private ConfigurationPageWrapper addCurrentPage(IConfigurationPage configurationPage)
		{
	    	configurationPage.reset (launchConfigurationWC);
    		ConfigurationPageWrapper confPageWrapper = new ConfigurationPageWrapper(configurationPage);
    		super.addPage (confPageWrapper);	
    		return confPageWrapper;
		}


		/**
	     * Invoked when the user clicks the finish button of
	     * the wizard.
	     */
		public boolean performFinish()
		{
			try
			{
				boolean status = true;
				
				/* Input is a profiling type */
				if (isInputProfilingType)
				{
					IProfilingType profilingType = profilingSetType.getProfilingType();
					
					/* Validate the configuration and display an error if it is invalid */
					String errorMessage = profilingType.validateConfiguration(profilingSetManageWC);
					
					/* There is an error message */
					if (errorMessage != null)
					{
						dummyConfigurationPage.setErrorMessage(errorMessage);
						dummyConfigurationPage.setPageComplete(false);
						
						return false;
					}
					/* The configuration is valid */
					else
					{
						dummyConfigurationPage.setErrorMessage(null);
						dummyConfigurationPage.setPageComplete(true);
						
						/* We'll need to persist the attributes of the profiling type in the working copy of the profiling set manager.
						 * This process will involve:
						 * 
						 * 1) Removing attributes from the default profiling set that is owned by the profiling type
						 * 2) Retrieving the attributes of the profiling type
						 * 3) Augmenting the profiling attributes with a contributor id
						 * 4) Adding the attributes to the default profiling set
						 */
						
						/* 1) Remove attributes */
						Map attr = profilingSetManageWC.getDefaultSet().getAttributes();
						String profilingTypeId = profilingSetType.getId();
						Collection keys = attr.keySet();
						List removalList = new ArrayList();
						Object currentKey;
						for (Iterator iter = keys.iterator(); iter.hasNext();)
						{
							currentKey = iter.next();
							ProfilingAttribute attribute = (ProfilingAttribute)attr.get(currentKey);
							if (profilingTypeId.equals(attribute.getContributorId()))
								removalList.add(currentKey);
						}
						for (int i = 0, markedDeletedElementsSize = removalList.size(); i < markedDeletedElementsSize; i++)
							attr.remove(removalList.get(i));
						
						
						/* 2) Retrieve attributes */
						ProfilingAttribute[] profilingAttributes = profilingType.getAttributes();
						
						if (profilingAttributes != null)							
						{
							/* 3) Augment attributes */
							for (int i = 0; i < profilingAttributes.length; profilingAttributes[i++].setContributorId(profilingTypeId));
							
							/* 4) Add the attributes */
							for (int i = 0; i < profilingAttributes.length; i++)
								attr.put(profilingAttributes[i].getName(), profilingAttributes[i]);							
						}
					}
					
					tabUpdater.update();					
				}
				
				/* Input is not a profiling type */
				else					
				{
					status = ((IConfiguration)configurationLoader.getConfigurationClass()).finishConfiguration(launchConfigurationWC);
					if (status == false)
						// added for enhancement 141450
						// Liz Dancy
						{
							confPageWrapperLastPage.setErrorMessage(LauncherMessages.ERROR_NO_PROBES_SELECTED);
												
						}
					/* We have to save our changes if we manually constructed the working copy */
					if (isWorkingCopyConstructed && status)
						launchConfigurationWC.doSave();
					else
						tabUpdater.update();
				}
								
				/* Attempt to select the tree item */
				ISelection selection = treeViewer.getSelection();
				Object selectedItem = null;
				if (selection instanceof StructuredSelection)
					selectedItem = ((StructuredSelection)selection).getFirstElement();
				
				if (selectedItem == null)
					return status;
			
				AgentTreeItem item = null;
				if  (selectedItem instanceof ParentChildNode || 
					(selectedItem instanceof AgentTreeItem && 
							((item = (AgentTreeItem)selectedItem).getAnalysisType() != null || item.getProfilingType() != null)))
					ProfileLaunchUtil.changeItemCheckedState(treeViewer, listener, selectedItem, true);				
				return status; 														
			} 
			catch (Exception e)
			{
				LauncherUtility.openErrorWithDetail(LauncherMessages.LAUNCHER_COMMON_ERROR_TITLE, 
										LauncherMessages.ERROR_CONFIG_FINISH, e);
				return false;
			}			
		}
		
	}
	
	
	/**
	 * Wraps an IConfigurationPage so that it can be used as a wizard page.
	 * 
	 * @author Ali Mehregani
	 */
	private static class ConfigurationPageWrapper extends WizardPage
	{
		/* The configuration page */
		private IConfigurationPage configuraitonPage;
		
		/* Stores the profiling type input.  Meaningful if isInputProfilingType is set */
		private IProfilingSetType profilingSetType;
		
		/* True if the input is a profiling type */
		private boolean isInputProfilingType;
		
		/* The error handler */
		private IStatusListener errorHandler;
		
		/**
		 * The constructor
		 * 
		 * @param configurationPage The configuration page that this wizard page should be
		 * constructed of.
		 */
		public ConfigurationPageWrapper (IConfigurationPage configurationPage)
		{
			super(configurationPage.getPageName(), configurationPage.getTitle(), configurationPage.getWizardBanner());
			super.setDescription(configurationPage.getDescription());
			this.configuraitonPage = configurationPage;
			this.isInputProfilingType = false;		
						
			/* Register a status listener that will update the wizard accordingly */
			errorHandler = new IStatusListener(){

				public void handleErrorEvent(String error)
				{
					setErrorMessage(error);
					setPageComplete(false);
				}

				public void handleOKEvent()
				{
					setErrorMessage(null);
					setPageComplete(true);
				}};
				
			configurationPage.addErrorListener(errorHandler);
		}

		
		/**
		 * This constructor is used when the input is a profiling type rather than an
		 * analysis type.  This is used for backward compatibility.
		 *  
		 * @param profilingSetManageWC The working copy of the profiling set manager
		 * @param profilingSetType The profiling type that will be used as input
		 */
		public ConfigurationPageWrapper(ProfilingSetsManagerCopy profilingSetManageWC, IProfilingSetType profilingSetType)
		{
			super(ConfigurationPageWrapper.class.getName(), TraceMessages.HPF_T, TraceUIImages.INSTANCE.getImageDescriptor(TraceUIImages.IMG_PROF_SET_WIZ_BAN));
			super.setDescription(LauncherMessages.CONFIGURATION_PROFILING_TYPE_DESC);
			this.profilingSetType = profilingSetType;
			this.isInputProfilingType = true;			
		}


		/**
		 * Create the controls of the wizard
		 * 
		 * @param parent The parent composite
		 */
		public void createControl(Composite parent)
		{
			Composite pageComposite = new Composite (parent, SWT.NONE);
			pageComposite.setLayout(new GridLayout());
			pageComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
						
			if (isInputProfilingType)
				profilingSetType.getProfilingType().createControl(pageComposite, profilingSetManageWC);
			else
				configuraitonPage.createControl(pageComposite);
			
			super.setControl (pageComposite);
		}

	}

}
