/**********************************************************************
 * Copyright (c) 2006, 2009 IBM Corporation, Intel Corporation.
 * 
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LauncherUtility.java,v 1.29 2009/03/03 21:13:21 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.trace.ui.internal.launcher.core;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.lang.reflect.InvocationTargetException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.StringTokenizer;
import java.util.Vector;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.hyades.internal.execution.local.common.Options;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.AgentConfigurationEntry;
import org.eclipse.hyades.internal.execution.local.control.InactiveProcessException;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.models.hierarchy.HierarchyFactory;
import org.eclipse.hyades.models.hierarchy.TRCEnvironmentVariable;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.trace.internal.ui.TraceConstants;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.core.TraceFilterManager;
import org.eclipse.hyades.trace.ui.internal.launcher.IProfileLaunchConfigurationConstants;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfileLaunchUtil;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfilingSetsManager;
import org.eclipse.hyades.trace.ui.internal.launcher.SelectedAgent;
import org.eclipse.hyades.trace.ui.internal.launcher.AttachAgentTabProviders.AgentTreeItem;
import org.eclipse.hyades.trace.ui.internal.util.FilterSetElement;
import org.eclipse.hyades.trace.ui.internal.util.FilterTableElement;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSet;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSetType;
import org.eclipse.hyades.trace.ui.launcher.ProfilingAttribute;
import org.eclipse.jdt.launching.IVMRunner;
import org.eclipse.jdt.launching.VMRunnerConfiguration;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.CheckboxTreeViewer;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.Clipboard;
import org.eclipse.swt.dnd.TextTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.List;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIConstants;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIPlugin;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceConstants;
import org.eclipse.tptp.trace.ui.internal.launcher.core.DataCollectorTreeContentProvider.ParentChildNode;
import org.eclipse.tptp.trace.ui.provisional.launcher.IDataCollectorBaseLauncher;
import org.eclipse.tptp.trace.ui.provisional.launcher.IDataCollectorMutualLauncher;
import org.eclipse.tptp.trace.ui.provisional.launcher.IDataCollectorSelfManageableLauncher;
import org.eclipse.tptp.trace.ui.provisional.launcher.ILaunchValidator;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.IWorkingSet;

/**
 * A utility class used by the launcher classes
 * 
 * @author Ali Mehregani
 */
public class LauncherUtility 
{
	/* Indicates that an object is of type AnlaysisType */
	public static final byte ANALYSIS_TYPE = 1;
	
	/* Indicates that an object is of type IProfilingSetType */
	public static final byte IPROFILING_SET_TYPE = 2;


	/**
	 * Constructs and returns a CoreException with the arguments passed in.
	 * 
	 * @param severity The severity
	 * @param message The error message 
	 * @param exception The exception that should be wrapped
	 * @return A core exception
	 */
	public static CoreException createCoreException(int severity, String message, Exception exception)
	{
		IStatus status = new Status (severity, UIPlugin.getPluginId(), 0, message == null ? "" : message, exception); 
		return new CoreException(status);
	}
	
	/**
	 * Equivalent to CoreException (severity, message, null)
	 */
	public static CoreException createCoreException(int severity, String message)
	{
		return createCoreException(severity, message, null);
	}
		
	
	/**
	 * Returns the value of hashtable.get(key).  If this entry does not exist and 'create' is true,
	 * then use valueType and 0-argument constructor to instantiate an object and set the value 
	 * of hashtable.get(key) to the object created.
	 * 
	 * @param hashtable The hashtable to retrieve data from
	 * @param key The key of the hashtable
	 * @param valueType The value type of the object that will be created if 'create' is set to true
	 * @param create Creates an object if set to true
	 * @return The object stored in hashtable.get(key)
	 */
	public static Object getHashtableEntry (Hashtable hashtable, Object key, Class valueType, boolean create)
	{
		Object hashtableEntry = hashtable.get(key);
		if (hashtableEntry == null && create)
		{
			try
			{
				hashtableEntry = valueType.newInstance();
				hashtable.put(key, hashtableEntry);
			} 
			catch (Exception e)
			{
				return null;
			}
		}
		return hashtableEntry;		
	}
	
	
	/**
	 * Makes the label's text bold.
	 * 
	 * @param label The label that will be bold.
	 */
	public static void setBold(Label label) 
	{
		FontData[] fontData = label.getFont().getFontData();
		for (int i = 0; i < fontData.length; i++) {
			fontData[i].setStyle(fontData[i].getStyle() | SWT.BOLD);
		}
		label.setFont(new Font(null,fontData));	
	}
	
	
	/**
	 * Used for displaying messages to the user when an unexpected event occurs.
	 * 
	 * @param parent The parent shell
	 * @param title The title of the dialog
	 * @param message The message to be displayed
	 * @param cause The cause of the error
	 */
	public static void openMessageWithDetail(Shell parent, int severity, String title, String message, String cause)
	{
		
		class CustomErrorDialog extends ErrorDialog
		{
			private String cause;
			private static final int LIST_ITEM_COUNT = 7;
			private List list;
			private boolean isListDisplayed;
			private Clipboard clipboard;
			
			public CustomErrorDialog(Shell parentShell, int severity, String dialogTitle, String message, String cause) 
			{
				super (parentShell, dialogTitle, message, new Status(severity, UIPlugin.getPluginId(), severity, "", new Throwable(cause)), IStatus.OK | IStatus.INFO | IStatus.WARNING | IStatus.ERROR);
				this.cause = cause;
				isListDisplayed = false;
			}
			
			public void openErrorDialog()
			{
				this.open();
			}			
			
		    			
		    protected org.eclipse.swt.widgets.List createDropDownList(Composite parent) 
		    {
		    	if (isListDisplayed)
		    	{
		    		isListDisplayed = false;		    		
		    		list.dispose();
		    		return list;
		    	}
		    	
		    	isListDisplayed = true;
		    	
		    	// create the list
		    	list = new List(parent, SWT.BORDER | SWT.H_SCROLL | SWT.V_SCROLL | SWT.MULTI);

		    	// fill the list
		    	if (cause != null)
		    	{
			    	StringTokenizer st = new StringTokenizer (cause, "\n");
			    	String indents = "";
			    	boolean addedIndentation = false;
			    	while (st.hasMoreTokens())
			    	{
			    		String currentToken = st.nextToken();
			    		currentToken = currentToken.replace ('\r', ' ');
			    		currentToken = currentToken.trim();
			    		if (!addedIndentation && currentToken.startsWith("at"))
			    		{
			    			indents += "  ";
			    			addedIndentation = true;
			    		}
			    		else if (addedIndentation && !currentToken.startsWith("at"))
			    			addedIndentation = false;
			    		
			    		
			    		list.add(indents + currentToken);
			    	}
		    	}
		    	
		        GridData data = new GridData(GridData.HORIZONTAL_ALIGN_FILL
		                | GridData.GRAB_HORIZONTAL | GridData.VERTICAL_ALIGN_FILL
		                | GridData.GRAB_VERTICAL);
		        data.heightHint = list.getItemHeight() * LIST_ITEM_COUNT;
		        data.horizontalSpan = 2;
		        list.setLayoutData(data);
		        list.setFont(parent.getFont());
		        
		        Menu copyMenu = new Menu(list);
		        MenuItem copyItem = new MenuItem(copyMenu, SWT.NONE);
		        copyItem.addSelectionListener(new SelectionListener() {
		            /*
		             * @see SelectionListener.widgetSelected (SelectionEvent)
		             */
		            public void widgetSelected(SelectionEvent e) {
		                copyToClipboard();
		            }

		            /*
		             * @see SelectionListener.widgetDefaultSelected(SelectionEvent)
		             */
		            public void widgetDefaultSelected(SelectionEvent e) {
		                copyToClipboard();
		            }
		        });
		        copyItem.setText(LauncherMessages.LAUNCHER_COMMON_COPY);
		        list.setMenu(copyMenu);
		    	
		        return list;
		    }
		    
		    private void copyToClipboard() {
		        if (clipboard != null)
		            clipboard.dispose();
		        StringBuffer clipBoardBuffer = new StringBuffer();		       
		        
		        String[] listEntrySelections = list.getSelection();
		        for (int i = 0; i < listEntrySelections.length; i++)
		        	clipBoardBuffer.append(listEntrySelections[i] + LauncherConstants.LINE_SEPARATOR);
		        clipboard = new Clipboard(list.getDisplay());
		        clipboard.setContents(new Object[] { clipBoardBuffer.toString() },
		                new Transfer[] { TextTransfer.getInstance() });
		    }


		}		
				
		CustomErrorDialog errorDialog = new CustomErrorDialog (parent, severity, title, message, cause);
		errorDialog.openErrorDialog();
	}
	
	public static void openMessageWithDetail(final int severity, final String title, final String message, Throwable t)
	{		
		String stackTrace = (t == null ? null : getExceptionStackTrace(t));
		Throwable exceptionCause = (t != null && t.getCause() != null ? t.getCause() : t);
			
		final String cause = (exceptionCause != null ? exceptionCause.getClass().getName() + "\n" + exceptionCause.getMessage() + "\n \n" + stackTrace : "");
		final IWorkbench workbench = UIPlugin.getDefault().getWorkbench();
		workbench.getDisplay().asyncExec(new Runnable()
		{

			public void run() 
			{
				openMessageWithDetail(workbench.getActiveWorkbenchWindow().getShell(), severity, title, message, cause);
				
			}});
		
	}
	
	
	public static void openErrorWithDetail(final String title, final String message, Throwable t)
	{		
		openMessageWithDetail(IStatus.ERROR, title, message, t); 		
	}
	
	
	public static void openWarningWithDetail(final String title, final String message, Throwable t)
	{		
		openMessageWithDetail(IStatus.WARNING, title, message, t); 		
	}
	
	/**
	 * Serialize the stack trace of a throwable into string and returns the result
	 * 
	 * @param t The throwable
	 * @return A string serialization of t's stack trace
	 */
	public static String getExceptionStackTrace(Throwable t)
	{
		Throwable throwable = findCause(t);
				
		StringWriter stackTrace = new StringWriter(); 
		throwable.printStackTrace(new PrintWriter(stackTrace));	
		
		return stackTrace.toString();
	}

	
	/**
	 * Determines the cause of an exception by returning the exception that is 
	 * wrapped in 't'
	 * 
	 * @param t The throwable
	 * @return The root cause of 't'
	 */
	public static Throwable findCause (Throwable t)
	{
		Throwable cause = null;
		if (t instanceof InvocationTargetException)
			cause = ((InvocationTargetException)t).getTargetException();			
		else if (t instanceof CoreException)
			cause = ((CoreException)t).getStatus().getException();				
		
		if (cause == null)
			return t;
		return findCause(cause);
	}
	
	
	/**
	 * This method is commonly used for backward compatibility.  It retrieves the data collector and
	 * analysis types that correspond to the profiling set that has the passed id.
	 * 
	 * @param profilingSetId A profiling set id.
	 * @return The data collectors and analysis types that correspond to the profiling set with the id
	 * passed in.
	 */
	public static ParentChildNode[] getSelectedDataCollectors (String profilingSetId)
	{
		if (profilingSetId == null)
			return null;
		
		ProfilingSetsManager manager = ProfilingSetsManager.instance();
		IProfilingSet set = (IProfilingSet)manager.getProfilingSets().get(profilingSetId);
		Vector itemsSelected = new Vector();
		if (set != null)
		{
			/* What are the set's profiling types? */
			java.util.List profilingTypes = set.getProfilingTypes();
			for (int i = 0, profilingTypeCount = profilingTypes.size(); i < profilingTypeCount; i++) 
			{
				ParentChildNode parentChildNode = findAnalysisTypeItem((String)profilingTypes.get(i));
				
				if (parentChildNode != null)
					itemsSelected.add(parentChildNode);
			}
		}
		
		ParentChildNode[] items = new ParentChildNode[itemsSelected.size()];
		itemsSelected.toArray(items);
		return items;
	}
	
	
	private static ParentChildNode findAnalysisTypeItem(String analysisTypeId) 
	{
		/* If this happens to be one of the tree profiling types that we converted 
		 * to analysis types, then change the id. */
		if ("org.eclipse.hyades.profilingType.execution".equals(analysisTypeId))
			analysisTypeId = "org.eclipse.tptp.analysisType.execution";
		else if ("org.eclipse.hyades.profilingType.memoryHeap".equals(analysisTypeId))
			analysisTypeId = "org.eclipse.tptp.analysisType.memoryHeap";
		else if ("org.eclipse.hyades.profilingType.methodCoverage".equals(analysisTypeId))
			analysisTypeId = "org.eclipse.tptp.analysisType.methodCoverage";
		
		/* Try and locate the profiling type by first looking under the JVMPI data collector */
		ProfilingSetsManager manager = ProfilingSetsManager.instance();
		DataCollector jvmpiDataCollector = DataCollectorManager.getInstance().getDataCollector(LauncherConstants.JVMPI_DATA_COLLECTOR_ID);		
		Object child = manager.getProfilingTypes().get(analysisTypeId);
		
		if (child == null)
			child = getAnalysisType (jvmpiDataCollector, analysisTypeId);
		
		if (child != null)
			return new ParentChildNode (jvmpiDataCollector, child);
		else
		{
			/* Start looking for the analysis type under each data collector.  Return the first match */
			DataCollector[] dataCollectors = DataCollectorManager.getInstance().getDataCollectors();
			for (int i = 0; i < dataCollectors.length; i++)
			{
				child = getAnalysisType(dataCollectors[i], analysisTypeId);
				if (child != null)
					return new ParentChildNode (dataCollectors[i], child);
			}
		}
				
		return null;
	}
	
	
	private static Object getAnalysisType (DataCollector dataCollector, String analysisTypeId)
	{
		AnalysisType[] analysisTypes = dataCollector.getApplicableAnalysisTypes();
		if (analysisTypes == null)
			return null;
		
		for (int i = 0; i < analysisTypes.length; i++) 
		{
			if (analysisTypes[i].getId().equals(analysisTypeId))
				return analysisTypes[i];
		}
		
		return null;
	}
	
	
	/**
	 * Use this method for serializing what is stored in a profiling set that is
	 * associated with a launch configuration.  This method should only be used if
	 * ATTR_PROFILING_COLLECTOR_AND_ANALYSIS is not available.
	 * 
	 * @param The id of the profiling set associated with a configuration
	 * @return A string serialization of the data collectors and analysis types that 
	 * the profiling set includes
	 */
	public static String serializeSelection(String profilingSetId)
	{
		if (profilingSetId == null)
			return null;
		
		ParentChildNode[] parentChildNodes = getSelectedDataCollectors(profilingSetId);
		Hashtable dataCollectors = new Hashtable();
		Vector selection = new Vector();
		
		for (int i = 0; i < parentChildNodes.length; i++) 
		{
			if (dataCollectors.get(parentChildNodes[i].parent) == null)
			{
				selection.add(parentChildNodes[i].parent);
				dataCollectors.put(parentChildNodes[i].parent, new Integer(selection.size()));
			}
				
			selection.add(((Integer)dataCollectors.get(parentChildNodes[i].parent)).intValue(), parentChildNodes[i]);
		}
		
		Object[] selectedElements = new Object[selection.size()];
		selection.toArray(selectedElements);
		return serializeSelection (selectedElements);
	}
	
	
	/**
	 * Equivalent to serializeSelection(checkBoxTreeViewer.getCheckedElements());
	 */
	public static String serializeSelection(CheckboxTreeViewer checkBoxTreeViewer)
	{
		return serializeSelection(checkBoxTreeViewer.getCheckedElements());
	}
	
	
	/**
	 * Serializes the tree selection to a string using the format specified by 
	 * {@link org.eclipse.hyades.trace.ui.internal.launcher.IProfileLaunchConfigurationConstants#ATTR_PROFILING_COLLECTOR_AND_ANALYSIS}
	 * 
	 * @param checkBoxTreeViewer The selected items representing data collectors and analysis types
	 * @return A string serialization of the tree selection
	 */
	public static String serializeSelection(Object[] selection)
	{
		String serializedTreeSelection = "";
		Object childNode;
		boolean lastSelectionWasDataCollector = false;
		for (int i = 0; i < selection.length; i++)
		{			
			childNode = null;
			if (selection[i] instanceof DataCollector)
			{
				if (serializedTreeSelection.length() > 0)
					serializedTreeSelection += ";";
				serializedTreeSelection += ((DataCollector)selection[i]).getId();	
				lastSelectionWasDataCollector = true;
			}
			else if (selection[i] instanceof ParentChildNode) 
			{
				childNode = ((ParentChildNode)selection[i]).child;
				String typeID = null;
				
				if (childNode instanceof AnalysisType)
					typeID = ((AnalysisType)childNode).getId();
				/* For backward compatibility */
				else if (childNode instanceof IProfilingSetType)
					typeID = ((IProfilingSetType)childNode).getId();
				
				if (typeID == null)
					continue;
				
				if (lastSelectionWasDataCollector)
					serializedTreeSelection += ":";
				else
					serializedTreeSelection += ",";
				
				serializedTreeSelection += typeID;
				lastSelectionWasDataCollector = false;
			}
		}
		
		return serializedTreeSelection;
	}
	
	
	/**
	 * Equivalent to unserializeSelection(conf.getAttribute(IProfileLaunchConfigurationConstants.
	 * ATTR_PROFILING_COLLECTOR_AND_ANALYSIS))
	 */
	public static Hashtable unserializeSelection (ILaunchConfiguration conf)
	{		
		try
		{
			String dataCollectorsStr = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_COLLECTOR_AND_ANALYSIS, (String)null);
			return unserializeSelection(dataCollectorsStr);
		} 
		catch (Exception e1)
		{
			/* Display an error */
			LauncherUtility.openErrorWithDetail(LauncherMessages.LAUNCHER_COMMON_ERROR_TITLE, LauncherMessages.ERROR_DCM_NOT_FOUND, e1);			
			return null;
		}
	}
	
	
	/**
	 * Parse the value of the attribute IProfileLaunchConfigurationConstants.ATTR_PROFILING_COLLECTOR_AND_ANALYSIS and
	 * return a hashtable that will indicate the data collectors and their associated analysis types
	 * that the user has selected.  The hashtable's content will be:
	 * <p>
	 * 	KEY = An object of type {@link org.eclipse.tptp.trace.ui.internal.launcher.core.DataCollector} corresponding
	 *  	  to a data collector that the user has selected. 
	 *  VALUE = A {@link java.util.List} with mixed types {@link org.eclipse.tptp.trace.ui.internal.launcher.core.AnalysisType} 
	 *  		and {@link org.eclipse.hyades.trace.ui.launcher.IProfilingSetType}.
	 * </p>
	 */
	public static Hashtable unserializeSelection (String dataCollectorsStr)
	{
		Hashtable collectorAnalysisMap = new Hashtable();
		java.util.List childTypesContainer = new ArrayList();

		
		if (dataCollectorsStr == null)
			return null;	
		
		StringTokenizer datacollectorTokenizer = new StringTokenizer(dataCollectorsStr, ";");
		while (datacollectorTokenizer.hasMoreTokens())
		{
			String selectedDataCollector = datacollectorTokenizer.nextToken();
			int dataCollectorInx = selectedDataCollector.indexOf(':');
			String dataCollectorID;
			
			if (dataCollectorInx == -1)
				dataCollectorID = selectedDataCollector;
			else
			{
				dataCollectorID = selectedDataCollector.substring(0, dataCollectorInx);
				StringTokenizer analysisTypesSelected = new StringTokenizer(selectedDataCollector.substring(dataCollectorInx + 1), ","); 
				
				/* Parse the analysis types */
				childTypesContainer = new ArrayList();
				
				while (analysisTypesSelected.hasMoreTokens())
				{
					String childTypeId = analysisTypesSelected.nextToken();					
					AnalysisType analysisType = AnalysisTypeManager.getInstance().getAnalysisType(childTypeId);					
					
					/* Add the analysis type, if one was found */
					if (analysisType != null)
						childTypesContainer.add(analysisType);
					
					/* Otherwise determine if the entry corresponds to an IProfilingSetType (for backward compatibility) */
					else
					{
						IProfilingSetType profilingSetType = (IProfilingSetType)ProfilingSetsManager.instance().getProfilingTypes().get(childTypeId);
						if (profilingSetType != null)
							childTypesContainer.add(profilingSetType);
					}
						
				}
			}
			
			DataCollector dataCollector = DataCollectorManager.getInstance().getDataCollector(dataCollectorID);

			/* Display a warning about the unresolved data collector */
			if (dataCollector == null)
			{
				LauncherUtility.openWarningWithDetail(
						LauncherMessages.LAUNCHER_COMMON_WARNING_TITLE, 
						NLS.bind(LauncherMessages.ERROR_DCM_INVALID, dataCollectorID), 
						null);
				continue;
			}
			
			
			collectorAnalysisMap.put(dataCollector, childTypesContainer);		
		}

		
		return collectorAnalysisMap;
	}
	
	
	
	/**
	 * Returns a list of the default options -- by default
	 * everything is turned off.
	 *  
	 * @return A default set of options for the Agent Controller
	 */
	public static java.util.List getDefaultProfilingOptions() 
	{
		java.util.List list = new ArrayList();
		String[][] options = Options.OPTIONS_DEFAULT;

		final String PREFIX = "SETOPTION_";
		for (int i=0;i<options.length;++i) 
		{
			list.add(new ProfilingAttribute(PREFIX + options[i][0], options[i][1]));
		}
		
		return list;
	}
	
	/**
	 * Returns a list of the default options -- by default everything is turned off.
	 *  
	 * @return A default set of options for the Agent Controller
	 */
	public static ProfilingAttribute[] getDefaultProfilingOptionsAttributes() 
	{
		String[][] options = Options.OPTIONS_DEFAULT;

		ProfilingAttribute[] attributes = new ProfilingAttribute[options.length];
		
		final String PREFIX = "SETOPTION_";
		for (int i=0;i<options.length;++i) 
		{
			attributes[i] = new ProfilingAttribute(PREFIX + options[i][0], options[i][1]);
		}
		
		return attributes;
	}
	
	/**
	 * Removes all duplicate option keys. The resulting value is
	 * the one with the highest precedence.
	 * 
	 * (extracted this method from the deprecated class: ProfilingSetsManager)
	 */
	public static java.util.List filterDuplicateOptions(java.util.List original) 
	{
		java.util.List result = new ArrayList();
		Iterator iter = original.iterator();
		while (iter.hasNext()) {
			addProfilingOption(result, (ProfilingAttribute)iter.next());
		}
		return result;
	}
	
	
	/**
	 * Add a profiling option to the set of existing options. If the option already exists,
	 * update the existing option value if necessary.
	 * (extracted this method from the deprecated class: ProfilingSetsManager)
	 * 
	 * @param options A list of options
	 * @param option The option that is being added
	 */
	public static void addProfilingOption(java.util.List options, ProfilingAttribute option)
	{
		final String PREFIX = "SETOPTION_";
		for(int idx=0; idx<options.size(); idx++)
		{
			ProfilingAttribute pOption = (ProfilingAttribute)options.get(idx);
			if(pOption.getName().equals(option.getName()))
			{
				if(pOption.getValue().equals(Options.OPTION_VALUE_FALSE))
				{
					//false value is overridden by any other value
					pOption.setValue(option.getValue());
				}
				else if(pOption.getValue().equals(Options.OPTION_VALUE_NONE))
				{
					//none value is overridden by any other value
					pOption.setValue(option.getValue());
				}
				else if(option.getValue().equals(Options.OPTION_VALUE_FALSE))
				{
					//false value does not override anything
					return;
				}
				else if(pOption.getValue().equals(Options.OPTION_VALUE_NONE))
				{
					//none value does not override anything
					return;
				}
				else if(pOption.getName().equals(PREFIX + Options.OPTION_STACK_INFORMATION))
				{
					//OPTION_VALUE_NORMAL take precedence to OPTION_VALUE_NONE
					//OPTION_VALUE_BOUNDARY take precedence to OPTION_VALUE_NORMAL
					if(pOption.getValue().equals(Options.OPTION_VALUE_BOUNDARY))
						 return; //higher precedence
					
					if(!option.getValue().equals(Options.OPTION_VALUE_NONE))
						 pOption.setValue(option.getValue());
				}
				else if(pOption.getName().equals(PREFIX + Options.OPTION_TRACE_MODE))
				{
					if(pOption.getValue().equals(Options.OPTION_VALUE_NOOBJECTCORRELATION))
						pOption.setValue(option.getValue()); // override no object correlation value with any other value				
				}
				
				return;
			}
		}
		
		options.add(option);
	}

	
	/**
	 * Returns the attribute values of the configuration element based on the 
	 * attribute names passed in.
	 * 
	 * @param configuration The configuration element
	 * @param attributeNames The attribute names
	 * @return A string array of size attributeNames.length corresponding to the
	 * attribute values of 'configuration'.
	 */
	public static String[] getAttributes(IConfigurationElement configuration, String[] attributeNames)
	{
		if (attributeNames == null || attributeNames.length <= 0)
			return null;
		
		String[] attributeValues = new String[attributeNames.length];
		for (int i = -1; ++i < attributeNames.length; attributeValues[i] = configuration.getAttribute(attributeNames[i]));
		return attributeValues;
	}

	
	/**
	 * Returns true if num is a positive non-zero integer; false otherwise
	 * 
	 * @param num The number in string format
	 * @return True if num > 0; false otherwise
	 */
	public static boolean isInputPositiveInteger(String num)
	{
		try 
		{
			if (Integer.parseInt(num)<=0) return false;
		}
		catch(Exception exc)
		{
			return false;
		}
		return true;
	}

	
	/**
	 * Opens an info dialog with the title and message passed in
	 * 
	 * @param title The title of the dialog
	 * @param message The message that will be displayed
	 */
	public static void openInfo(String title, String message)
	{
		MessageDialog.openInformation(UIPlugin.getActiveWorkbenchShell(), title, message);
	}	
	
	
	/**
	 * Returns the port of Agent Controller that's stored in the launch configuration 
	 * 
	 * @param conf The launch configuration
	 * @return The port that launch configuration is running on
	 * @throws CoreException If attributes can't be retrieved.
	 */
	public static int getPort (ILaunchConfiguration conf) throws CoreException
	{
		IPreferenceStore store = CommonUIPlugin.getDefault().getPreferenceStore();
		int port = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PORT, store.getInt(CommonUIConstants.LOCALHOST_PORT));
		return port;		
	}
	
	
	/**
	 * Resolves and returns the name, parameter, and VM parameter of the
	 * process passed in.
	 * 
	 * @param process The process
	 * @return A string[] of size 3 with the content {name, parameter, vmparameter}
	 * @throws InactiveProcessException If the process is inactive
	 */
	public static String[] resolveProcessAttributes(Process process) throws InactiveProcessException
	{        
        String pName = process.getParameters();

        if (pName == null) {
            pName = LauncherMessages.LAUNCHER_COMMON_UNKNOWN;
        }

        String params = "";
        String vmparam = "";

        /* If the process happens to be a java profiled process, then use the following format:
         * 
         */
        if (process.getExecutable().startsWith("java")) {
            String exeName = process.getParameters();

            if (exeName == null) {
                exeName = LauncherMessages.LAUNCHER_COMMON_UNKNOWN;
            }

            if (!exeName.startsWith("-X")) { //log agents
                exeName = "-X " + exeName;
            }
            
            exeName = exeName.trim();
            int idx = exeName.indexOf(" ");

            if ((idx != -1) && (idx < (exeName.length() + 1))) {
                exeName = exeName.substring(idx + 1).trim();
            }

            while (exeName.startsWith("-") || exeName.startsWith("\"")) {
                idx = findVMendIdx(exeName);

                if (idx != -1) {
                    vmparam += (" " + exeName.substring(0, idx));
                    exeName = exeName.substring(idx + 1).trim();
                } else {
                    vmparam = exeName;
                    exeName = "";
                }
            }

            idx = exeName.indexOf(" ");

            if (idx != -1) {
                params = exeName.substring(idx);
                exeName = exeName.substring(0, idx);
            }

            pName = exeName;
        }
        
        return new String[]{pName, params, vmparam};
	}
	
    private static int findVMendIdx(String vmargs) 
    {
        String space = " ";
        String quote = "\"";
        int startIdx = 0;
        int endIdx;
        int qIdx;

        endIdx = vmargs.indexOf(space);

        if (endIdx != -1) {
            qIdx = vmargs.substring(startIdx, endIdx).indexOf(quote);

            if (qIdx != -1) {
                startIdx = endIdx + vmargs.substring(endIdx + 1).indexOf(quote) + 1;
                endIdx = startIdx + findVMendIdx(vmargs.substring(startIdx + 1)) + 1;
            }
        }

        return endIdx;
    }

    
    /**
     * Update the state with the status passed in.
     * 
     * @param status The new status of the agent
     * @param source The source whose status needs to be changed
     */
	public static void sendProfileEvent(final int profileEvent, final Object source)
	{
		Display d = Display.getDefault();

		d.asyncExec(new Runnable() {
                public void run() {                    
                    ProfileEvent event = UIPlugin.getDefault().getProfileEvent();

                    event.setSource(source);
                    event.setType(profileEvent);
                    UIPlugin.getDefault().notifyProfileEventListener(event);
                }
		});
	}

	
	/**
	 * Converts the passed in string array to a flat string with each
	 * entry separated with a space
	 *  
	 * @param attributeValue The String array to convert
	 * @return A flat string
	 */
	public static String convertToString(String[] attributeValue) 
	{		
		if (attributeValue.length == 0) {
			return null;
		}
		return ProfileLaunchUtil.convertToDelimitedString(attributeValue, ' ');
	}
	
	
	/**
	 * If 'selectedElement' happens to be a data collector selected under 'launchConfiguration', then
	 * its DataCollectorAssociationData is returned.
	 * 
	 * @param launchConfiguration The launch configuration
	 * @param selectedElement The selected element
	 * @return The associated DataCollectorAssociationData between launchConfiguration and selectedElement.
	 */
	public static DataCollectorAssociationData findDataCollectorAssociationData(ILaunchConfiguration launchConfiguration, Object selectedElement)
	{
		try
		{
			DataCollector dataCollector = null;
			if (selectedElement instanceof DataCollector)
			{
				dataCollector = ((DataCollector)selectedElement);
			}
			else if (selectedElement instanceof AgentTreeItem)
			{
				dataCollector = ((AgentTreeItem)selectedElement).getDataCollector();
			}
			
			if (dataCollector != null)
			{
				DataCollectorAssociation dataCollectionAssociation = DataCollectorManager.getInstance().getDataCollectorAssociator(launchConfiguration.getType().getIdentifier());			
				if (dataCollectionAssociation != null)
					return dataCollectionAssociation.getDataCollectorAssociationData(dataCollector.getId());
			}
		} 
		catch (Exception e)
		{			
		}
		return null;
	}
	
	
	/**
	 * Creates an agent configuration entry with the name and value passed in.
	 * The type of the entry is set to LauncherConstants#SET_OPTION_PREFIX
	 * 
	 * @param name The name of the configuration entry
	 * @param value The value of the configuration entry
	 * @return An agent configuration entry
	 */
	public static AgentConfigurationEntry createAgentConfigurationEntry (String name, String value)
	{
		AgentConfigurationEntry entry = new AgentConfigurationEntry();
		entry.setName(name);
		entry.setValue(value);
		entry.setType(LauncherConstants.SET_OPTION_PREFIX);
		
		return entry;
	}
	
	
	/**
	 * Associates an agent with an analysis type by adding the appropriate agent
	 * configuration entry.  If the agent happens to be null, then the association is
	 * not made and only the configuration entry is returned.
	 * 
	 * @param agent The agent to associate the analysis type with (can be null)
	 * @param id The id of the analysis type.
	 * 
	 * @return The configuration entry that is associated with the agent.
	 */
	public static AgentConfigurationEntry associateAnalysisType(Agent agent, String id)
	{	
		String name = LauncherConstants.ANALYSIS_TYPE_ASSOCIATION + "(" + id + ")";
		String value = id;
		
		/* return existing analysis type if it exists */
		for (int i=0; agent != null && i < agent.getConfiguration().size(); i++) {
			AgentConfigurationEntry entry = agent.getConfiguration().getEntryAt(i);
			if (entry.getName().equals(name) && entry.getValue().equals(value))
				return entry;
		}
		
		AgentConfigurationEntry entry = createAgentConfigurationEntry(LauncherConstants.ANALYSIS_TYPE_ASSOCIATION + "(" + id + ")", id);		
		if (agent != null)
			agent.getConfiguration().addEntry(entry);
		
		return entry;
	}
	
	
	/**
	 * <p>
	 * Validates the data collector and analysis types selected and returns a hashtable with the following
	 * content: <br>
	 * KEY = A data collector launch delegate
	 * VALUE = The associated data collector id
	 * </p>
	 * <p>
	 * 	null is returned if any of the validation fails
	 * </p>
	 * 
	 * @param selectedLaunchItems The selected launch items:  KEY = An object of type {@link DataCollector} VALUE = A list of type {@link AnalysisType} or {@link IProfilingSetType}
	 * @param configuration The launch configuration
	 * @param monitor The progress monitor
	 * @return A hashtable describing the launch delegate & data collector association
	 * 
	 * @throws CoreException In case of any unexpected error
	 */
	public static Hashtable validateLaunchItems(Hashtable selectedLaunchItems, ILaunchConfiguration configuration, IProgressMonitor monitor) throws CoreException
	{
		DataCollectorManager dataCollectorManager = DataCollectorManager.getInstance();
		Enumeration dataCollectors = selectedLaunchItems.keys(); 
		IDataCollectorBaseLauncher dataCollectorLaunchDelegate;
		Hashtable delegateCollectorAssociation = new Hashtable();
		
		while (dataCollectors.hasMoreElements())
		{
			DataCollector dataCollector = (DataCollector)dataCollectors.nextElement();
			dataCollectorLaunchDelegate = dataCollectorManager.getDataCollectorLaunchDelegate(dataCollector.getId(), configuration);
			
			/* Display a warning about not being able to resolve the delegate of the data collector */
			if (dataCollectorLaunchDelegate == null)
			{
				LauncherUtility.openWarningWithDetail(LauncherMessages.LAUNCHER_COMMON_WARNING_TITLE, 
						NLS.bind(LauncherMessages.WARNING_DCM_UNRESOLVED_DELEGATE, dataCollector.getId()), null);
				continue;
			}

			delegateCollectorAssociation.put(dataCollectorLaunchDelegate, dataCollector.getId());	
			
			/* We'll need to notify the selected data collectors and analysis types of the launch.  If any
			 * one of them returns a status that is not OK, then display an appropriate message.  If any
			 * of the status returned is IStatus.ERROR, then halt the launch. */
			DataCollectorAssociationData associationData = LauncherUtility.findDataCollectorAssociationData(configuration, dataCollector);
			if (associationData != null)
			{
				if (!continueLaunch (associationData.getValidator(), configuration, monitor))
					return null;
			}
			
			/* For every analysis type do a validation */
			ArrayList analysisTypes = (ArrayList)selectedLaunchItems.get(dataCollector);
			for (int i = 0, analysisTypeCount = analysisTypes.size(); i < analysisTypeCount; i++) 
			{
				Object currentType = analysisTypes.get(i);
				if (currentType instanceof AnalysisType)
				{
					if (!continueLaunch(((AnalysisType)currentType).getValidator(), configuration, monitor))
						return null;
				}
			}			
		}	
		
		return delegateCollectorAssociation;
	}
	
	
	/**
	 * Checks the status returned by the validator.  The boolean returned is an 
	 * indicator as to whether the launch should continue or not.
	 * 
	 * @param validator Used to validate the status
	 * @return A flag indicating whether the launch should continue or not.
	 */
	public static boolean continueLaunch(ILaunchValidator validator, ILaunchConfiguration configuration, IProgressMonitor monitor) 
	{
		if (validator == null)
			return true;
		
		IStatus status = validator.launchNotification(configuration);
		int severity;
		if (status == null || (severity = status.getSeverity()) == IStatus.OK)
			return true;
		
		/* We need to display the message */
		LauncherUtility.openMessageWithDetail(status.getSeverity(), "", status.getMessage(), status.getException());
		if (severity == IStatus.ERROR)
		{
			monitor.setCanceled(true);
			return false;
		}
		return true;
	}
	
	
	/** 
	 * For every non-mutual launch delegate invoke the launch method and for every mutual
	 * launch delegate, invoke the preLaunch method.  
	 * 
	 * @return A list that includes only the launch delegates of all the delegates passed in
	 * @throws CoreException In case of any error 
	 */
	public static ArrayList delegateInit (ArrayList launchDelegates, ILaunchConfiguration conf, String mode, ILaunch launch, IProgressMonitor monitor) throws CoreException
	{
		ArrayList mutualLaunchDelegates = new ArrayList();
		for (int i = 0, delegateCount = launchDelegates.size(); i < delegateCount; i++)
		{			
			IDataCollectorBaseLauncher currentDelegate = (IDataCollectorBaseLauncher)launchDelegates.get(i);
			
			/* Mutual launcher */
			if (currentDelegate.isMutualLauncher())
			{
				((IDataCollectorMutualLauncher)currentDelegate).preLaunch(conf, mode, launch, new SubProgressMonitor(monitor, 1));
				mutualLaunchDelegates.add(currentDelegate);
			}
			/* Non-mutual launcher */
			else
			{
				((IDataCollectorSelfManageableLauncher)currentDelegate).launch(conf, mode, launch, new SubProgressMonitor(monitor, 1));					
			}
		}	
		
		return mutualLaunchDelegates;
	}
	
	
	public static IWorkingSet[] getActiveWorkingSets() {
		class GetActiveWorkbenchWindowRunnable implements Runnable {
			private IWorkbenchWindow window;
			
			public IWorkbenchWindow getActiveWorkbenchWindow() {
				return window;
			}
			
			public void run() {
				window = UIPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow();
			}
		}

		GetActiveWorkbenchWindowRunnable runnable = new GetActiveWorkbenchWindowRunnable();
		Display.getDefault().syncExec(runnable);
		IWorkbenchWindow workbenchWindow = runnable.getActiveWorkbenchWindow();
		IWorkbenchPage activePage = workbenchWindow.getActivePage();
		return activePage.getWorkingSets();
	}
	
	public static void storeAutogeneratedFilterSet(final ArrayList filters) {
		String gereratedFilterSetName = TraceConstants.AUTO_GENERATED_FILTER_SET;

		ArrayList filtersSetList = TraceFilterManager.getInstance().getFilterSetCopy();
		FilterSetElement generatedFilterSet = null;
		for (int i = 0; i < filtersSetList.size(); i++) {
			FilterSetElement filterSet = (FilterSetElement) filtersSetList.get(i);
			if (gereratedFilterSetName.equals(filterSet.getName())) {
				generatedFilterSet = filterSet;
				break;
			}
		}
		if (generatedFilterSet == null) {
			generatedFilterSet = new FilterSetElement(gereratedFilterSetName);
			generatedFilterSet.setName(gereratedFilterSetName);
			filtersSetList.add(generatedFilterSet);
		}
		
		FilterTableElement finalFilter = new FilterTableElement("*", "*", "EXCLUDE");
		filters.add(finalFilter);
		
		generatedFilterSet.setChildren(filters);
		TraceFilterManager.getInstance().storeFilterSetList(filtersSetList);
	}

	public static void removeAutogeneratedFilterSet() {
		ArrayList filtersSetList = TraceFilterManager.getInstance().getFilterSetCopy();
		for (int i = 0; i < filtersSetList.size(); i++) {
			FilterSetElement filterSet = (FilterSetElement) filtersSetList.get(i);
			if (TraceConstants.AUTO_GENERATED_FILTER_SET.equals(filterSet.getName())) {
				filtersSetList.remove(i);
				break;
			}
		}
		TraceFilterManager.getInstance().storeFilterSetList(filtersSetList);
	}

	public static boolean isAutoFilteringCriteria(final ILaunchConfiguration config) throws CoreException {
		return config.getAttribute(IProfileLaunchConfigurationConstants.ATTR_AUTO_FILTER_CRITERIA, false);
	}
	
	public static ILaunchConfiguration addExtendedConfigurationDefaults(final ILaunchConfiguration config) {
		ILaunchConfigurationWorkingCopy wc;
		try {
			wc = config.getWorkingCopy();
			ProfilingSetsManager manager = ProfilingSetsManager.instance();
			IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();
			boolean autoFilteringCriteriaDefault = store.getBoolean(TraceConstants.AUTO_FILTER_CRITERIA_OPTION);
			wc.setAttribute(IProfileLaunchConfigurationConstants.ATTR_AUTO_FILTER_CRITERIA, autoFilteringCriteriaDefault);
			wc.setAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_SET, manager.getDefaultSet() == null ? null : manager.getDefaultSet().getId());
			wc.setAttribute(IProfileLaunchConfigurationConstants.ATTR_FILTER_SET, autoFilteringCriteriaDefault ? TraceConstants.AUTO_GENERATED_FILTER_SET : LauncherConstants.DEFAULT_FILTER_ID);
			return wc.doSave();
		} catch (CoreException exception) {
		}
		return config;
	}
	
	/**
	 * Retrieves the file with 'path' from a plug-in in the workspace
	 * 
	 * @param path The path of the file to retrieve
	 */
	public static File getFileInPlugin(IPath path)
	{
		//return LaunchingPlugin.getFileInPlugin(path);
		try {
			URL installURL = new URL(Platform.getBundle("org.eclipse.jdt.launching").getEntry("/"), path.toString()); //$NON-NLS-1$
			URL localURL = FileLocator.toFileURL(installURL);
			return new File(localURL.getFile());
		} catch (IOException ioe) {
			return null;
		}
	}
	
	

	

	/**
	 * Used to extract attributes from configuration associated with
	 * a JUnit launch item
	 * 
	 * @author Ali Mehregani
	 */
	public static class JUnitConfigurationExtended extends org.eclipse.jdt.junit.launcher.JUnitLaunchConfigurationDelegate
	{
		private IVMRunner runner;
		private VMRunnerConfiguration configuration;
		
		public IVMRunner getVMRunner(ILaunchConfiguration configuration, String mode) throws CoreException
		{
			if (runner == null)
			{
				runner = new IVMRunner()
				{
					public void run(VMRunnerConfiguration configuration, ILaunch launch, IProgressMonitor monitor) throws CoreException
					{
						JUnitConfigurationExtended.this.configuration = configuration;
					}
				};
			}
			
			return runner;
		}		
		
		
		public VMRunnerConfiguration getVMRunnerConfiguration()
		{
			return configuration;
		}
	}
	

	/**
	 * Using the configuration passed in retrieve the agents that have been selected
	 * under the agents tab.
	 * 
	 * @param configuration The launch configuration
	 * @return The selected agents
	 * @throws CoreException 
	 */
	public static SelectedAgent[] retrieveSelectedAgents(ILaunchConfiguration configuration) throws CoreException
	{
		String selection = configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_AGENTS, ""); //$NON-NLS-1$
		StringTokenizer selectedItems = new StringTokenizer(selection, ";"); //$NON-NLS-1$
		java.util.List selectedAgents = new ArrayList();
		
		while (selectedItems.hasMoreTokens())
		{
			String selectedItem = selectedItems.nextToken();
			SelectedAgent agentAttribute = new SelectedAgent(selectedItem);
			
			if (agentAttribute.getPid() != null && agentAttribute.getCollectorId() != null)
				selectedAgents.add(agentAttribute);
		}
		
		return (SelectedAgent[])selectedAgents.toArray(new SelectedAgent[selectedAgents.size()]);
	}
	
	
	/**
	 * Adds an environment variable to the process proxy with the name 
	 * {@link LauncherConstants#LAUNCH_CONFIGURATION_ID} and the value set to launchConfigId
	 * 
	 * @param processProxy The process proxy
	 * @param launchConfigId A launch configuration id
	 */
	public static void addLaunchConfigId (TRCProcessProxy processProxy, String launchConfigId)
	{
		TRCEnvironmentVariable variable = HierarchyFactory.eINSTANCE.createTRCEnvironmentVariable();
		variable.setName(CommonUITraceConstants.LAUNCH_CONFIGURATION_ID);
		variable.setValue(launchConfigId);		
		processProxy.getEnvironmentVariables().add(variable);
	}
}
