/**********************************************************************
 * Copyright (c) 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.trace.ui.internal.launcher.core;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.tptp.trace.ui.provisional.launcher.IAvailabilityTester;
import org.eclipse.tptp.trace.ui.provisional.launcher.ILightConfiguration;

/**
 * Loads non-editable configurations.
 * 
 * @author Ali Mehregani
 */
public class LightConfigurationLoader
{
	/* The id of the configuration */
	protected String id;
	
	/* The configuration class */
	protected ILightConfiguration configurationClass;
	
	/* Indicates whether the configuration is editable */
	protected boolean editable;
	
	/* The availability tester object */
	protected IAvailabilityTester availabilityTester;
	
	
	/**
	 * Constructor - Requires an Id
	 * 
	 * @param id The id of the configuration
	 */
	protected LightConfigurationLoader(String id)
	{
		this.id = id;
		this.editable = false;
	}
	
	
	/**
	 * Construct an instance of this class based on the configuration element passed in.
	 * 
	 * @param configuration The configuration element
	 * @return An instance of this class based on the details of the configuration element
	 * passed in.  If the element happens to be invalid, then null will be returned.
	 */
	public static LightConfigurationLoader constructInstance(IConfigurationElement configuration)
	{		
		String[] attributeValues = LauncherUtility.getAttributes(configuration, new String[] {"id", "class", "testAvailabilityClass"});
		
		/* Id is a required field */
		if (attributeValues[0] == null)
			return null;
		
		ILightConfiguration configurationClass = null;
		IAvailabilityTester availabilityTester = null;
		try
		{
			if (attributeValues[1] != null)
				configurationClass = (ILightConfiguration)configuration.createExecutableExtension("class");
			
			if (attributeValues[2] != null)
				availabilityTester = (IAvailabilityTester)configuration.createExecutableExtension("testAvailabilityClass");
		} 
		catch (Exception e)
		{
			return null;
		}
		
		LightConfigurationLoader lcl = new LightConfigurationLoader(attributeValues[0]);
		lcl.setConfigurationClass(configurationClass);
		lcl.setAvailabilityTester(availabilityTester);
		
		return lcl;
	}
	
	
	public String getId()
	{
		return id;
	}


	public void setId(String id)
	{
		this.id = id;
	}
	
	public ILightConfiguration getConfigurationClass()
	{
		return configurationClass;
	}


	public void setConfigurationClass(ILightConfiguration configurationClass)
	{
		this.configurationClass = configurationClass;
	}


	public boolean isEditable()
	{
		return editable;
	}


	public void setEditable(boolean editable)
	{
		this.editable = editable;
	}


	public IAvailabilityTester getAvailabilityTester()
	{
		return availabilityTester;
	}


	public void setAvailabilityTester(IAvailabilityTester availabilityTester)
	{
		this.availabilityTester = availabilityTester;
	}
}
