/**********************************************************************
 * Copyright (c) 2006 IBM Corporation, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.tptp.trace.ui.internal.launcher.deleg.application;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.IDebugEventSetListener;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.model.IDebugTarget;
import org.eclipse.debug.core.model.IProcess;
import org.eclipse.hyades.trace.ui.internal.util.FilterTableElement;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.jdt.launching.IJavaLaunchConfigurationConstants;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;

public class AbstractJavaAppletDelegate extends AbstractJavaDelegate implements IDebugEventSetListener
{	
	/** The working directory */
	private File workingDir;

	/** The launch */
	private ILaunch launch;
	
	/**
	 * Mapping of ILaunch objects to File objects that represent the .html file
	 * used to initiate the applet launch.  This is used to delete the .html
	 * file when the launch terminates.
	 */
	private static Map fgLaunchToFileMap = new HashMap();

	public AbstractJavaAppletDelegate(String agentName, String agentType)
	{
		super(agentName, agentType);
	}

	public void preLaunch(ILaunchConfiguration configuration, String mode, ILaunch launch, IProgressMonitor monitor) throws CoreException
	{	
		workingDir = javaLaunchConfigurationDelegate.verifyWorkingDirectory(configuration);
		this.launch = launch;
		super.preLaunch(configuration, mode, launch, monitor);
	}
		
	protected String getWorkingDirectory(ILaunchConfiguration conf)
	{		
		return workingDir.getAbsolutePath();		
	}

	protected String getProgramArguments(ILaunchConfiguration conf)
	{
		File htmlFile = buildHTMLFile(conf, workingDir);
		
		// Add a debug listener if necessary 
		if (fgLaunchToFileMap.isEmpty()) {
			DebugPlugin.getDefault().addDebugEventListener(this);
		}
		
		// Add a mapping of the launch to the html file 
		fgLaunchToFileMap.put(launch, htmlFile);
		try
		{
			javaLaunchConfigurationDelegate.setDefaultSourceLocator(launch, conf);
		} catch (CoreException e)
		{			
			e.printStackTrace();
		}

		
		return htmlFile.getName();
	}

	protected String getMainTypeName(ILaunchConfiguration conf)
	{
		try
		{
			javaLaunchConfigurationDelegate.verifyMainTypeName(conf);
			return conf.getAttribute(IJavaLaunchConfigurationConstants.ATTR_APPLET_APPLETVIEWER_CLASS, IJavaLaunchConfigurationConstants.DEFAULT_APPLETVIEWER_CLASS);
		} catch (CoreException e)
		{
			e.printStackTrace();
			return null;
		}		
	}

	protected String getVMArguments(ILaunchConfiguration conf)
	{
		try
		{
			return getJavaPolicyFile(workingDir) + ' ' + javaLaunchConfigurationDelegate.getVMArguments(conf);
		} catch (CoreException e)
		{			
			e.printStackTrace();
			return null;
		}
	
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.IDebugEventSetListener#handleDebugEvents(org.eclipse.debug.core.DebugEvent[])
	 */
	public void handleDebugEvents(DebugEvent[] events) {
		for (int i = 0; i < events.length; i++) {
			DebugEvent event = events[i];
			Object eventSource = event.getSource();
			switch(event.getKind()) {
				
				// Delete the HTML file used for the launch
				case DebugEvent.TERMINATE :
					if (eventSource != null) {
						ILaunch launch = null;
						if (eventSource instanceof IProcess) {
							IProcess process = (IProcess) eventSource;
							launch = process.getLaunch();
						} else if (eventSource instanceof IDebugTarget) {
							IDebugTarget debugTarget = (IDebugTarget) eventSource;
							launch = debugTarget.getLaunch();
						}
						File temp = (File) fgLaunchToFileMap.get(launch);
						if (temp != null) {
							try {
								fgLaunchToFileMap.remove(launch);
								temp.delete();
							} finally {
								if (fgLaunchToFileMap.isEmpty()) {
									DebugPlugin.getDefault().removeDebugEventListener(this);
								}
							}
						}
					}
					break;
			}
		}
	}
	
	
	/**
	 * Using the specified launch configuration, build an HTML file that specifies the
	 * applet to launch.  Return the name of the HTML file.
	 * 
	 * @param dir the directoru in which to make the file
	 */
	private File buildHTMLFile(ILaunchConfiguration configuration, File dir) {
		FileWriter writer = null;
		File tempFile = null;
		try {
			String name = javaLaunchConfigurationDelegate.getMainTypeName(configuration);
			tempFile = new File(dir, name + System.currentTimeMillis() + ".html"); //$NON-NLS-1$ //$NON-NLS-2$
			writer = new FileWriter(tempFile);
			writer.write("<html>\n"); //$NON-NLS-1$
			writer.write("<body>\n"); //$NON-NLS-1$
			writer.write("<applet code="); //$NON-NLS-1$
			writer.write(name);
			writer.write(".class "); //$NON-NLS-1$
			String appletName = configuration.getAttribute(IJavaLaunchConfigurationConstants.ATTR_APPLET_NAME, ""); //$NON-NLS-1$
			if (appletName.length() != 0) {
				writer.write("NAME =\"" + appletName + "\" "); //$NON-NLS-1$ //$NON-NLS-2$
			}
			writer.write("width=\""); //$NON-NLS-1$
			writer.write(Integer.toString(configuration.getAttribute(IJavaLaunchConfigurationConstants.ATTR_APPLET_WIDTH, 200))); //$NON-NLS-1$
			writer.write("\" height=\""); //$NON-NLS-1$
			writer.write(Integer.toString(configuration.getAttribute(IJavaLaunchConfigurationConstants.ATTR_APPLET_HEIGHT, 200))); //$NON-NLS-1$
			writer.write("\" >\n"); //$NON-NLS-1$
			Map parameters = configuration.getAttribute(IJavaLaunchConfigurationConstants.ATTR_APPLET_PARAMETERS, new HashMap());
			if (parameters.size() != 0) {
				Iterator iterator= parameters.entrySet().iterator();
				while(iterator.hasNext()) {
		 			Map.Entry next = (Map.Entry) iterator.next();
					writer.write("<param name="); //$NON-NLS-1$
					writer.write(getQuotedString((String)next.getKey()));
					writer.write(" value="); //$NON-NLS-1$
					writer.write(getQuotedString((String)next.getValue()));
					writer.write(">\n"); //$NON-NLS-1$
				}
			}
			writer.write("</applet>\n"); //$NON-NLS-1$
			writer.write("</body>\n"); //$NON-NLS-1$
			writer.write("</html>\n"); //$NON-NLS-1$
		} catch(IOException e) {
		} catch(CoreException e) {
		} finally {
			if (writer != null) {
				try {
					writer.close();
				} catch(IOException e) {
				}
			}
		}
		if (tempFile == null) {
			return null;
		}
		return tempFile;
	}
	
	
	private String getQuotedString(String string) {
		if (string.indexOf('"') == -1) {
			return '"' + string + '"';
		} else {
			return '\'' + string + '\'';
		}
	}
	
	
	/**
	 * Returns the system property string for the policy file
	 * 
	 * @param workingDir the working directory
	 * @return system property for the policy file
	 */
	public String getJavaPolicyFile(File workingDir) {
			File file = new File(workingDir, "java.policy.applet");//$NON-NLS-1$ 
			if (!file.exists()) {
				// copy it to the working directory
				File test = LauncherUtility.getFileInPlugin(new Path("java.policy.applet")); //$NON-NLS-1$
				try {
					byte[] bytes = getFileByteContent(test);
					BufferedOutputStream outputStream = new BufferedOutputStream(new FileOutputStream(file));
					outputStream.write(bytes);
					outputStream.close();
				} catch (IOException e) {
					return "";//$NON-NLS-1$
				}
			}
		return "-Djava.security.policy=java.policy.applet";//$NON-NLS-1$
	}
	
	/**
	 * Returns the contents of the given file as a byte array.
	 * @throws IOException if a problem occured reading the file.
	 */
	protected static byte[] getFileByteContent(File file) throws IOException {
		InputStream stream = null;
		try {
			stream = new BufferedInputStream(new FileInputStream(file));
			return getInputStreamAsByteArray(stream, (int) file.length());
		} finally {
			if (stream != null) {
				try {
					stream.close();
				} catch (IOException e) {
				}
			}
		}
	}
	
	
	/**
	 * Returns the given input stream's contents as a byte array.
	 * If a length is specified (ie. if length != -1), only length bytes
	 * are returned. Otherwise all bytes in the stream are returned.
	 * Note this doesn't close the stream.
	 * @throws IOException if a problem occured reading the stream.
	 */
	protected static byte[] getInputStreamAsByteArray(InputStream stream, int length) throws IOException {
		byte[] contents;
		if (length == -1) {
			contents = new byte[0];
			int contentsLength = 0;
			int bytesRead = -1;
			do {
				int available = stream.available();

				// resize contents if needed
				if (contentsLength + available > contents.length) {
					System.arraycopy(
						contents,
						0,
						contents = new byte[contentsLength + available],
						0,
						contentsLength);
				}

				// read as many bytes as possible
				bytesRead = stream.read(contents, contentsLength, available);

				if (bytesRead > 0) {
					// remember length of contents
					contentsLength += bytesRead;
				}
			} while (bytesRead > 0);

			// resize contents if necessary
			if (contentsLength < contents.length) {
				System.arraycopy(
					contents,
					0,
					contents = new byte[contentsLength],
					0,
					contentsLength);
			}
		} else {
			contents = new byte[length];
			int len = 0;
			int readSize = 0;
			while ((readSize != -1) && (len != length)) {
				// See PR 1FMS89U
				// We record first the read size. In this case len is the actual read size.
				len += readSize;
				readSize = stream.read(contents, len, length - len);
			}
		}

		return contents;
	}

	protected ArrayList getLaunchTypeWorkspaceFilters(final ILaunchConfiguration config) throws CoreException {
		ArrayList result = super.getLaunchTypeWorkspaceFilters(config);
		String appletViewerClassName = config.getAttribute(IJavaLaunchConfigurationConstants.ATTR_APPLET_APPLETVIEWER_CLASS, IJavaLaunchConfigurationConstants.DEFAULT_APPLETVIEWER_CLASS);
		result.add(new FilterTableElement(appletViewerClassName+ "*", "*", TraceMessages.INCLUDE));
		return result;
	}
}
