/**********************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.tptp.trace.ui.internal.launcher.deleg.application;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.hyades.internal.execution.local.common.ActiveAgentListCommand;
import org.eclipse.hyades.internal.execution.local.common.AgentDetailsCommand;
import org.eclipse.hyades.internal.execution.local.common.CommandElement;
import org.eclipse.hyades.internal.execution.local.common.Constants;
import org.eclipse.hyades.internal.execution.local.common.ControlMessage;
import org.eclipse.hyades.internal.execution.local.common.ProcessExitedCommand;
import org.eclipse.hyades.internal.execution.local.common.QueryAgentDetailsCommand;
import org.eclipse.hyades.internal.execution.local.common.QueryAgentListCommand;
import org.eclipse.hyades.internal.execution.local.common.QueryProcessListCommand;
import org.eclipse.hyades.internal.execution.local.common.RegisteredProcessListCommand;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.AgentFactory;
import org.eclipse.hyades.internal.execution.local.control.AgentImpl;
import org.eclipse.hyades.internal.execution.local.control.CommandHandler;
import org.eclipse.hyades.internal.execution.local.control.Connection;
import org.eclipse.hyades.internal.execution.local.control.InactiveProcessException;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.NodeImpl;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.internal.execution.local.control.ProcessActiveException;
import org.eclipse.hyades.internal.execution.local.control.ProcessImpl;
import org.eclipse.hyades.internal.execution.local.control.ProcessListener;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.ProfileUIManager;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.util.PDCoreUtil;
import org.eclipse.osgi.util.NLS;
import org.eclipse.tptp.trace.ui.internal.launcher.application.AgentDiscovererConfiguration;
import org.eclipse.tptp.trace.ui.internal.launcher.application.IDiscovererLaunchConfConstants;
import org.eclipse.tptp.trace.ui.internal.launcher.application.AgentDiscovererConfiguration.AgentDiscovererFilterEntry;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherMessages;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;
 
public class AgentDiscovererDelegateHelper
{
	/* The agent name and type */
	public static String LAD_NAME = "Agent Discoverer";
	public static String LAD_TYPE = "org.eclipse.tptp.trace.ui.loggingAgentDiscoverer";
	
	
	public static Process createProcess(Node node)
	{
		/* We need to create a process of our own here because the RAC has the limitaion of
		 * not invoking processExit methods of registered agents if the process doesn't
		 * have any registered agents that the RAC is aware of. */
		Process process = new CustomProcess(node);
		if(node instanceof NodeImpl) 
		{
			process.addProcessListener(((NodeImpl)node).getProcessListener());
		}
		
		try 
		{
			process.setExecutable("java.exe");
		} 
		catch (ProcessActiveException e) 
		{
			/* This should never happen */
			e.printStackTrace();
		}
		
		return process;
	}
	
	
	public static void registerProcessListener(final Process process, final TRCProcessProxy trcProcessProxy, final DynamicAgentDiscoverer dynamicDiscoverer, final ILoggingAgentDiscovererDelegate logAgentDiscoverer)
	{
		try
		{		
			/* Register a listener that will begin the process of the agent discovery 
			 * after the process */
			process.addProcessListener(new ProcessListener(){				

				public void processLaunched(Process process)
				{				
					dynamicDiscoverer.start();
					
					/* The agent should become active as soon as the process is launched.
					 * Simulate an active event */
					logAgentDiscoverer.activateAgent(trcProcessProxy);
				}

				public void processExited(Process process)
				{
					dynamicDiscoverer.terminate();
				}

			});
						 
		} 
		catch (Exception e)
		{
			LauncherUtility.openErrorWithDetail(LauncherMessages.LAUNCHER_COMMON_ERROR_TITLE, 
					LauncherMessages.ERROR_LOG_DISCOVER_INITIALIZE, e);
		}		
	}
	
	
	public static void activateAgent(TRCProcessProxy processProxy, TRCAgentProxy agentProxy)
	{
		/* Set the attributes of the agent */
		agentProxy.setAttached(true);
		agentProxy.setActive(true);
		agentProxy.setMonitored(true);
		
		/* Update the UI to indicate a refresh */
		LauncherUtility.sendProfileEvent(ProfileEvent.START_MONITOR, agentProxy);
	}
	
	
	public interface ILoggingAgentDiscovererDelegate
	{
		public void activateAgent(TRCProcessProxy processProxy);
	}
	
	
	/**
	 * This class is used to dynamically discover the logging agents registered with a specific
	 * process.
	 * 
	 * @author Ali Mehregani
	 */
	public static class DynamicAgentDiscoverer implements Runnable
	{
		/* An incremental wait time */
		private static final int INCREMENTAL_WAIT = 100;
		
		/* Total incremental waits */
		private static final int TOTAL_INC_WAIT = 34;
		
		/* Represents the process to be launched */
		private Process process;
		
		/* The process model entity */
		private TRCProcessProxy trcProcessProxy;
		
		/* The agent model entity */
		private TRCAgentProxy trcAgentProxy;
		
		/* The process id - available for convenience (value should be the same as process.getProcessId() */
		private long processId;
				
		/* Indicates whether thid discoverer should be enable or not */
		private boolean enabled;

		/* The list of agents that have caused an error and should be ignored when querying for agents 
		 * KEY = agent name + agent type
		 * Value = Boolean.TRUE */		
		private Hashtable agentBlackList;
		
		/* The command handler that will handle query commands */
		private QueryCommandHandler queryCommandHandler;

		/* Points to the last agent list queried 
		 * KEY = "agent name"
		 * Value = Boolean.TRUE */
		private Hashtable lastAgentList;
		
		/* The frequency at which the discoverer will query the process for its available agents */
		private long frequency;

		/* The filteration in place */
		private AgentDiscovererFilterEntry[] filters;
		
		/* Indicates if a message has been processed */
		private boolean messageProcessed;

		
		/**
		 * Constructor
		 * @param process The process to be launched
		 * @param trcProcessProxy The process model entity 
		 * @throws org.eclipse.tptp.platform.execution.exceptions.InactiveProcessException 
		 */
		public DynamicAgentDiscoverer(ILaunchConfiguration conf, Process process, TRCProcessProxy trcProcessProxy)
		{
			this.process = process;
			this.trcProcessProxy = trcProcessProxy;
			queryCommandHandler = new QueryCommandHandler();
			
			/* Do the initialization based on the configuration */
			frequency = -1;
			filters = null; 
			try
			{
				frequency = conf.getAttribute(IDiscovererLaunchConfConstants.ATTR_OPT_FREQUENCY, AgentDiscovererConfiguration.DEFAULT_FREQUENCY);
				filters = AgentDiscovererConfiguration.unserializeFilters(conf.getAttribute(IDiscovererLaunchConfConstants.ATTR_FILTER, AgentDiscovererConfiguration.DEFAULT_FILTERS));
			} 
			catch (CoreException e)
			{
				if (frequency == -1)
					frequency = AgentDiscovererConfiguration.DEFAULT_FREQUENCY;
				
				if (filters == null)
					filters = AgentDiscovererConfiguration.unserializeFilters(AgentDiscovererConfiguration.DEFAULT_FILTERS);
			}
						
		}

		public void setAgentProxy(TRCAgentProxy trcAgentProxy)
		{
			this.trcAgentProxy = trcAgentProxy;
		}
		
		/**
		 * Terminate the operation.
		 */
		public void terminate()
		{
			enabled = false;
			
			if (trcAgentProxy != null)
			{
				trcAgentProxy.setActive(false);
				trcAgentProxy.getProcessProxy().setActive(false);
				LauncherUtility.sendProfileEvent(ProfileEvent.TERMINATE, trcAgentProxy);
			}
				
			
			synchronized (DynamicAgentDiscoverer.this)
			{
				DynamicAgentDiscoverer.this.notify();
			}
		}


		/**
		 * Start the operation of discovering and automatically attaching
		 * to agents.
		 */
		public void start()
		{
			try
			{
				processId = Long.parseLong(process.getProcessId());
			} 
			catch (InactiveProcessException e)
			{
				LauncherUtility.openErrorWithDetail(LauncherMessages.LAUNCHER_COMMON_ERROR_TITLE, 
						LauncherMessages.ERROR_LOG_DISCOVER_INITIALIZE, e);
				return;
			}
			
			enabled = true;
			new Thread(this, "Query Agents").start();
		}


		/**
		 * Begin the process of periodically monitoring the agents registered
		 */
		public void run()
		{
			/* The agent discovery loop -- a query of agents is made every
			 * x amount of time (where x is specified in the configuration) */
			while (enabled)
			{
				try
				{
					/* Use a filtered list of agents that belong to the process that this launch
					 * delegate is associated with in order to attach to the logging agents */
					if (trcAgentProxy != null && trcAgentProxy.isMonitored())
						discoverLogAgents();
			
					synchronized (this)
					{
						wait(frequency);
					}
				} 
				catch (Throwable t)
				{
					/* Bail out */
					LauncherUtility.openErrorWithDetail("", t.getMessage(), t);
					terminate();
					break;
				}
			}
						
		}


		/**
		 * Retrieve a list of logging agents that are associated with the process
		 * that this launch delegate is associated with.  Only logging agents that
		 * are not yet attached to are returned.
		 *  
		 * @return
		 */
		private void discoverLogAgents()
		{
			try
			{
				/* If the process has died, then we should send out an exit command that
				 * will then invoke all process listeners */
				if (!isProcessAlive())
				{
					terminate();
					
					if (process instanceof CustomProcess)
					{
						ProcessExitedCommand pec = new ProcessExitedCommand();
						pec.setProcessId(processId);
						((CustomProcess)process).handleCommand(pec);
					}
					return;
				}
				
				List availableAgents = queryAvailableAgents();
				if (availableAgents == null)
					return;
				
				Agent currentAgent;
				for (int i = 0, agentSize = availableAgents.size(); i < agentSize; i++)
				{
					currentAgent = (Agent)availableAgents.get(i);
					
					/* Add the agent only if:
					 * 1) It's not of type agent discoverer
					 * 2) We still haven't attached to it
					 * 3) It's not excluded by the filteration */
					if (!currentAgent.getType().equals(LAD_TYPE) && 
						!currentAgent.isAttached() && agentIsIncluded(currentAgent))
					{
						try
						{								
							final TRCAgentProxy loggingAgent = PDCoreUtil.attachToAgent(trcProcessProxy, currentAgent);																								
							PDCoreUtil.doAttach(currentAgent, loggingAgent);
							
							UIPlugin.getDefault().getWorkbench().getDisplay().syncExec(new Runnable()
							{
								public void run()
								{
									/* Start monitoring the logging agent immediately */
									ProfileUIManager.getInstance().startMonitoring(loggingAgent);
								}

							});
						} 
						catch (Exception e)
						{
							LauncherUtility.openErrorWithDetail(LauncherMessages.LAUNCHER_COMMON_ERROR_TITLE, 
									NLS.bind(LauncherMessages.ERROR_LOG_DISCOVER_ATTACH, currentAgent.getName()), e);
							
							addToBlackList(currentAgent);
						} 
					}
				}				
			} 
			catch (Exception e1)
			{
				LauncherUtility.openErrorWithDetail(LauncherMessages.LAUNCHER_COMMON_ERROR_TITLE, 
						LauncherMessages.ERROR_LOG_DISCOVER_UNEXPECTED, e1);
				
				terminate();
			} 
		}


		private void addToBlackList(Agent currentAgent)
		{
			if (agentBlackList == null)
				agentBlackList = new Hashtable();
			
			agentBlackList.put(currentAgent.getName() + currentAgent.getType(), Boolean.TRUE);
		}


		private boolean agentIsIncluded(Agent currentAgent)
		{
			String agentName = currentAgent.getName(), agentType = currentAgent.getType();
			
			/* Is agent listed under the black list? */
			if (agentBlackList != null && agentBlackList.get(agentName + agentType) != null)
				return false;
			
			/* Is agent excluded by the filter set */
			boolean included = false;
			for (int i = 0; i < filters.length; i++)
			{
				int match = filters[i].match(agentName);
				
				included = match == AgentDiscovererFilterEntry.INCLUDE;
				if (included || match == AgentDiscovererFilterEntry.EXCLUDE)
					break;
			}
				
			/* If agent is excluded, then add it to the black list so that we don't
			 * step through the filters again */
			if (!included)
				addToBlackList(currentAgent);
			
			return included;
		}
		
		
		/**
		 * Returns true if the process associated with this agent is still alive; otherwise
		 * false is returned
		 * 
		 * @return A flag indicating the process status
		 * @throws InactiveProcessException 
		 * @throws IOException 
		 */
		private boolean isProcessAlive() throws IOException, InactiveProcessException
		{
			ControlMessage message = new ControlMessage();
			QueryProcessListCommand qplCommand = new QueryProcessListCommand();
			
			message.appendCommand(qplCommand);
			sendMessage(process.getNode().getConnection(), message);
			
			if (!messageProcessed)
				return false;
			
			long[] pids = queryCommandHandler.getAvailableProcesses();
			if (pids == null)
				return false;
			
			for (int i = 0; i < pids.length; i++) 
			{
				if (pids[i] == processId)
					return true;
			}
			return false;
		}
		
		
		/**
		 * Returns a list of agents that are available under the process associated with
		 * this launch.
		 * @return Available list of the associated process.
		 * @throws InactiveProcessException 
		 * @throws IOException 
		 */
		private List queryAvailableAgents() throws InactiveProcessException, IOException
		{		
			/* Get the connection and create the query agent message */
			Connection connection = process.getNode().getConnection();
			ControlMessage message = new ControlMessage();
			QueryAgentListCommand qalCommand = new QueryAgentListCommand();
			qalCommand.setProcessId(processId);
			message.appendCommand(qalCommand);
			
			/* Send out the message */	
			sendMessage (connection, message);			
			
			/* The message didn't get processed for some reason */
			if (!messageProcessed)
				return null;
							
			/* Find out the details of the agents */			
			Hashtable eAgents = queryCommandHandler.getAgentsMetaInfo();
			
			/* Bail out quickly if there are no observable changes */
			if (eAgents.size() == 0 || listEqual(lastAgentList, eAgents))
				return null;
			
			lastAgentList = eAgents;
			
			/* Find the details of the agents */
			queryCommandHandler.getAvailableAgents().clear();
			for (Enumeration availableAgents = eAgents.keys(); availableAgents.hasMoreElements();)
			{
				String agentName = (String)availableAgents.nextElement();

				/* We only care about logging agents */
				if (agentName.equals(LAD_NAME))
					continue;
				
				/* Query for the details of the agent */
				message = new ControlMessage();
				QueryAgentDetailsCommand qadCommand = new QueryAgentDetailsCommand();
				qadCommand.setProcessId(processId);
				qadCommand.setAgentName(agentName);
				message.appendCommand(qadCommand);
				sendMessage (connection, message);
			}
			
			/* The message didn't get processed for some reason */
			if (!messageProcessed)
				return null;
			
			return queryCommandHandler.getAvailableAgents();
		}
		
		
		
		/**
		 * Returns true if the agent lists are the same
		 * 
		 * @param agentList1
		 * @param agentList2
		 * @return True if agent list 1 is the same as agent list 2.  The name
		 * field of an agent is used to do the comparison.
		 */
		private boolean listEqual(Hashtable agentList1, Hashtable agentList2)
		{
			if ((agentList1 == null && agentList2 != null) || (agentList1 != null && agentList2 == null))
				return false;
			
			int agentListSize1 = agentList1.size(), agentListSize2 = agentList2.size();
			if (agentListSize1 != agentListSize2)
				return false;
			
			String currentAgentName;
			for (Enumeration agentNames = agentList1.keys(); agentNames.hasMoreElements();)
			{
				currentAgentName = (String)agentNames.nextElement();
				if (agentList2.get(currentAgentName) == null)
					return false;
			}
			
			return true;
		}


		private void sendMessage(Connection connection, ControlMessage message) throws IOException
		{
			messageProcessed = false;
			connection.sendMessage(message, queryCommandHandler);
			int waitCounter = 0;
			
			while (waitCounter < TOTAL_INC_WAIT && !messageProcessed)
			{
				synchronized(this)
				{
					try
					{
						waitCounter++;
						this.wait(INCREMENTAL_WAIT);
					}
					catch (Exception e)
					{
						return;
					}
				}
			}
		}



		public class QueryCommandHandler implements CommandHandler 
		{
			/* Keeps track of a list of meta data for agents 
			 * KEY = agent name
			 * VALUE = Boolean.TRUE */
			private Hashtable agentMetaData; 
		
			/* Stores the available agents */
			private List availableAgents;
			
			/* Stores the list of processes that are currently available */
			private long[] availableProcesses;
			
			public QueryCommandHandler()
			{
				agentMetaData = new Hashtable();
				availableAgents = new ArrayList();				
			}
			
			
			public void incommingCommand(Node node, CommandElement command) 
			{
				int tag = (int) command.getTag();
				long processId;
				switch (tag) 
				{			
					/* Queries for meta data of agents (just the names) */
					case (int) Constants.RA_PROCESS_LIST:
												
						RegisteredProcessListCommand rplCmd = (RegisteredProcessListCommand) command;
						availableProcesses = rplCmd.getProcessList();
						
						break;

					/* Queries for meta data of agents (just the names) */
					case (int) Constants.RA_AGENT_LIST: 
						
						agentMetaData.clear();
						ActiveAgentListCommand aalCmd = (ActiveAgentListCommand) (command);
					
						processId = aalCmd.getProcessId();						
						if (processId != DynamicAgentDiscoverer.this.processId)
							break;
						
						String[] agentNames = aalCmd.getAgents();
												
						if (agentNames != null) 
						{
							for (int i = 0; i < agentNames.length; i++) 
								agentMetaData.put(agentNames[i], Boolean.TRUE);							
						}
						
						break;

					/* Queiries detailed information on agents */ 
					case (int) Constants.RA_AGENT_DETAILS: 
												
						AgentDetailsCommand adCmd = (AgentDetailsCommand) command;
	
						processId = adCmd.getProcessId();						
						if (processId != DynamicAgentDiscoverer.this.processId)
							break;
						
						String agentName = adCmd.getAgentName();
						String agentType = adCmd.getAgentType();
						String agentUUID = adCmd.getAgentUUID();
	
												
						Agent ag = process.getAgent(agentName);
						
						/* Create and associate the agent with the process if it is not yet there */
						if (ag == null) 
						{
							ag = AgentFactory.createAgent(process, agentName);
							ag.setActive(true);
							
							ag.setType(agentType);
							ag.setUUID(agentUUID);
							ag.addAgentListener((ProcessImpl) process);						
						}
						availableAgents.add(ag);
						
						break;
					}
				
				messageProcessed = true;
			}

			public long[] getAvailableProcesses()
			{
				return availableProcesses;
			}
			
			public Hashtable getAgentsMetaInfo()
			{
				return (Hashtable)agentMetaData.clone();
			}
			

			public List getAvailableAgents()
			{
				return availableAgents;
			}
		}
	}
	
	
	public static class CustomProcess extends ProcessImpl
	{

		public CustomProcess(Node node) 
		{
			super(node);			
		}
		
		
		/**
		 * Increase the visibility of the handle command
		 * method
		 */
		public void handleCommand(CommandElement command)
		{
			super.handleCommand(command);
		}
	}
	
	
	/**
	 * This dummy class is used to fake an execution agent.
	 */
	public static class DummyAgent extends AgentImpl
	{
		public DummyAgent(Process process) 
		{
			super(process, AgentDiscovererDelegateHelper.LAD_NAME);
			setType(AgentDiscovererDelegateHelper.LAD_TYPE);
		}
	}
}
