/**********************************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.tptp.trace.ui.internal.launcher.deleg.application;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.List;
import java.util.Vector;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.hyades.internal.execution.local.common.CustomCommand;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.AgentConfigurationEntry;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.internal.execution.local.control.ProcessListener;
import org.eclipse.hyades.internal.execution.local.control.Variable;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.launcher.IExtendedProfilingType;
import org.eclipse.hyades.trace.ui.internal.launcher.IProfileLaunchConfigurationConstants;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfileLaunchUtil;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfilingSetType;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfilingSetsManager;
import org.eclipse.hyades.trace.ui.internal.piclient.PIProcessListener;
import org.eclipse.hyades.trace.ui.internal.util.PDCoreUtil;
import org.eclipse.hyades.trace.ui.launcher.IProfilingType;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIConstants;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIPlugin;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceConstants;
import org.eclipse.tptp.trace.ui.internal.launcher.core.DataCollector;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherConstants;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;
import org.eclipse.tptp.trace.ui.provisional.launcher.ProcessParameters;

public class PIDelegateHelper 
{
 
	public static PIProcessListener configureProcess(Process process, TRCProcessProxy trcProcessProxy, ILaunchConfiguration conf, String classpath) throws CoreException 
	{		
		PIProcessListener listener = new PIProcessListener(trcProcessProxy.getNode(), trcProcessProxy);
		listener.monitor(true);
		process.addProcessListener(listener);
        listener.setLaunchMode(1);
        listener.setAutoMonitoring(conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_AUTO_MONITORING, true));
		listener.getProcessVariableList().add(new Variable("UICLASSPATH", classpath));
		PDCoreUtil.setEnvironmentVariable(process, ProfileLaunchUtil.getEnvironmentVariables(conf), listener);
		
		return listener;
	}
	
	
	public static Agent createAgent(ILaunchConfiguration conf, Process process, TRCProcessProxy trcProcessProxy, String agentName, String agentType, String profileFile, PIProcessListener listener) throws CoreException
	{
		ProfilingSetsManager manager = ProfilingSetsManager.instance();
		Vector options = manager.getOptions(conf);
		Agent agent = PDCoreUtil.createAgent(process, manager.getFilters(conf), options, trcProcessProxy, agentName, agentType);
	    if (profileFile != null) 
	    {
	    	agent.setProfileFile(profileFile);
	    }
	    agent.addAgentListener(listener);
	    listener.addAgent(agent);
	    
	    
	    AgentConfigurationEntry[] confEntries = PeriodicPoll.getOption(options, new String[] {
	    		IProfileLaunchConfigurationConstants.ATTR_SHOW_EXECUTION_FLOW,
	    		IProfileLaunchConfigurationConstants.ATTR_EXEC_POLLING_FREQ_MODE,
	    		IProfileLaunchConfigurationConstants.ATTR_EXEC_POLLING_FREQ});
	    String[] pollingOptions = new String[3];
	    for (int i = 0; i < confEntries.length; i++)
		{
	    	pollingOptions[i] = confEntries[i] == null ? null : confEntries[i].getValue();
		}
	    
	    PeriodicPoll periodicPoll = PeriodicPoll.createInstance(pollingOptions, trcProcessProxy, listener, agent);
	    if (periodicPoll != null)
	    	periodicPoll.start();
	   
		return agent;
	}


	public static ProcessParameters getProcessParameter(ProcessParameters processParameters, ILaunchConfiguration conf)
	{		
		processParameters.addExecutableParameter(PDCoreUtil.getPIVMArgument(PDCoreUtil.PI_MODE_CONTROLLED));
		
		/* Unserialize the launch configuration. Key: data collector objects. Value: List of either analysis types or profiling set types */
		Hashtable hashTable = LauncherUtility.unserializeSelection(conf);
		if (hashTable == null) {
			return processParameters;
		}
		
		Enumeration dataCollectorKeys = hashTable.keys();
		
		/* Step through the data collectors and check for the Probe Insertion profiling set */
		while (dataCollectorKeys.hasMoreElements())
		{			
			DataCollector dataCollector = (DataCollector)dataCollectorKeys.nextElement();
			
			/* Java Profiling data collector */
			if (dataCollector.getId().equals(LauncherConstants.JVMPI_DATA_COLLECTOR_ID))
			{
				/* List of either analysis types or profiling set types */
				List valueList = (List) hashTable.get(dataCollector);
				for (int i=0; i < valueList.size(); i++)
				{
					/* Is this a profiling set type? */
					if (valueList.get(i) instanceof ProfilingSetType)
					{
						ProfilingSetType profilingSetType = (ProfilingSetType) valueList.get(i);
						
						/* Is this the Probe Insertion profiling set type? */
						if (profilingSetType.getId().equals("org.eclipse.tptp.probekit.ui.ProbekitProfilingType"))
						{ 
							/* Get the profiling type */
							IProfilingType profilingType = profilingSetType.getProfilingType();
							
							/* This check isn't really necessary, since we know it's the Probe Insertion profiling set type.
							 * Just being done for correctness */
							if (profilingType instanceof IExtendedProfilingType)
							{
								/* Add the extra VM arguments specified by this profiling type to processParameter */
								processParameters.addExecutableParameter(((IExtendedProfilingType)profilingType).getVMArguments(conf));								
							}							
						}						
					}
				}
			}
			
		}
		
		return processParameters;
	}


	
	/**
	 * The following class is used by profiling sessions that include the execution compressed option.
	 * The class will perform periodic polls based on an automatic or a manual frequency poll period.
	 * In automatic mode, the polling frequency period is defined as a function of incomming events.
	 * If the incomming events exceed a threashold, then the frequency period is increased.  Similarly
	 * the period is decreased if it falls under the threshold.  Given an application with a long life
	 * span, the frequency period should eventually converge. <br>
	 * The formula used to calculate the polling frequency in automated mode is:
	 * <pre>
	 * f(x) = F + ceil(((T-x)/T) * I), where 
	 * F = current frequency period, 
	 * x = amount of incomming data, 
	 * T = threshold for the incomming data.
	 * I = incremental unit. 
	 * </pre>
	 * 
	 * This formula guarantees that the incremental/decremental value will be bounded by the interval: [0, I].  The 
	 * frequency period is forced to always be bound between [1, 120] seconds.
	 * @author Ali Mehregani
	 */
	public static class PeriodicPoll extends Thread
	{
		/** The incomming events threshold: 3KB */
		private static final int INCOMING_EVENTS_THRESHOLD = 3072;
		
		/** The incremental unit */
		private static final int INCREMENTAL_UNIT = 10;
		
		/** The poll frequency (in seconds) */
		private double pollFrequency;
		
		/** The static poll frequency */
		private boolean isAutomated;
		
		/** The process proxy */
		private TRCProcessProxy processProxy;
		
		/** The associated agent */
		private Agent agent;
		
		/** The process/agent listener of the agent */
		private PIProcessListener listener;
		
		/** Indicates when the process has been terminated */
		private boolean isProcessTerminated;
		
		
		/**
		 * Use this method to create an instance of this class.  If the class returns a null value, then 
		 * the current profile session does not require regular polling
		 * 
		 * @param options The options of the profiling session
		 * @param trcProcessProxy The process proxy
		 * @param listener The process/agent listener 
		 * @param agent The execution agent
		 * 
		 * @return An instance of this class iff the profiling session requires regular polling; otherwise
		 * null is returned.
		 */
		public static PeriodicPoll createInstance (String[] options, TRCProcessProxy trcProcessProxy, PIProcessListener listener, Agent agent)
		{	    
		   
		    /* Is this trace a compressed execution? */
		    if (options[0] != null && "false".equalsIgnoreCase(options[0]))
		    {
		    	boolean isPollAutomatic = options[1] == null || "true".equalsIgnoreCase(options[1]);
		    	
		    	/* Automatic poll */
		    	if (isPollAutomatic)
		    	{
		    		return new PeriodicPoll(trcProcessProxy, listener, agent);
		    	}
		    	
		    	/* Manually set poll */
		    	else
		    	{
		    		int pollingFrequency = options[2] == null ? 10 : Integer.parseInt(options[2]);
		    		return new PeriodicPoll(trcProcessProxy, listener, agent, pollingFrequency);
		    	}
		    }
		    
		    return null;		
		}
		
		/**
		 * Make the constructor invisible
		 * 
		 * @param processProxy The process proxy
		 * @param listener The process/agent listener 
		 * @param agent The execution agent
		 */
		private PeriodicPoll(TRCProcessProxy processProxy, PIProcessListener listener, Agent agent)
		{
			this(processProxy, listener, agent, 10);
			isAutomated = true;
		}
		
		
		/**
		 * Make the constructor invisible
		 * 
		 * @param processProxy The process proxy
		 * @param listener The process/agent listener 
		 * @param agent The execution agent
		 * @param pollingFrequency The initial polling frequency
		 */
		private PeriodicPoll(TRCProcessProxy processProxy, PIProcessListener listener, Agent agent, int pollingFrequency)
		{			
			this.processProxy = processProxy;
			this.listener = listener;	
			this.agent = agent;
			this.pollFrequency = pollingFrequency;
			isAutomated = false;
			
			
			/* Register a listener so we can terminate the periodic poll thread when the
			 * process terminates */
			agent.getProcess().addProcessListener(new ProcessListener(){

				public void processExited(Process process) 
				{
					isProcessTerminated = true;					
				}

				public void processLaunched(Process process) 
				{					
				}});
		}
		
		
		public void run()
		{		
			boolean isMonitored;
			
			/* The polling loop */
			while (true)
			{
				if (isProcessTerminated || agent == null)
					break;
				
				isMonitored = agent.isMonitored();
				if (isMonitored)
				{					
					try 
					{
						TRCNode trcNode = processProxy.getNode();
						if (trcNode == null)
							break;
						
						Node node = PDCoreUtil.profileConnect(trcNode.getName(), String.valueOf(trcNode.getPort()));
						if (node == null) 
							break;
						
						CustomCommand command = new CustomCommand();
						command.setData("COLLECTDATA");
						agent.invokeCustomCommand(command);

					}					
					catch (Exception exc) 
					{
						exc.printStackTrace();
						break;
					}
				}
				
				try 
				{
					Thread.sleep((long)(Math.ceil(pollFrequency) * 1000));
				} 
				catch (InterruptedException e) 
				{					
				}
				
				/* Recalculate the polling frequency, if it is set to automatic */
				if (isMonitored && isAutomated && listener.getDataProcessor() != null)
				{					
					int lastEventSize = listener.getDataProcessor().getSizeOfLastEvent();
					
					if (lastEventSize > 0)
					{			
						pollFrequency = pollFrequency + (Math.ceil((((double)(INCOMING_EVENTS_THRESHOLD - lastEventSize))/((double)INCOMING_EVENTS_THRESHOLD)) * INCREMENTAL_UNIT));						
						
						/* Bound the frequency period to [1, 120] seconds */
						if (pollFrequency < 1)
							pollFrequency = 1;
						else if (pollFrequency > 120)
							pollFrequency = 120;
											
						listener.getDataProcessor().resetStats();
					}
					
				}
			}
		}
		
		public static AgentConfigurationEntry[] getOption(Vector options, String[] optionKeys) 
		{
			AgentConfigurationEntry[] confEntries = new AgentConfigurationEntry[optionKeys.length];
			Hashtable optionKeysHash = new Hashtable();
			for (int i = 0; i < optionKeys.length; optionKeysHash.put(optionKeys[i++], new Integer(i - 1)));
			
			for (int i = 0, optionsCount = options.size(); i < optionsCount; i++) 
			{
				AgentConfigurationEntry currentConfEntry = (AgentConfigurationEntry)options.get(i);
				Integer inx;
				if ((inx = (Integer)optionKeysHash.get(currentConfEntry.getName())) != null)
					confEntries[inx.intValue()] = currentConfEntry;
			}
			
			return confEntries;
		}
	}

	private static IJVMVersionDetector jvmVersionDetector;
	
	public static void setJvmVersionDetector(IJVMVersionDetector jvmVersionDetector) {
		PIDelegateHelper.jvmVersionDetector = jvmVersionDetector;
	}
	
	public static class JVMVersionDetector {
		private String host;
		private int port;

		public JVMVersionDetector (final ILaunchConfiguration configuration) {
			UIPlugin.getDefault().getWorkbench().getDisplay().syncExec(new Runnable() {
				public void run() {
					try {	
						JVMVersionDetector.this.host = configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_HOSTNAME, CommonUITraceConstants.LOCAL_HOST);
						JVMVersionDetector.this.port = configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PORT, CommonUIPlugin.getDefault().getPreferenceStore().getDefaultInt(CommonUIConstants.LOCALHOST_PORT));		
					} 
					catch (CoreException e)	{
						UIPlugin.getDefault().log(e);
					}
				}
			});
					
		}
		
		public JVMVersionDetector (String host, int port) {
			this.host = host;
			this.port = port;
		}
		
		public String retrieveVersionOutput() {
			if (jvmVersionDetector != null) { 
				String ver = jvmVersionDetector.getJVMVersion(host, port);
				if (ver != null && ver.length() > 0) return ver;
			}

			return DefaultJVMVersionDetector.getInstance().getJVMVersion(host, port);
		}
	}
}
