/**********************************************************************
 * Copyright (c) 2006, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.trace.ui.provisional.control.provider;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.hyades.trace.ui.internal.core.TraceUIImages;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceConstants;
import org.eclipse.tptp.trace.ui.internal.control.provider.application.ControlMessages;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;



/**
 * An abstract class that should be subclassed by a control provider for an agent.  
 * This class provides the standard control items for the states that an agent can
 * be in:
 * <ul>
 * 	<li> attached </li>
 *  <li> detached </li>
 *  <li> paused </li>
 *  <li> resumed/start monitoring </li>
 * </ul>
 *  
 * The semantics of each state is further described under the create methods for 
 * each of the four control items.
 *  
 * @author Ali Mehregani
 */
public abstract class AbstractAgentControlProvider extends ProcessControlProvider 
{
	/** The id of the control items contributed */
	public static final String ATTACH_ITEM = "org.eclipse.hyades.trace.ui.internal.popupMenu.AttachAction";
	public static final String DETACH_ITEM = "org.eclipse.hyades.trace.ui.internal.popupMenu.DetachAction";
	public static final String START_MONITORING = "org.eclipse.hyades.trace.ui.internal.monitorsview.popupMenu.startMonitoring";
	public static final String PAUSE_MONITORING = "org.eclipse.hyades.trace.ui.internal.popupMenu.stopMonitoring";
	
	
	/** The attach control item */
	private IControlItem attachItem;
	
	/** The detach control item */
	private IControlItem detachItem;
	
	/** The resume control item */
	private IControlItem resumeItem;
	
	/** The pause control item */
	private IControlItem pauseItem;
	
	/**
	 * @see org.eclipse.tptp.trace.ui.provisional.control.provider.AbstractLogicalItem.initializeControlItems()
	 */
    protected void initializeControlItems()
	{
    	super.initializeControlItems();
    	addControlItem (createAttachControlItem());
    	addControlItem (createDetachControlItem());
    	addControlItem (createStartMonitoringControlItem());
    	addControlItem (createPauseMonitoringControlItem());  
	}
    

	/**
	 * Subclasses are expected to provide an agent state modifier that 
	 * can be used to alter the state of an agent.
	 *  
	 * @return The agent state modifier
	 */
    abstract public IAgentStateModifier getAgentStateModifier();
    
    
    /**
     * Overwrite this method to alter the attach control
     * item.
     * 
     * @return The attach control item
     */
    protected IControlItem createAttachControlItem()
    {
    	if (attachItem != null)
    		return attachItem;
    	
    	class AttachControlItem extends AgentControlItem
    	{		
    		public AttachControlItem()
    		{
    			super(getAgentStateModifier(), ATTACH_ITEM, CommonUITraceConstants.PROFILE_ATTACH_GROUP, ControlMessages.CONTROL_ITEM_ATTACH,  TraceUIImages.INSTANCE.getImageDescriptor("c", TraceUIImages.IMG_DETACH_AGENT));
    			setDisabledImageDescriptor(TraceUIImages.INSTANCE.getImageDescriptor("d", TraceUIImages.IMG_ATTACH_AGENT));
    		}
    		
			public void run()
			{
				IAgentStateModifier agentStateModifier = getAgentStateModifier();
											
				try
				{
					/* attach to the agent */					
					agentStateModifier.attach();
				} catch (CoreException e)
				{
					/* Display status */
					LauncherUtility.openMessageWithDetail(e.getStatus().getSeverity(), "", e.getMessage(), e);
				}				
			}
			
			public boolean isEnabled()
			{
				return getAgentStateModifier().canAttach();
			}		
    	}
    	
    	
    	attachItem = new AttachControlItem();
    	return attachItem;
    }
    
    protected IControlItem createDetachControlItem()
    {
    	if (detachItem != null)
    		return detachItem;
    	
    	class DetachControlItem extends AgentControlItem
    	{		
    		public DetachControlItem()
    		{
    			super(getAgentStateModifier(), DETACH_ITEM, CommonUITraceConstants.PROFILE_ATTACH_GROUP, ControlMessages.CONTROL_ITEM_DETACH,  TraceUIImages.INSTANCE.getImageDescriptor("c", TraceUIImages.IMG_DETACH_AGENT));
    			setDisabledImageDescriptor(TraceUIImages.INSTANCE.getImageDescriptor("d", TraceUIImages.IMG_DETACH_AGENT));
    		}
    		
			public void run()
			{
				IAgentStateModifier agentStateModifier = getAgentStateModifier();
											
				try
				{
					/* detach the agent */					
					agentStateModifier.detach();
				} catch (CoreException e)
				{
					/* Display status */
					LauncherUtility.openMessageWithDetail(e.getStatus().getSeverity(), "", e.getMessage(), e);
				}				
			}
			
			public boolean isEnabled()
			{
				return getAgentStateModifier().canDetach();
			}		
    	}
    	
    	
    	detachItem = new DetachControlItem();
    	return detachItem;
    }
    
    protected IControlItem createStartMonitoringControlItem()
    {
    	if (resumeItem != null)
    		return resumeItem;
    	
    	class ResumeControlItem extends AgentControlItem
    	{		
    		public ResumeControlItem()
    		{
    			super(getAgentStateModifier(), START_MONITORING, CommonUITraceConstants.PROFILE_MONITOR_GROUP, ControlMessages.CONTROL_ITEM_RESUME, TraceUIImages.INSTANCE.getImageDescriptor("c", TraceUIImages.IMG_RESUME));
    			setDisabledImageDescriptor(TraceUIImages.INSTANCE.getImageDescriptor("d", TraceUIImages.IMG_RESUME));
    		}
    		
			public void run()
			{
				IAgentStateModifier agentStateModifier = getAgentStateModifier();
				
				if (agentStateModifier.canDetach()) {
					try
					{
						/* Resume the agent */
						agentStateModifier.startMonitoring();
					} catch (CoreException e)
					{
						/* Display status */
						LauncherUtility.openMessageWithDetail(e.getStatus().getSeverity(), "", e.getMessage(), e);
					}
					
				} else {
					try
					{
						/* attach to the agent */					
						agentStateModifier.attach();
					} catch (CoreException e)
					{
						/* Display status */
						LauncherUtility.openMessageWithDetail(e.getStatus().getSeverity(), "", e.getMessage(), e);
					}	
				}
			}
			
			public boolean isEnabled()
			{
				if (getAgentStateModifier().canDetach()) {
					return getAgentStateModifier().canResume();
				}
				return getAgentStateModifier().canAttach();
			}		
    	}
    	
    	
    	resumeItem = new ResumeControlItem();
    	return resumeItem;
    }
    
    protected IControlItem createPauseMonitoringControlItem()
    {
    	if (pauseItem != null)
    		return pauseItem;
    	
    	class PauseControlItem extends AgentControlItem
    	{		
    		public PauseControlItem()
    		{
    			super(getAgentStateModifier(), PAUSE_MONITORING, CommonUITraceConstants.PROFILE_MONITOR_GROUP, ControlMessages.CONTROL_ITEM_PAUSE, TraceUIImages.INSTANCE.getImageDescriptor("c", TraceUIImages.IMG_PAUSE));
    			setDisabledImageDescriptor(TraceUIImages.INSTANCE.getImageDescriptor("d", TraceUIImages.IMG_PAUSE));
    		}
    		
			public void run()
			{
				IAgentStateModifier agentStateModifier = getAgentStateModifier();
											
				try
				{
					/* Pause the agent */					
					agentStateModifier.pauseMonitoring();
				} catch (CoreException e)
				{
					/* Display status */
					LauncherUtility.openMessageWithDetail(e.getStatus().getSeverity(), "", e.getMessage(), e);
				}				
			}
			
			public boolean isEnabled()
			{
				return getAgentStateModifier().canPause();
			}		
    	}
    	
    	
    	pauseItem = new PauseControlItem();
    	return pauseItem;
    }
    
    
    private static class AgentControlItem extends ControlItem
    {
		private IAgentStateModifier stateModifier;
    	
    	public AgentControlItem (IAgentStateModifier stateModifier, String itemId, String groupName, String text, ImageDescriptor imageDescriptor)
    	{
    		super(itemId, groupName, text, imageDescriptor);
    		this.stateModifier = stateModifier;
    	}
    			
		public void setInput(StructuredSelection input)
		{
			stateModifier.setInput(input);
		}
	
		
		public IAgentStateModifier getStateModifier()
		{
			return stateModifier;
		}
		
		public boolean equals (Object o)
		{
			if (!(o instanceof AgentControlItem))
				return false;
			
			return stateModifier == ((AgentControlItem)o).getStateModifier() && super.equals(o);
		}
			
	}
   
}
