/**********************************************************************
 * Copyright (c) 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.trace.ui.provisional.control.provider;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.hyades.ui.internal.navigator.INavigator;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.tptp.trace.ui.internal.control.provider.TraceControlItemManager;

/**
 * An abstract class that provides minimum functionality for control providers.
 * Contributors can extends this class or provide a direct implementation of 
 * {@link IControlProvider}.
 *
 * @author Ali Mehregani
 */
public abstract class AbstractControlProvider implements IControlProvider 
{
	/** A flag which indicates that the input has changed */
	protected boolean isInputDifferent;
	
	/** 
	 * A flag that indicates if the context has changed (this can happen if for example the profiling
	 * monitor is closed and re-opend)
	 */
	protected boolean isContextDifferent;
	
	/** Stores the control items */
	protected List controlItems;

	/** Stores the current input */
	protected StructuredSelection input;

	/** The context of the controls */
	private INavigator context;
	
	
	/**
	 * @see IControlProvider#initialize(String)
	 */
	public void initialize(String entity) 
	{

	}
	
	/**
	 * Default constructor
	 */
	public AbstractControlProvider()
	{
		controlItems = new ArrayList();
	}

	/**
	 * @see IControlProvider#setInput(StructuredSelection)
	 */
	public void setInput(StructuredSelection input)
    {
		if (!isInputSame(this.input, input))
		{
			isInputDifferent = true;
			this.input = input;
		}	
    }

	
	private boolean isInputSame(StructuredSelection firstInput, StructuredSelection secondInput)
	{
		if (firstInput == null || secondInput == null || firstInput.size() != secondInput.size())
			return false;
		
		for (Iterator firstCollectionItems = firstInput.iterator(), secondCollectionItems = secondInput.iterator(); firstCollectionItems.hasNext();)
		{
			if (!firstCollectionItems.next().equals(secondCollectionItems.next()))
				return false;
		}
		return true;
	}
	

	/**
	 * @see IControlProvider#createControlItems()
	 */
	public IControlItem[] createControlItems()
	{
		isContextDifferent = context != TraceControlItemManager.getNavigator();
		if (isInputDifferent || isContextDifferent)
		{
			context = TraceControlItemManager.getNavigator();
			initializeControlItems();
			
			/* Walk through each and notify them of the input change */
			for (int i = 0, controlItemCount = controlItems.size(); i < controlItemCount; i++)
			{
				((IControlItem)controlItems.get(i)).setInput(input);
			}
				
			isInputDifferent = false;
			isContextDifferent = false;
		}
		
		if (controlItems == null)
			return null;
		
		IControlItem[] controlItems = new IControlItem[this.controlItems.size()];
		this.controlItems.toArray(controlItems);
		
		return controlItems;
	}
	
	/**
	 * Populates the control items managed by this control provider.
	 * A control item can be added by invoking <code>addControlItem(IControlItem)</code>
	 */
	abstract protected void initializeControlItems();

	
    /**
     * Add 'item' to the list of control items maintained by this
     * class.  The item must not be null
     * 
     * @param item The item to add
     */
    protected void addControlItem(IControlItem item)
	{
		if (item != null)
			controlItems.add(item);
	}
    
    /**
     * An adapter from an IAction to a IControlItem
     * 
     * @author Ali Mehregani
     */
    public static class ControlItemAdapter extends ControlItem
    {
		private IAction action;
		private String groupName;
		
		public ControlItemAdapter(String groupName, IAction selectionProviderAction)
		{
			this.groupName = groupName;
			this.action = selectionProviderAction;
		}
		
		
		public IAction getAction()
		{
			return action;
		}
		
		public String getGroup()
		{
			return groupName;
		}
		
		public String getText()
		{
			return action.getText();
		}

		public ImageDescriptor getImageDescriptor() 
		{
			return action.getImageDescriptor();
		}
		
		public void run()
		{
			action.run();
		}
    }
}
