/**********************************************************************
 * Copyright (c) 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.trace.ui.provisional.control.provider;


import org.eclipse.core.runtime.CoreException;
import org.eclipse.hyades.trace.ui.internal.core.TraceUIImages;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceConstants;
import org.eclipse.tptp.trace.ui.internal.control.provider.application.ControlMessages;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;


/**
 * Provides an abstract control provider for a process.  Subclasses are
 * expected to provide a 0-argument constructor that will invoke the constructor
 * of this class with an IProcessStateModifier object.
 *  
 * @author Ali Mehregani
 */
public abstract class AbstractProcessControlProvider extends ResourceItemControlProvider 
{
	/** The id of the contributed items */
	private static final String TERMINATE_ITEM = "org.eclipse.hyades.trace.ui.internal.popupMenu.TerminateAction";
	
	/** The terminate item */
	private IControlItem terminateItem;
	
	/**
	 * Subclasses are expected to provide a process state modifier that 
	 * can be used to alter the state of a process.
	 *  
	 * @return The process state modifier
	 */
    abstract public IProcessStateModifier getProcessStateModifier();

    
    
	/**
	 * @see org.eclipse.tptp.trace.ui.provisional.control.provider.AbstractLogicalItem.initializeControlItems()
	 */
    protected void initializeControlItems()
	{
    	super.initializeControlItems();
    	    	
    	/* Add the terminate option */
    	addControlItem (createTerminateControlItem());
    }



    /**
     * Overwrite this method to modify the terminate control
     * item.
     * 
     * @return Ther terminate control item
     */
	protected IControlItem createTerminateControlItem()
	{
		if (terminateItem != null)
			return terminateItem;
			
		class TerminateAction extends ControlItem
		{
			public TerminateAction()
			{
				super (TERMINATE_ITEM, CommonUITraceConstants.PROFILE_TERMINATE_GROUP, ControlMessages.CONTROL_ITEM_TERMINATE, TraceUIImages.INSTANCE.getImageDescriptor("c", TraceUIImages.IMG_TERMINATE));
				setDisabledImageDescriptor(TraceUIImages.INSTANCE.getImageDescriptor("d", TraceUIImages.IMG_TERMINATE));
			}
			
			public void run()
			{
				IProcessStateModifier processStateModifier = getProcessStateModifier();
											
				try
				{
					/* Terminate the process */
					processStateModifier.terminate();
				} catch (CoreException e)
				{
					/* Display status */
					LauncherUtility.openMessageWithDetail(e.getStatus().getSeverity(), "", e.getMessage(), e);
				}				
			}
			
			public boolean isEnabled()
			{
				return getProcessStateModifier().canTerminate();
			}
			
			public void setInput(StructuredSelection input)
			{
				getProcessStateModifier().setInput(input);
			}
			
			public IProcessStateModifier getStateModifier()
			{
				return getProcessStateModifier();
			}
			
			public boolean equals (Object o)
			{
				if (!(o instanceof TerminateAction))
					return false;
				
				return getProcessStateModifier() == ((TerminateAction)o).getStateModifier() && super.equals(o);
			}

		}
			
				
		terminateItem = new TerminateAction();
		return terminateItem;
	}
}
