/**********************************************************************
 * Copyright (c) 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.trace.ui.provisional.control.provider;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.viewers.StructuredSelection;

/**
 * An agent state modifier is used to change the state of an 
 * agent.
 *  
 * @author Ali Mehregani
 */
public interface IAgentStateModifier
{
	/**
	 * Detaches the agent.  This should cause the current session of the agent
	 * to be removed.
	 * 
	 * @throws CoreException wraps any exception that may occur.
	 */
	public void detach() throws CoreException;
	
	
	/**
	 * Attaches to the agent.  Attaching begins a new session but doesn't necessarily start
	 * the data collection.  Clients are expected to invoke startMonitoring after attach for
	 * the agent to begin the data collection.
	 * 
	 * @throws CoreException wraps any exception that may occur.
	 */
    public void attach() throws CoreException;
    
    
    /**
     * Starts monitoring the affiliated agent.  The agent is expected to open its data channel
     * and start collecting the requested data
     *  
     * @throws CoreException wraps any exception that may occur.
     */
    public void startMonitoring() throws CoreException;
    
    
    /**
     * Pauses the affiliated agent.  The agent is expected to temporarily suspend any collection
     * of data.  Notice that the difference between this method and detach is that it maintains
     * the current session and temporarily suspends data collection.
     *  
     * @throws CoreException wraps any exception that may occur.
     */
    public void pauseMonitoring() throws CoreException;
    
    
    /**
     * Indicates whether attach can be invoked for the affiliated agent.  This 
     * method should return false if the affiliated agent is in a state where 
     * attach is not possible (e.g. terminated or already attached)
     * 
     * @return true if attach operation can be performed; false otherwise
     */
	public boolean canAttach();

	
    /**
     * Indicates whether detach can be invoked for the affiliated agent.  This 
     * method should return false if the affiliated agent is in a state where 
     * detach is not possible (e.g. terminated or already detached)
     * 
     * @return true if the detach operation can be performed; false otherwise
     */	
	public boolean canDetach();
	
	
    /**
     * Indicates whether startMonitoring can be invoked for the affiliated agent.  This 
     * method should return false if the affiliated agent is in a state where 
     * startMonitoring is not possible (e.g. terminated, detached or already in resume state)
     * 
     * @return true if the pauseMonitoring operation can be performed; false otherwise
     */	
	public boolean canResume();
	
	
    /**
     * Indicates whether pauseMonitoring can be invoked for the affiliated agent.  This 
     * method should return false if the affiliated agent is in a state where 
     * pause is not possible (e.g. terminated, detached or paused)
     * 
     * @return true if the pauseMonitoring operation can be performed; false otherwise
     */	
	public boolean canPause();
	
	
	/**
	 * Sets the current input for this state modifier.  The input should be
	 * treated as the context for this state modifier.
	 * 
	 * @param input The input
	 */
	public void setInput(StructuredSelection input);
}
