/**********************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.trace.ui.provisional.control.provider;

import java.util.HashSet;
import java.util.Iterator;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.util.PDCoreUtil;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.osgi.util.NLS;
import org.eclipse.tptp.trace.ui.internal.control.provider.application.ControlMessages;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;


/**
 * A concrete implementation of <code>AbstractProcessItem</code> that delegates
 * the termination of a process to the Agent Controller.
 * 
 * @author Ali Mehregani
 */
public class ProcessControlProvider extends AbstractProcessControlProvider 
{
	
	/** The modifier that will alter the state of the process */
	private ProcessStateModifier processStateModifier;
	
	
	public ProcessControlProvider()
	{
		processStateModifier = new ProcessStateModifier();
	}
	
	
	public IProcessStateModifier getProcessStateModifier()
	{		
		return processStateModifier;
	}

	
	public static class ProcessStateModifier implements IProcessStateModifier
	{
		private StructuredSelection input;
		
		public void setInput (StructuredSelection input)
		{
			this.input = input;
		}
		
		
		public void terminate() throws CoreException
		{
			HashSet processProxies = new HashSet();
			for (Iterator processes = input.iterator(); processes.hasNext();)
			{
				TRCProcessProxy processProxy = findProcessProxy(processes.next());
				if (!processProxies.contains(processProxy) && processProxy != null)
				{
					processProxies.add(processProxy);
					terminateProcess(processProxy);
				}
			}
				
		}
		
		public boolean canTerminate()
		{
			boolean canTerminate = input.size() > 0;
			HashSet processProxies = new HashSet();
			for (Iterator processes = input.iterator(); canTerminate && processes.hasNext();)
			{
				TRCProcessProxy processProxy = findProcessProxy(processes.next());
				if (!processProxies.contains(processProxy) && processProxy != null)
				{
					processProxies.add(processProxy);
					canTerminate = canTerminate && processProxy.isActive();
				}
			}
					
			return canTerminate;
		}
		
		private TRCProcessProxy findProcessProxy(Object element)
		{
			if (element instanceof TRCProcessProxy)
				return (TRCProcessProxy)element;
			else if (element instanceof TRCAgentProxy)
				return ((TRCAgentProxy)element).getProcessProxy();
			
			return null;
		}
		
		private void terminateProcess (TRCProcessProxy process) throws CoreException
		{	
			CoreException error = null;
			
			try
			{
				String title = TraceMessages.TRC_MSGT;	
				String msg = NLS.bind(TraceMessages.TERMINATE_Q, new Object[] {process.getName(), String.valueOf(process.getPid())});
				
				if(! MessageDialog.openQuestion(UIPlugin.getActiveWorkbenchShell(), title, msg))
					return;
				
				
				TRCNode modelNode = process.getNode();
				String host = modelNode.getName();
				Node node = PDCoreUtil.profileConnect(host, String.valueOf(modelNode.getPort()));
				if(node == null)
					return;
				
				Process p = node.getProcess(String.valueOf(process.getPid()));			
				
				if(p != null && p.isActive())
				{
					node.killProcess(p);
				}	
				
				process.setActive( false );
			}
			catch (Exception e)
			{
				/* Wrap the exception and return */
				error = new CoreException (
						new Status(IStatus.ERROR, 
							UIPlugin.getPluginId(), 
							IStatus.ERROR, 
							NLS.bind(ControlMessages.ERROR_PROCESS_TERMINATE, 
									String.valueOf(process.getPid())),
							e));
			}

			LauncherUtility.sendProfileEvent(ProfileEvent.TERMINATE, process);
			

			/* If the agent happens to be detached, then we'll need to explicitly 
			 * send a stop monitoring event */
			EList agentProxies = process.getAgentProxies();
			for (int i = 0, agentCount = agentProxies.size(); i < agentCount; i++) 
			{
				TRCAgentProxy agentProxy = (TRCAgentProxy)agentProxies.get(i);		        
				if (!agentProxy.isAttached())
				{
					agentProxy.setMonitored(false);
					agentProxy.setActive(false);					
					LauncherUtility.sendProfileEvent(ProfileEvent.TERMINATE, agentProxy);
				}					
			}
 	        
			if (error != null)
				throw error;
		}
	}

}
